<?php
/**
 * Plugin Name: Arewa APK Apps Manager
 * Plugin URI: https://arewa.dev/arewa-apk-apps-manager/
 * Description: A professional plugin to display and manage APK apps on your WordPress site with ratings, comments, and Elementor Pro support.
 * Version: 1.0.4
 * Author: A-Said
 * Author URI: https://arewa.dev/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: arewa-apk-apps-manager
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('AREWA_APK_VERSION', '1.0.4');
define('AREWA_APK_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('AREWA_APK_PLUGIN_URL', plugin_dir_url(__FILE__));
define('AREWA_APK_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main plugin class
 */
class Arewa_APK_App {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->init_hooks();
    }
    
    private function init_hooks() {
        // Note: Activation and deactivation hooks are registered outside the class
        // Initialize plugin
        add_action('plugins_loaded', array($this, 'load_textdomain'));
        add_action('init', array($this, 'init'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        
        // Verify data integrity on plugin load (ensures data is preserved)
        add_action('plugins_loaded', array($this, 'verify_data_integrity'), 5);
        
        // Enqueue scripts and styles
        // Use priority 999 to ensure plugin CSS loads after theme CSS but before WordPress Additional CSS
        // This ensures plugin CSS can override theme styles and still be customizable
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'), 999);
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        
        // AJAX handlers
        add_action('wp_ajax_arewapap_submit_comment', array($this, 'ajax_submit_comment'));
        add_action('wp_ajax_nopriv_arewapap_submit_comment', array($this, 'ajax_submit_comment'));
        add_action('wp_ajax_arewapap_submit_rating', array($this, 'ajax_submit_rating'));
        add_action('wp_ajax_nopriv_arewapap_submit_rating', array($this, 'ajax_submit_rating'));
        add_action('wp_ajax_arewapap_submit_vote', array($this, 'ajax_submit_vote'));
        add_action('wp_ajax_nopriv_arewapap_submit_vote', array($this, 'ajax_submit_vote'));
        
        // Admin reply handler (only for logged-in admins)
        add_action('wp_ajax_arewapap_admin_reply', array($this, 'ajax_admin_reply'));
        add_action('wp_ajax_arewapap_delete_admin_reply', array($this, 'ajax_delete_admin_reply'));
        
        // Comments filter handler
        add_action('wp_ajax_arewapap_filter_comments', array($this, 'ajax_filter_comments'));
        add_action('wp_ajax_nopriv_arewapap_filter_comments', array($this, 'ajax_filter_comments'));
        
        // Download counter handler
        add_action('wp_ajax_arewapap_track_download', array($this, 'ajax_track_download'));
        add_action('wp_ajax_nopriv_arewapap_track_download', array($this, 'ajax_track_download'));
        
        // Import/Export handlers (admin only)
        add_action('wp_ajax_arewapap_export_settings', array($this, 'ajax_export_settings'));
        add_action('wp_ajax_arewapap_import_settings', array($this, 'ajax_import_settings'));
        
        // Load Import/Export AJAX handlers early (before admin_init)
        $this->load_import_export_handlers();
        
        // Quick Edit support for apps list
        add_action('manage_arewa_apk_app_posts_custom_column', array($this, 'quick_edit_column_data'), 10, 2);
        add_action('quick_edit_custom_box', array($this, 'quick_edit_custom_box'), 10, 2);
        add_action('bulk_edit_custom_box', array($this, 'bulk_edit_custom_box'), 10, 2);
        add_action('save_post', array($this, 'arewa_apk_save_quick_edit_data'), 10, 2);
        add_action('wp_ajax_arewapap_bulk_edit_app', array($this, 'arewa_apk_save_bulk_edit_data'));
        
        // Prevent WordPress from auto-changing status to 'future' when date is in future
        add_filter('wp_insert_post_data', array($this, 'prevent_auto_future_status'), 10, 2);
        
        // Shortcode
        add_shortcode('arewa_apk_app', array($this, 'render_shortcode'));
        
        // SEO: Add structured data and meta tags when shortcode is used
        // Use lower priority (99) to allow SEO plugins to run first and avoid conflicts
        add_action('wp_head', array($this, 'add_seo_meta_tags'), 99);
        
        // Elementor integration
        add_action('elementor/widgets/register', array($this, 'register_elementor_widget'), 10, 1);
        
        // Template loading for single app pages
        add_filter('single_template', array($this, 'load_single_app_template'));
        
        // Template loading for archive pages - use template_include with high priority to override theme templates
        add_filter('archive_template', array($this, 'load_archive_app_template'), 99);
        add_filter('template_include', array($this, 'load_archive_app_template_include'), 99);
        add_filter('taxonomy_template', array($this, 'load_taxonomy_template'), 99);
        
        // Set posts per page for archive
        add_action('pre_get_posts', array($this, 'set_archive_posts_per_page'));
    }
    
    /**
     * Load custom template for single app pages
     */
    public function load_single_app_template($template) {
        global $post;
        
        if ($post && $post->post_type === 'arewa_apk_app') {
            $plugin_template = AREWA_APK_PLUGIN_DIR . 'templates/single-arewa_apk_app.php';
            if (file_exists($plugin_template)) {
                return $plugin_template;
            }
        }
        
        return $template;
    }
    
    /**
     * Load custom template for archive app pages
     */
    public function load_archive_app_template($template) {
        if (is_post_type_archive('arewa_apk_app')) {
            $plugin_template = AREWA_APK_PLUGIN_DIR . 'templates/archive-arewa_apk_app.php';
            if (file_exists($plugin_template)) {
                return $plugin_template;
            }
        }
        
        return $template;
    }
    
    /**
     * Load custom template for archive app pages using template_include (higher priority)
     */
    public function load_archive_app_template_include($template) {
        if (is_post_type_archive('arewa_apk_app')) {
            $plugin_template = AREWA_APK_PLUGIN_DIR . 'templates/archive-arewa_apk_app.php';
            if (file_exists($plugin_template)) {
                return $plugin_template;
            }
        }
        
        return $template;
    }
    
    /**
     * Load custom template for taxonomy pages
     */
    public function load_taxonomy_template($template) {
        if (is_tax('arewa_apk_category')) {
            $plugin_template = AREWA_APK_PLUGIN_DIR . 'templates/taxonomy-arewa_apk_category.php';
            if (file_exists($plugin_template)) {
                return $plugin_template;
            }
        }
        
        return $template;
    }
    
    /**
     * Set posts per page for archive
     */
    public function set_archive_posts_per_page($query) {
        if (!is_admin() && $query->is_main_query()) {
            if (is_post_type_archive('arewa_apk_app') || is_tax('arewa_apk_category')) {
                $plugin_settings = self::get_plugin_settings();
                $posts_per_page = absint($plugin_settings['posts_per_page'] ?? get_option('posts_per_page', 10));
                $query->set('posts_per_page', $posts_per_page);
            }
        }
    }
    
    public static function activate() {
        try {
            global $wpdb;
            
            // Ensure $wpdb is available
            if (!isset($wpdb) || !is_object($wpdb)) {
                return;
            }
            
            // IMPORTANT: Preserve all existing data
            // Check if data exists before creating defaults
            $existing_settings = get_option('arewa_apk_settings', false);
            $existing_download_count = get_option('arewa_apk_download_count', false);
            
            // Create custom table for comments (preserves existing data)
            $table_name = $wpdb->prefix . 'arewa_apk_comments';
            $charset_collate = $wpdb->get_charset_collate();
            
            $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            app_id bigint(20) DEFAULT 0,
            author_name varchar(255) NOT NULL,
            author_email varchar(255) NOT NULL,
            author_avatar varchar(255) DEFAULT NULL,
            comment_text text NOT NULL,
            rating int(1) NOT NULL DEFAULT 0,
            vote_helpful int(1) DEFAULT 0,
            vote_not_helpful int(1) DEFAULT 0,
            admin_reply_text text DEFAULT NULL,
            admin_reply_image varchar(255) DEFAULT NULL,
            admin_reply_url varchar(255) DEFAULT NULL,
            admin_reply_date datetime DEFAULT NULL,
            ip_address varchar(45) DEFAULT NULL,
            comment_date datetime DEFAULT CURRENT_TIMESTAMP,
            approved tinyint(1) DEFAULT 1,
            PRIMARY KEY (id),
            KEY app_id (app_id),
            KEY author_email (author_email),
            KEY comment_date (comment_date)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // Escape table name for safety
        $table_name_escaped = esc_sql($table_name);
        
        // Check if table exists before trying to modify it
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        // Table names and DDL statements (SHOW, DESCRIBE, ALTER) cannot use placeholders
        $table_check = $wpdb->get_var("SHOW TABLES LIKE '{$table_name_escaped}'");
        $table_exists = ($table_check === $table_name);
        
        if ($table_exists) {
            // Add columns if they don't exist (for existing installations)
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $columns_result = $wpdb->get_col("DESCRIBE `{$table_name_escaped}`");
            
            // Ensure $columns is an array
            $columns = is_array($columns_result) ? $columns_result : array();
            
            if (!in_array('author_avatar', $columns)) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                $result = $wpdb->query("ALTER TABLE `{$table_name_escaped}` ADD COLUMN author_avatar varchar(255) DEFAULT NULL AFTER author_email");
                if ($result !== false) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    $columns_result = $wpdb->get_col("DESCRIBE `{$table_name_escaped}`");
                    $columns = is_array($columns_result) ? $columns_result : $columns;
                }
            }
            
            if (!in_array('admin_reply_text', $columns)) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                $result = $wpdb->query("ALTER TABLE `{$table_name_escaped}` ADD COLUMN admin_reply_text text DEFAULT NULL");
                if ($result !== false) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    $columns_result = $wpdb->get_col("DESCRIBE `{$table_name_escaped}`");
                    $columns = is_array($columns_result) ? $columns_result : $columns;
                }
            }
            
            if (!in_array('admin_reply_image', $columns)) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                $result = $wpdb->query("ALTER TABLE `{$table_name_escaped}` ADD COLUMN admin_reply_image varchar(255) DEFAULT NULL");
                if ($result !== false) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    $columns_result = $wpdb->get_col("DESCRIBE `{$table_name_escaped}`");
                    $columns = is_array($columns_result) ? $columns_result : $columns;
                }
            }
            
            if (!in_array('admin_reply_url', $columns)) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                $result = $wpdb->query("ALTER TABLE `{$table_name_escaped}` ADD COLUMN admin_reply_url varchar(255) DEFAULT NULL");
                if ($result !== false) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    $columns_result = $wpdb->get_col("DESCRIBE `{$table_name_escaped}`");
                    $columns = is_array($columns_result) ? $columns_result : $columns;
                }
            }
            
            if (!in_array('admin_reply_date', $columns)) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                $wpdb->query("ALTER TABLE `{$table_name_escaped}` ADD COLUMN admin_reply_date datetime DEFAULT NULL");
            }
            
            // Add app_id column if it doesn't exist (for multiple apps support)
            if (!in_array('app_id', $columns)) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                $wpdb->query("ALTER TABLE `{$table_name_escaped}` ADD COLUMN app_id bigint(20) DEFAULT 0 AFTER id");
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                $wpdb->query("ALTER TABLE `{$table_name_escaped}` ADD INDEX app_id (app_id)");
            }
        }
        
        // Only set default options if they don't exist (preserves existing settings)
        // This ensures that when you reinstall, your settings are NOT overwritten
        if (false === $existing_settings) {
            $default_options = array(
                'app_name' => 'My App',
                'app_version' => '1.0.0',
                'app_icon' => '',
                'app_screenshot' => '',
                'apk_url' => '',
                'about_app' => '',
                'comments_per_page' => 10,
            );
            
            add_option('arewa_apk_settings', $default_options);
        }
        
        // Only set default download count if it doesn't exist (preserves existing count)
        if (false === $existing_download_count) {
            add_option('arewa_apk_download_count', 0);
        }
        
        // Initialize plugin settings if they don't exist
        $existing_plugin_settings = get_option('arewa_apk_plugin_settings', false);
        if (false === $existing_plugin_settings) {
            $default_plugin_settings = array(
                'enable_comments' => 'yes',
                'allow_avatar_upload' => 'yes',
                'comments_per_page' => 10,
                'require_approval' => 'no',
                'show_download_button' => 'no',
                'show_short_description' => 'no',
                'show_categories' => 'no',
                'show_version' => 'no',
                'show_last_updated' => 'no',
                'show_download_count' => 'no',
                'show_rating' => 'yes',
                'show_excerpt' => 'no',
                'posts_per_page' => get_option('posts_per_page', 10),
            );
            add_option('arewa_apk_plugin_settings', $default_plugin_settings);
        }
        
        // Register taxonomies first (needed before creating terms)
        // We need to register them manually here since init() hasn't run yet
        self::register_taxonomies_for_activation();
        
        // Flush rewrite rules to ensure taxonomies are registered
        flush_rewrite_rules(false);
        
        // Create default categories if they don't exist
        self::create_default_categories();
        
        // Create default tags if they don't exist
        self::create_default_tags();
        
        } catch (Exception $e) {
            // Log error but don't prevent activation
            if (defined('WP_DEBUG') && WP_DEBUG && function_exists('error_log')) {
                error_log('Arewa APK App activation error: ' . $e->getMessage());
            }
        }
    }
    
    public static function deactivate() {
        // Cleanup if needed
    }
    
    public function load_textdomain() {
        // WordPress automatically loads plugin translations since version 4.6
        // No need to manually call load_plugin_textdomain() for plugins hosted on WordPress.org
    }
    
    public function init() {
        // Register Custom Post Type for Apps
        $this->register_post_type();
        
        // Register Taxonomy for App Categories
        $this->register_taxonomy();
        
        // Register Taxonomy for App Tags
        $this->register_tags_taxonomy();
        
        // Ensure default categories and tags exist (in case they weren't created during activation)
        self::create_default_categories();
        self::create_default_tags();
        
        // Register meta boxes for app settings (only if not using custom edit page)
        // add_action('add_meta_boxes', array($this, 'register_meta_boxes'));
        // add_action('save_post', array($this, 'save_app_meta'), 10, 2);
        
        // Modify row actions to use custom edit page
        add_filter('post_row_actions', array($this, 'modify_row_actions'), 10, 2);
        
        // Modify Add New button to use custom page instead of Gutenberg
        add_filter('get_edit_post_link', array($this, 'modify_edit_post_link'), 10, 3);
        add_action('admin_head', array($this, 'modify_add_new_button'));
    }
    
    /**
     * Register Custom Post Type for Apps
     */
    private function register_post_type() {
        $labels = array(
            'name'                  => _x('APK Apps', 'Post Type General Name', 'arewa-apk-apps-manager'),
            'singular_name'         => _x('APK App', 'Post Type Singular Name', 'arewa-apk-apps-manager'),
            'menu_name'             => __('APK Apps', 'arewa-apk-apps-manager'),
            'name_admin_bar'        => __('APK App', 'arewa-apk-apps-manager'),
            'archives'              => __('App Archives', 'arewa-apk-apps-manager'),
            'attributes'            => __('App Attributes', 'arewa-apk-apps-manager'),
            'parent_item_colon'     => __('Parent App:', 'arewa-apk-apps-manager'),
            'all_items'             => __('All Apps', 'arewa-apk-apps-manager'),
            'add_new_item'          => __('Add New App', 'arewa-apk-apps-manager'),
            'add_new'               => __('Add New', 'arewa-apk-apps-manager'),
            'new_item'              => __('New App', 'arewa-apk-apps-manager'),
            'edit_item'             => __('Edit App', 'arewa-apk-apps-manager'),
            'update_item'           => __('Update App', 'arewa-apk-apps-manager'),
            'view_item'             => __('View App', 'arewa-apk-apps-manager'),
            'view_items'            => __('View Apps', 'arewa-apk-apps-manager'),
            'search_items'          => __('Search App', 'arewa-apk-apps-manager'),
            'not_found'             => __('Not found', 'arewa-apk-apps-manager'),
            'not_found_in_trash'    => __('Not found in Trash', 'arewa-apk-apps-manager'),
            'featured_image'        => __('App Icon', 'arewa-apk-apps-manager'),
            'set_featured_image'    => __('Set app icon', 'arewa-apk-apps-manager'),
            'remove_featured_image' => __('Remove app icon', 'arewa-apk-apps-manager'),
            'use_featured_image'    => __('Use as app icon', 'arewa-apk-apps-manager'),
            'insert_into_item'      => __('Insert into app', 'arewa-apk-apps-manager'),
            'uploaded_to_this_item' => __('Uploaded to this app', 'arewa-apk-apps-manager'),
            'items_list'            => __('Apps list', 'arewa-apk-apps-manager'),
            'items_list_navigation' => __('Apps list navigation', 'arewa-apk-apps-manager'),
            'filter_items_list'     => __('Filter apps list', 'arewa-apk-apps-manager'),
        );
        
        $args = array(
            'label'                 => __('APK App', 'arewa-apk-apps-manager'),
            'description'           => __('APK Apps for download', 'arewa-apk-apps-manager'),
            'labels'                => $labels,
            'supports'              => array('title', 'editor', 'thumbnail', 'excerpt', 'custom-fields'),
            'taxonomies'            => array('arewa_apk_category'),
            'hierarchical'          => false,
            'public'                => true,
            'show_ui'               => true,
            'show_in_menu'          => false, // We'll add it to our custom menu
            'menu_position'         => 30,
            'menu_icon'             => 'dashicons-smartphone',
            'show_in_admin_bar'     => true,
            'show_in_nav_menus'     => true,
            'can_export'            => true,
            'has_archive'           => true,
            'archive_slug'          => 'apps',
            'exclude_from_search'   => false,
            'publicly_queryable'    => true,
            'capability_type'       => 'post',
            'show_in_rest'          => true,
            'rewrite'               => array('slug' => 'app', 'with_front' => false),
        );
        
        register_post_type('arewa_apk_app', $args);
    }
    
    /**
     * Register taxonomies for activation hook (static method)
     */
    public static function register_taxonomies_for_activation() {
        // Register post type first
        $labels = array(
            'name'                  => _x('APK Apps', 'Post Type General Name', 'arewa-apk-apps-manager'),
            'singular_name'         => _x('APK App', 'Post Type Singular Name', 'arewa-apk-apps-manager'),
            'menu_name'             => __('APK Apps', 'arewa-apk-apps-manager'),
            'name_admin_bar'        => __('APK App', 'arewa-apk-apps-manager'),
            'archives'              => __('App Archives', 'arewa-apk-apps-manager'),
            'attributes'            => __('App Attributes', 'arewa-apk-apps-manager'),
            'parent_item_colon'     => __('Parent App:', 'arewa-apk-apps-manager'),
            'all_items'             => __('All Apps', 'arewa-apk-apps-manager'),
            'add_new_item'          => __('Add New App', 'arewa-apk-apps-manager'),
            'add_new'               => __('Add New', 'arewa-apk-apps-manager'),
            'new_item'              => __('New App', 'arewa-apk-apps-manager'),
            'edit_item'             => __('Edit App', 'arewa-apk-apps-manager'),
            'update_item'           => __('Update App', 'arewa-apk-apps-manager'),
            'view_item'             => __('View App', 'arewa-apk-apps-manager'),
            'view_items'            => __('View Apps', 'arewa-apk-apps-manager'),
            'search_items'          => __('Search App', 'arewa-apk-apps-manager'),
            'not_found'             => __('Not found', 'arewa-apk-apps-manager'),
            'not_found_in_trash'    => __('Not found in Trash', 'arewa-apk-apps-manager'),
            'featured_image'       => __('Featured Image', 'arewa-apk-apps-manager'),
            'set_featured_image'    => __('Set featured image', 'arewa-apk-apps-manager'),
            'remove_featured_image' => __('Remove featured image', 'arewa-apk-apps-manager'),
            'use_featured_image'    => __('Use as featured image', 'arewa-apk-apps-manager'),
            'insert_into_item'      => __('Insert into app', 'arewa-apk-apps-manager'),
            'uploaded_to_this_item' => __('Uploaded to this app', 'arewa-apk-apps-manager'),
            'items_list'            => __('Apps list', 'arewa-apk-apps-manager'),
            'items_list_navigation' => __('Apps list navigation', 'arewa-apk-apps-manager'),
            'filter_items_list'     => __('Filter apps list', 'arewa-apk-apps-manager'),
        );
        
        $args = array(
            'label'                 => __('APK App', 'arewa-apk-apps-manager'),
            'description'           => __('APK Apps for download', 'arewa-apk-apps-manager'),
            'labels'                => $labels,
            'supports'              => array('title', 'editor', 'thumbnail', 'excerpt', 'custom-fields'),
            'taxonomies'            => array('arewa_apk_category', 'arewa_apk_tag'),
            'hierarchical'          => false,
            'public'                => true,
            'show_ui'               => true,
            'show_in_menu'          => false,
            'menu_position'         => 30,
            'menu_icon'             => 'dashicons-smartphone',
            'show_in_admin_bar'     => true,
            'show_in_nav_menus'     => true,
            'can_export'            => true,
            'has_archive'           => true,
            'archive_slug'          => 'apps',
            'exclude_from_search'   => false,
            'publicly_queryable'    => true,
            'capability_type'       => 'post',
            'show_in_rest'          => true,
            'rewrite'               => array('slug' => 'app', 'with_front' => false),
        );
        
        register_post_type('arewa_apk_app', $args);
        
        // Register category taxonomy
        $category_labels = array(
            'name'                       => _x('App Categories', 'Taxonomy General Name', 'arewa-apk-apps-manager'),
            'singular_name'              => _x('App Category', 'Taxonomy Singular Name', 'arewa-apk-apps-manager'),
            'menu_name'                  => __('Categories', 'arewa-apk-apps-manager'),
            'all_items'                  => __('All Categories', 'arewa-apk-apps-manager'),
            'parent_item'                => __('Parent Category', 'arewa-apk-apps-manager'),
            'parent_item_colon'          => __('Parent Category:', 'arewa-apk-apps-manager'),
            'new_item_name'              => __('New Category Name', 'arewa-apk-apps-manager'),
            'add_new_item'               => __('Add New Category', 'arewa-apk-apps-manager'),
            'edit_item'                  => __('Edit Category', 'arewa-apk-apps-manager'),
            'update_item'                => __('Update Category', 'arewa-apk-apps-manager'),
            'view_item'                  => __('View Category', 'arewa-apk-apps-manager'),
            'separate_items_with_commas' => __('Separate categories with commas', 'arewa-apk-apps-manager'),
            'add_or_remove_items'       => __('Add or remove categories', 'arewa-apk-apps-manager'),
            'choose_from_most_used'      => __('Choose from the most used', 'arewa-apk-apps-manager'),
            'popular_items'             => __('Popular Categories', 'arewa-apk-apps-manager'),
            'search_items'               => __('Search Categories', 'arewa-apk-apps-manager'),
            'not_found'                 => __('Not Found', 'arewa-apk-apps-manager'),
            'no_terms'                  => __('No categories', 'arewa-apk-apps-manager'),
            'items_list'                => __('Categories list', 'arewa-apk-apps-manager'),
            'items_list_navigation'     => __('Categories list navigation', 'arewa-apk-apps-manager'),
        );
        
        $category_args = array(
            'labels'                     => $category_labels,
            'hierarchical'               => true,
            'public'                      => true,
            'show_ui'                     => true,
            'show_admin_column'           => true,
            'show_in_nav_menus'           => true,
            'show_tagcloud'               => true,
            'show_in_rest'                => true,
            'rewrite'                     => array('slug' => 'app-category', 'with_front' => false),
        );
        
        register_taxonomy('arewa_apk_category', array('arewa_apk_app'), $category_args);
        
        // Register tags taxonomy
        $tag_labels = array(
            'name'                       => _x('App Tags', 'Taxonomy General Name', 'arewa-apk-apps-manager'),
            'singular_name'              => _x('App Tag', 'Taxonomy Singular Name', 'arewa-apk-apps-manager'),
            'menu_name'                  => __('Tags', 'arewa-apk-apps-manager'),
            'all_items'                  => __('All Tags', 'arewa-apk-apps-manager'),
            'parent_item'                => null,
            'parent_item_colon'          => null,
            'new_item_name'              => __('New Tag Name', 'arewa-apk-apps-manager'),
            'add_new_item'               => __('Add New Tag', 'arewa-apk-apps-manager'),
            'edit_item'                  => __('Edit Tag', 'arewa-apk-apps-manager'),
            'update_item'                => __('Update Tag', 'arewa-apk-apps-manager'),
            'view_item'                  => __('View Tag', 'arewa-apk-apps-manager'),
            'separate_items_with_commas' => __('Separate tags with commas', 'arewa-apk-apps-manager'),
            'add_or_remove_items'        => __('Add or remove tags', 'arewa-apk-apps-manager'),
            'choose_from_most_used'      => __('Choose from the most used', 'arewa-apk-apps-manager'),
            'popular_items'              => __('Popular Tags', 'arewa-apk-apps-manager'),
            'search_items'               => __('Search Tags', 'arewa-apk-apps-manager'),
            'not_found'                  => __('Not Found', 'arewa-apk-apps-manager'),
            'no_terms'                   => __('No tags', 'arewa-apk-apps-manager'),
            'items_list'                 => __('Tags list', 'arewa-apk-apps-manager'),
            'items_list_navigation'       => __('Tags list navigation', 'arewa-apk-apps-manager'),
        );
        
        $tag_args = array(
            'labels'                     => $tag_labels,
            'hierarchical'               => false,
            'public'                     => true,
            'show_ui'                    => true,
            'show_admin_column'          => true,
            'show_in_nav_menus'          => true,
            'show_tagcloud'              => true,
            'show_in_rest'                => true,
            'rewrite'                    => array('slug' => 'app-tag', 'with_front' => false),
        );
        
        register_taxonomy('arewa_apk_tag', array('arewa_apk_app'), $tag_args);
    }
    
    /**
     * Register Taxonomy for App Categories
     */
    private function register_taxonomy() {
        $labels = array(
            'name'                       => _x('App Categories', 'Taxonomy General Name', 'arewa-apk-apps-manager'),
            'singular_name'              => _x('App Category', 'Taxonomy Singular Name', 'arewa-apk-apps-manager'),
            'menu_name'                  => __('Categories', 'arewa-apk-apps-manager'),
            'all_items'                  => __('All Categories', 'arewa-apk-apps-manager'),
            'parent_item'                => __('Parent Category', 'arewa-apk-apps-manager'),
            'parent_item_colon'          => __('Parent Category:', 'arewa-apk-apps-manager'),
            'new_item_name'              => __('New Category Name', 'arewa-apk-apps-manager'),
            'add_new_item'               => __('Add New Category', 'arewa-apk-apps-manager'),
            'edit_item'                  => __('Edit Category', 'arewa-apk-apps-manager'),
            'update_item'                => __('Update Category', 'arewa-apk-apps-manager'),
            'view_item'                  => __('View Category', 'arewa-apk-apps-manager'),
            'separate_items_with_commas' => __('Separate categories with commas', 'arewa-apk-apps-manager'),
            'add_or_remove_items'       => __('Add or remove categories', 'arewa-apk-apps-manager'),
            'choose_from_most_used'      => __('Choose from the most used', 'arewa-apk-apps-manager'),
            'popular_items'             => __('Popular Categories', 'arewa-apk-apps-manager'),
            'search_items'               => __('Search Categories', 'arewa-apk-apps-manager'),
            'not_found'                 => __('Not Found', 'arewa-apk-apps-manager'),
            'no_terms'                  => __('No categories', 'arewa-apk-apps-manager'),
            'items_list'                => __('Categories list', 'arewa-apk-apps-manager'),
            'items_list_navigation'     => __('Categories list navigation', 'arewa-apk-apps-manager'),
        );
        
        $args = array(
            'labels'                     => $labels,
            'hierarchical'               => true,
            'public'                      => true,
            'show_ui'                     => true,
            'show_admin_column'           => true,
            'show_in_nav_menus'           => true,
            'show_tagcloud'               => true,
            'show_in_rest'                => true,
            'rewrite'                     => array('slug' => 'app-category', 'with_front' => false),
        );
        
        register_taxonomy('arewa_apk_category', array('arewa_apk_app'), $args);
    }
    
    /**
     * Register Taxonomy for App Tags
     */
    private function register_tags_taxonomy() {
        $labels = array(
            'name'                       => _x('App Tags', 'Taxonomy General Name', 'arewa-apk-apps-manager'),
            'singular_name'              => _x('App Tag', 'Taxonomy Singular Name', 'arewa-apk-apps-manager'),
            'menu_name'                  => __('Tags', 'arewa-apk-apps-manager'),
            'all_items'                  => __('All Tags', 'arewa-apk-apps-manager'),
            'parent_item'                => null,
            'parent_item_colon'          => null,
            'new_item_name'              => __('New Tag Name', 'arewa-apk-apps-manager'),
            'add_new_item'               => __('Add New Tag', 'arewa-apk-apps-manager'),
            'edit_item'                  => __('Edit Tag', 'arewa-apk-apps-manager'),
            'update_item'                => __('Update Tag', 'arewa-apk-apps-manager'),
            'view_item'                  => __('View Tag', 'arewa-apk-apps-manager'),
            'separate_items_with_commas' => __('Separate tags with commas', 'arewa-apk-apps-manager'),
            'add_or_remove_items'        => __('Add or remove tags', 'arewa-apk-apps-manager'),
            'choose_from_most_used'      => __('Choose from the most used', 'arewa-apk-apps-manager'),
            'popular_items'              => __('Popular Tags', 'arewa-apk-apps-manager'),
            'search_items'               => __('Search Tags', 'arewa-apk-apps-manager'),
            'not_found'                  => __('Not Found', 'arewa-apk-apps-manager'),
            'no_terms'                   => __('No tags', 'arewa-apk-apps-manager'),
            'items_list'                 => __('Tags list', 'arewa-apk-apps-manager'),
            'items_list_navigation'       => __('Tags list navigation', 'arewa-apk-apps-manager'),
        );
        
        $args = array(
            'labels'                     => $labels,
            'hierarchical'               => false,
            'public'                     => true,
            'show_ui'                    => true,
            'show_admin_column'          => true,
            'show_in_nav_menus'          => true,
            'show_tagcloud'              => true,
            'show_in_rest'                => true,
            'rewrite'                    => array('slug' => 'app-tag', 'with_front' => false),
        );
        
        register_taxonomy('arewa_apk_tag', array('arewa_apk_app'), $args);
    }
    
    /**
     * Create default categories on plugin activation
     */
    public static function create_default_categories() {
        // Check if taxonomy is registered
        if (!taxonomy_exists('arewa_apk_category')) {
            return;
        }
        
        $default_categories = array(
            'Games',
            'Education',
            'Productivity',
            'Video Player',
            'Music',
            'Social',
            'Shopping',
            'Finance',
            'Health & Fitness',
            'Travel',
            'News',
            'Photography',
            'Entertainment',
            'Sports',
            'Weather',
            'Food & Drink',
            'Lifestyle',
            'Business',
            'Books',
            'Tools'
        );
        
        foreach ($default_categories as $category_name) {
            // Check if category already exists
            $term = term_exists($category_name, 'arewa_apk_category');
            if (!$term) {
                $result = wp_insert_term($category_name, 'arewa_apk_category');
                if (is_wp_error($result)) {
                    // Log error but continue
                    if (defined('WP_DEBUG') && WP_DEBUG && function_exists('error_log')) {
                        error_log('Arewa APK: Failed to create category "' . $category_name . '": ' . $result->get_error_message());
                    }
                }
            }
        }
    }
    
    /**
     * Create default tags on plugin activation
     */
    public static function create_default_tags() {
        // Check if taxonomy is registered
        if (!taxonomy_exists('arewa_apk_tag')) {
            return;
        }
        
        $default_tags = array(
            'Free',
            'Paid',
            'Popular',
            'New',
            'Updated',
            'Featured',
            'Trending',
            'Offline',
            'Online',
            'HD',
            '4K',
            'Lightweight',
            'Fast',
            'Secure',
            'Ad-free',
            'Premium',
            'Open Source',
            'Cross-platform',
            'Mobile',
            'Tablet',
            'Android',
            'iOS',
            'Gaming',
            'Streaming',
            'Download',
            'Social Media',
            'Communication',
            'Productivity',
            'Entertainment',
            'Education',
            'Health',
            'Finance',
            'Shopping',
            'Travel',
            'News',
            'Music',
            'Video',
            'Photo',
            'Camera',
            'Weather',
            'Sports',
            'Food',
            'Lifestyle',
            'Business',
            'Books',
            'Tools',
            'Utility'
        );
        
        foreach ($default_tags as $tag_name) {
            // Check if tag already exists
            $term = term_exists($tag_name, 'arewa_apk_tag');
            if (!$term) {
                $result = wp_insert_term($tag_name, 'arewa_apk_tag');
                if (is_wp_error($result)) {
                    // Log error but continue
                    if (defined('WP_DEBUG') && WP_DEBUG && function_exists('error_log')) {
                        error_log('Arewa APK: Failed to create tag "' . $tag_name . '": ' . $result->get_error_message());
                    }
                }
            }
        }
    }
    
    /**
     * Register Meta Boxes for App Settings
     */
    public function register_meta_boxes() {
        add_meta_box(
            'arewa_apk_app_settings',
            __('App Settings', 'arewa-apk-apps-manager'),
            array($this, 'render_app_settings_meta_box'),
            'arewa_apk_app',
            'normal',
            'high'
        );
    }
    
    /**
     * Render App Settings Meta Box
     */
    public function render_app_settings_meta_box($post) {
        wp_nonce_field('arewa_apk_app_meta_box', 'arewa_apk_app_meta_box_nonce');
        
        // Get existing meta values
        $app_version = get_post_meta($post->ID, '_arewa_apk_version', true);
        $app_icon = get_post_meta($post->ID, '_arewa_apk_icon', true);
        $app_screenshots = get_post_meta($post->ID, '_arewa_apk_screenshots', true);
        if (!is_array($app_screenshots)) {
            $app_screenshots = array();
        }
        $apk_url = get_post_meta($post->ID, '_arewa_apk_url', true);
        $short_description = get_post_meta($post->ID, '_arewa_apk_short_description', true);
        $about_app = get_post_meta($post->ID, '_arewa_apk_about', true);
        $comments_per_page = get_post_meta($post->ID, '_arewa_apk_comments_per_page', true);
        if (empty($comments_per_page)) {
            $comments_per_page = 10;
        }
        $disable_seo = get_post_meta($post->ID, '_arewa_apk_disable_seo', true);
        if (empty($disable_seo)) {
            $disable_seo = 'no';
        }
        
        // Include meta box template
        include AREWA_APK_PLUGIN_DIR . 'includes/admin/app-meta-box.php';
    }
    
    /**
     * Modify row actions to use custom edit page
     */
    public function modify_row_actions($actions, $post) {
        if ($post->post_type === 'arewa_apk_app') {
            // Replace edit link with custom edit page
            if (isset($actions['edit'])) {
                $actions['edit'] = '<a href="' . admin_url('admin.php?page=arewa-apk-edit-app&app_id=' . $post->ID) . '">' . __('Edit', 'arewa-apk-apps-manager') . '</a>';
            }
            // Remove quick edit
            unset($actions['inline hide-if-no-js']);
        }
        return $actions;
    }
    
    /**
     * Modify edit post link to use custom edit page
     */
    public function modify_edit_post_link($link, $post_id, $context) {
        $post = get_post($post_id);
        if ($post && $post->post_type === 'arewa_apk_app') {
            return admin_url('admin.php?page=arewa-apk-edit-app&app_id=' . $post_id);
        }
        return $link;
    }
    
    /**
     * Modify Add New button in post list to use custom page
     */
    public function modify_add_new_button() {
        $screen = get_current_screen();
        if ($screen && $screen->post_type === 'arewa_apk_app' && $screen->base === 'edit') {
            $add_new_url = admin_url('admin.php?page=arewa-apk-add-app');
            $script = "jQuery(document).ready(function($) {
                // Modify Add New button
                var addNewButton = $('.page-title-action, .add-new-h2');
                if (addNewButton.length) {
                    addNewButton.attr('href', " . wp_json_encode($add_new_url) . ");
                }
                
                // Also modify any other Add New links
                $('a[href*=\"post-new.php?post_type=arewa_apk_app\"]').attr('href', " . wp_json_encode($add_new_url) . ");
            });";
            wp_add_inline_script('jquery', $script);
        }
    }
    
    /**
     * Save App Meta Data (legacy, for Gutenberg editor if enabled)
     */
    public function save_app_meta($post_id, $post) {
        // Check if this is an autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        // Check post type
        if ($post->post_type !== 'arewa_apk_app') {
            return;
        }
        
        // Check permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // Verify nonce
        if (!isset($_POST['arewa_apk_app_meta_box_nonce']) || 
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['arewa_apk_app_meta_box_nonce'])), 'arewa_apk_app_meta_box')) {
            return;
        }
        
        // Save meta fields
        if (isset($_POST['arewa_apk_version'])) {
            update_post_meta($post_id, '_arewa_apk_version', sanitize_text_field(wp_unslash($_POST['arewa_apk_version'])));
        }
        
        if (isset($_POST['arewa_apk_icon'])) {
            update_post_meta($post_id, '_arewa_apk_icon', esc_url_raw(wp_unslash($_POST['arewa_apk_icon'])));
        }
        
        if (isset($_POST['arewa_apk_feature_graphic'])) {
            update_post_meta($post_id, '_arewa_apk_feature_graphic', esc_url_raw(wp_unslash($_POST['arewa_apk_feature_graphic'])));
        }
        
        if (isset($_POST['arewa_apk_screenshots']) && is_array($_POST['arewa_apk_screenshots'])) {
            $screenshots = array();
            $screenshots_raw = wp_unslash($_POST['arewa_apk_screenshots']);
            foreach ($screenshots_raw as $screenshot) {
                $url = esc_url_raw($screenshot);
                if (!empty($url)) {
                    $screenshots[] = $url;
                }
            }
            update_post_meta($post_id, '_arewa_apk_screenshots', $screenshots);
        }
        
        if (isset($_POST['arewa_apk_url'])) {
            update_post_meta($post_id, '_arewa_apk_url', esc_url_raw(wp_unslash($_POST['arewa_apk_url'])));
        }
        
        if (isset($_POST['arewa_apk_short_description'])) {
            $short_description = sanitize_text_field(wp_unslash($_POST['arewa_apk_short_description']));
            // Limit to 100 characters
            if (strlen($short_description) > 100) {
                $short_description = substr($short_description, 0, 100);
            }
            update_post_meta($post_id, '_arewa_apk_short_description', $short_description);
        }
        
        if (isset($_POST['arewa_apk_about'])) {
            update_post_meta($post_id, '_arewa_apk_about', wp_kses_post(wp_unslash($_POST['arewa_apk_about'])));
        }
        
        if (isset($_POST['arewa_apk_comments_per_page'])) {
            update_post_meta($post_id, '_arewa_apk_comments_per_page', absint(wp_unslash($_POST['arewa_apk_comments_per_page'])));
        }
        
        if (isset($_POST['arewa_apk_disable_seo'])) {
            update_post_meta($post_id, '_arewa_apk_disable_seo', sanitize_text_field(wp_unslash($_POST['arewa_apk_disable_seo'])));
        } else {
            update_post_meta($post_id, '_arewa_apk_disable_seo', 'no');
        }
    }
    
    private function create_comment_tables() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'arewa_apk_comments';
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            app_id bigint(20) DEFAULT 0,
            author_name varchar(255) NOT NULL,
            author_email varchar(255) NOT NULL,
            author_avatar varchar(255) DEFAULT NULL,
            comment_text text NOT NULL,
            rating int(1) NOT NULL DEFAULT 0,
            vote_helpful int(1) DEFAULT 0,
            vote_not_helpful int(1) DEFAULT 0,
            admin_reply_text text DEFAULT NULL,
            admin_reply_image varchar(255) DEFAULT NULL,
            admin_reply_url varchar(255) DEFAULT NULL,
            admin_reply_date datetime DEFAULT NULL,
            ip_address varchar(45) DEFAULT NULL,
            comment_date datetime DEFAULT CURRENT_TIMESTAMP,
            approved tinyint(1) DEFAULT 1,
            PRIMARY KEY (id),
            KEY app_id (app_id),
            KEY author_email (author_email),
            KEY comment_date (comment_date)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // Add author_avatar column if it doesn't exist (for existing installations)
        // Escape table name for safety
        $table_name_escaped = esc_sql($table_name);
        
        // Check if columns exist and add them if needed
        $columns = $wpdb->get_col("DESCRIBE `{$table_name_escaped}`");
        
        if (!in_array('author_avatar', $columns)) {
            $wpdb->query(
                "ALTER TABLE `{$table_name_escaped}` ADD COLUMN author_avatar varchar(255) DEFAULT NULL AFTER author_email"
            );
        }
        
        // Add admin reply columns if they don't exist
        // Use a safer approach that doesn't rely on AFTER clause
        if (!in_array('admin_reply_text', $columns)) {
            $result = $wpdb->query(
                "ALTER TABLE `{$table_name_escaped}` ADD COLUMN admin_reply_text text DEFAULT NULL"
            );
            if ($result === false && !empty($wpdb->last_error)) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('Arewa APK: Failed to add admin_reply_text column: ' . $wpdb->last_error);
                }
            } else {
                // Re-check columns after adding
                $columns = $wpdb->get_col("DESCRIBE `{$table_name_escaped}`");
            }
        }
        
        if (!in_array('admin_reply_image', $columns)) {
            $result = $wpdb->query(
                "ALTER TABLE `{$table_name_escaped}` ADD COLUMN admin_reply_image varchar(255) DEFAULT NULL"
            );
            if ($result === false && !empty($wpdb->last_error)) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('Arewa APK: Failed to add admin_reply_image column: ' . $wpdb->last_error);
                }
            } else {
                // Re-check columns after adding
                $columns = $wpdb->get_col("DESCRIBE `{$table_name_escaped}`");
            }
        }
        
        if (!in_array('admin_reply_url', $columns)) {
            $result = $wpdb->query(
                "ALTER TABLE `{$table_name_escaped}` ADD COLUMN admin_reply_url varchar(255) DEFAULT NULL"
            );
            if ($result === false && !empty($wpdb->last_error)) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('Arewa APK: Failed to add admin_reply_url column: ' . $wpdb->last_error);
                }
            } else {
                // Re-check columns after adding
                $columns = $wpdb->get_col("DESCRIBE `{$table_name_escaped}`");
            }
        }
        
            if (!in_array('admin_reply_date', $columns)) {
                $result = $wpdb->query(
                    "ALTER TABLE `{$table_name_escaped}` ADD COLUMN admin_reply_date datetime DEFAULT NULL"
                );
                if ($result === false && !empty($wpdb->last_error)) {
                    if (defined('WP_DEBUG') && WP_DEBUG) {
                        error_log('Arewa APK: Failed to add admin_reply_date column: ' . $wpdb->last_error);
                    }
                } else {
                    // Re-check columns after adding
                    $columns = $wpdb->get_col("DESCRIBE `{$table_name_escaped}`");
                }
            }
            
            // Add app_id column if it doesn't exist (for multiple apps support)
            if (!in_array('app_id', $columns)) {
                $wpdb->query("ALTER TABLE `{$table_name_escaped}` ADD COLUMN app_id bigint(20) DEFAULT 0 AFTER id");
                $wpdb->query("ALTER TABLE `{$table_name_escaped}` ADD INDEX app_id (app_id)");
            }
        }
    
    public function add_admin_menu() {
        add_menu_page(
            __('Arewa APK App', 'arewa-apk-apps-manager'),
            __('APK Apps', 'arewa-apk-apps-manager'),
            'manage_options',
            'arewa-apk-apps-manager',
            array($this, 'render_admin_page'),
            'dashicons-smartphone',
            30
        );
        
        // Add submenu for All Apps (Custom Post Type)
        add_submenu_page(
            'arewa-apk-apps-manager',
            __('All Apps', 'arewa-apk-apps-manager'),
            __('All Apps', 'arewa-apk-apps-manager'),
            'edit_posts',
            'edit.php?post_type=arewa_apk_app'
        );
        
        // Add submenu for Add New App (custom page, not Gutenberg)
        add_submenu_page(
            'arewa-apk-apps-manager',
            __('Add New App', 'arewa-apk-apps-manager'),
            __('Add New App', 'arewa-apk-apps-manager'),
            'edit_posts',
            'arewa-apk-add-app',
            array($this, 'render_add_app_page')
        );
        
        // Add submenu for Edit App (custom page, not Gutenberg)
        add_submenu_page(
            null, // Hidden from menu
            __('Edit App', 'arewa-apk-apps-manager'),
            __('Edit App', 'arewa-apk-apps-manager'),
            'edit_posts',
            'arewa-apk-edit-app',
            array($this, 'render_edit_app_page')
        );
        
        // Add submenu for Categories
        add_submenu_page(
            'arewa-apk-apps-manager',
            __('App Categories', 'arewa-apk-apps-manager'),
            __('Categories', 'arewa-apk-apps-manager'),
            'manage_categories',
            'edit-tags.php?taxonomy=arewa_apk_category&post_type=arewa_apk_app'
        );
        
        // Plugin Settings page (global plugin configuration)
        add_submenu_page(
            'arewa-apk-apps-manager',
            __('Plugin Settings', 'arewa-apk-apps-manager'),
            __('Plugin Settings', 'arewa-apk-apps-manager'),
            'manage_options',
            'arewa-apk-apps-manager',
            array($this, 'render_admin_page')
        );
        
        add_submenu_page(
            'arewa-apk-apps-manager',
            __('Comments', 'arewa-apk-apps-manager'),
            __('Comments', 'arewa-apk-apps-manager'),
            'manage_options',
            'arewa-apk-comments',
            array($this, 'render_comments_page')
        );
        
        add_submenu_page(
            'arewa-apk-apps-manager',
            __('Import/Export', 'arewa-apk-apps-manager'),
            __('Import/Export', 'arewa-apk-apps-manager'),
            'manage_options',
            'arewa-apk-import-export',
            array($this, 'render_import_export_page')
        );
    }
    
    public function register_settings() {
        register_setting('arewa_apk_settings_group', 'arewa_apk_settings', array($this, 'sanitize_settings'));
        register_setting('arewa_apk_plugin_settings_group', 'arewa_apk_plugin_settings', array($this, 'sanitize_plugin_settings'));
    }
    
    public function sanitize_settings($input) {
        $sanitized = array();
        $sanitized['app_name'] = sanitize_text_field($input['app_name'] ?? '');
        $sanitized['app_version'] = sanitize_text_field($input['app_version'] ?? '1.0.0');
        $sanitized['app_icon'] = esc_url_raw($input['app_icon'] ?? '');
        
        // Handle multiple screenshots
        $screenshots = array();
        if (isset($input['app_screenshots']) && is_array($input['app_screenshots'])) {
            foreach ($input['app_screenshots'] as $screenshot) {
                $url = esc_url_raw($screenshot);
                if (!empty($url)) {
                    $screenshots[] = $url;
                }
            }
        }
        $sanitized['app_screenshots'] = $screenshots;
        // Keep old field for backward compatibility
        $sanitized['app_screenshot'] = !empty($screenshots) ? $screenshots[0] : '';
        
        $sanitized['apk_url'] = esc_url_raw($input['apk_url'] ?? '');
        // Sanitize about app text - allow HTML but sanitize it with wp_kses_post
        $sanitized['about_app'] = isset($input['about_app']) ? wp_kses_post($input['about_app']) : '';
        $sanitized['comments_per_page'] = absint($input['comments_per_page'] ?? 10);
        // Sanitize custom CSS - remove HTML tags and preserve CSS syntax
        $sanitized['custom_css'] = isset($input['custom_css']) ? wp_strip_all_tags($input['custom_css']) : '';
        // SEO disable option
        $sanitized['disable_seo'] = isset($input['disable_seo']) && $input['disable_seo'] === 'yes' ? 'yes' : 'no';
        return $sanitized;
    }
    
    public function enqueue_frontend_assets() {
        // Always enqueue CSS to prevent issues with different browsers/cache
        // Enqueue dashicons for star ratings and icons
        wp_enqueue_style('dashicons');
        
        // Get CSS file modification time for cache-busting
        $css_file_path = AREWA_APK_PLUGIN_DIR . 'assets/css/frontend.css';
        $css_version = AREWA_APK_VERSION;
        if (file_exists($css_file_path)) {
            // Add file modification time to version to force cache refresh when CSS changes
            $css_version .= '.' . filemtime($css_file_path);
        }
        
        // Enqueue plugin CSS with high priority (999) to load after theme CSS
        // This ensures plugin styles can override theme styles
        wp_enqueue_style('arewa-apk-frontend', AREWA_APK_PLUGIN_URL . 'assets/css/frontend.css', array('dashicons'), $css_version, 'all');
        
        // Enqueue archive styles
        $archive_css_path = AREWA_APK_PLUGIN_DIR . 'assets/css/archive-styles.css';
        $archive_css_version = AREWA_APK_VERSION;
        if (file_exists($archive_css_path)) {
            $archive_css_version .= '.' . filemtime($archive_css_path);
        }
        wp_enqueue_style('arewa-apk-archive-styles', AREWA_APK_PLUGIN_URL . 'assets/css/archive-styles.css', array('arewa-apk-frontend'), $archive_css_version, 'all');
        
        // Enqueue carousel styles
        $carousel_css_path = AREWA_APK_PLUGIN_DIR . 'assets/css/carousel-styles.css';
        $carousel_css_version = AREWA_APK_VERSION;
        if (file_exists($carousel_css_path)) {
            $carousel_css_version .= '.' . filemtime($carousel_css_path);
        }
        wp_enqueue_style('arewa-apk-carousel-styles', AREWA_APK_PLUGIN_URL . 'assets/css/carousel-styles.css', array('arewa-apk-frontend'), $carousel_css_version, 'all');
        
        // Add inline style marker to verify CSS is loaded
        // This helps debug CSS loading issues
        wp_add_inline_style('arewa-apk-frontend', '/* Arewa APK App CSS Loaded - Version: ' . esc_html($css_version) . ' */');
        
        // Note: Custom CSS from admin settings is kept for backward compatibility
        // WordPress Additional CSS (from Customizer) loads after this and will override it
        $settings = self::get_settings();
        $custom_css = $settings['custom_css'] ?? '';
        if (!empty($custom_css)) {
            // Add custom CSS inline - this loads after plugin CSS file but before WordPress Additional CSS
            wp_add_inline_style('arewa-apk-frontend', $custom_css);
        }
        
        // Enqueue JavaScript
        $js_file_path = AREWA_APK_PLUGIN_DIR . 'assets/js/frontend.js';
        $js_version = AREWA_APK_VERSION;
        if (file_exists($js_file_path)) {
            $js_version .= '.' . filemtime($js_file_path);
        }
        
        wp_enqueue_script('arewa-apk-frontend', AREWA_APK_PLUGIN_URL . 'assets/js/frontend.js', array('jquery'), $js_version, true);
        
        // Enqueue carousel script
        $carousel_js_path = AREWA_APK_PLUGIN_DIR . 'assets/js/carousel-script.js';
        $carousel_js_version = AREWA_APK_VERSION;
        if (file_exists($carousel_js_path)) {
            $carousel_js_version .= '.' . filemtime($carousel_js_path);
        }
        wp_enqueue_script('arewa-apk-carousel-script', AREWA_APK_PLUGIN_URL . 'assets/js/carousel-script.js', array('jquery'), $carousel_js_version, true);
        
        // Get upload size limit
        $upload_size_limit = wp_max_upload_size();
        $max_file_size = min($upload_size_limit, 2 * 1024 * 1024); // 2MB or WordPress limit, whichever is smaller
        
        wp_localize_script('arewa-apk-frontend', 'arewaAPK', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('arewapap_apk_nonce'),
            'maxUploadSize' => $max_file_size,
            'strings' => array(
                'error' => __('An error occurred. Please try again.', 'arewa-apk-apps-manager'),
                'success' => __('Thank you for your comment!', 'arewa-apk-apps-manager'),
                'required' => __('This field is required.', 'arewa-apk-apps-manager'),
                'invalidEmail' => __('Invalid email address.', 'arewa-apk-apps-manager'),
                'submitting' => __('Submitting...', 'arewa-apk-apps-manager'),
                'submit' => __('Submit Comment', 'arewa-apk-apps-manager'),
                'invalidFileType' => __('Please select a valid image file (JPEG, PNG, GIF, or WebP).', 'arewa-apk-apps-manager'),
                'fileTooLarge' => __('File size exceeds limit. Please select a smaller image.', 'arewa-apk-apps-manager'),
            )
        ));
    }
    
    public function enqueue_admin_assets($hook) {
        // Load admin assets on plugin admin pages
        // Check for both full hook name and partial match
        $load_assets = false;
        
        // Check if we're on plugin admin pages
        if (strpos($hook, 'arewa-apk') !== false || $hook === 'toplevel_page_arewa-apk-settings') {
            $load_assets = true;
        }
        
        // Check if we're on comments page
        if (isset($_GET['page'])) {
            $page = sanitize_text_field(wp_unslash($_GET['page']));
            if ($page === 'arewa-apk-comments') {
                $load_assets = true;
            }
        }
        
        // Check if we're on add/edit app pages
        if (isset($_GET['page'])) {
            $page = sanitize_text_field(wp_unslash($_GET['page']));
            if (in_array($page, array('arewa-apk-add-app', 'arewa-apk-edit-app'))) {
                $load_assets = true;
            }
        }
        
        // Check if we're on import-export page
        if (isset($_GET['page'])) {
            $page = sanitize_text_field(wp_unslash($_GET['page']));
            if ($page === 'arewa-apk-import-export') {
                $load_assets = true;
            }
        }
        
        // Check if we're on the apps list page (for Quick Edit)
        if ($hook === 'edit.php' && isset($_GET['post_type'])) {
            $post_type = sanitize_text_field(wp_unslash($_GET['post_type']));
            if ($post_type === 'arewa_apk_app') {
                $load_assets = true;
            }
        }
        
        // Check if we're editing or creating an APK App post
        if (in_array($hook, array('post.php', 'post-new.php'))) {
            $post_type = isset($_GET['post_type']) ? sanitize_text_field(wp_unslash($_GET['post_type'])) : '';
            if (empty($post_type) && isset($_GET['post'])) {
                $post_id = absint($_GET['post']);
                $post = get_post($post_id);
                if ($post && $post->post_type === 'arewa_apk_app') {
                    $load_assets = true;
                }
            } elseif ($post_type === 'arewa_apk_app') {
                $load_assets = true;
            }
        }
        
        if (!$load_assets) {
            return;
        }
        
        // Enqueue media uploader
        wp_enqueue_media();
        
        // Enqueue admin CSS
        $admin_css_path = AREWA_APK_PLUGIN_DIR . 'assets/css/admin.css';
        $admin_css_version = AREWA_APK_VERSION;
        if (file_exists($admin_css_path)) {
            $admin_css_version .= '.' . filemtime($admin_css_path);
        }
        wp_enqueue_style('arewa-apk-admin', AREWA_APK_PLUGIN_URL . 'assets/css/admin.css', array(), $admin_css_version);
        
        // Enqueue admin comments CSS
        $current_page = isset($_GET['page']) ? sanitize_text_field(wp_unslash($_GET['page'])) : '';
        if ($current_page === 'arewa-apk-comments') {
            $comments_css_path = AREWA_APK_PLUGIN_DIR . 'assets/css/admin-comments.css';
            $comments_css_version = AREWA_APK_VERSION;
            if (file_exists($comments_css_path)) {
                $comments_css_version .= '.' . filemtime($comments_css_path);
            }
            wp_enqueue_style('arewa-apk-admin-comments', AREWA_APK_PLUGIN_URL . 'assets/css/admin-comments.css', array('arewa-apk-admin'), $comments_css_version);
        }
        
        // Enqueue admin import-export CSS and JS
        if ($current_page === 'arewa-apk-import-export') {
            $import_export_css_path = AREWA_APK_PLUGIN_DIR . 'assets/css/admin-import-export.css';
            $import_export_css_version = AREWA_APK_VERSION;
            if (file_exists($import_export_css_path)) {
                $import_export_css_version .= '.' . filemtime($import_export_css_path);
            }
            wp_enqueue_style('arewa-apk-admin-import-export', AREWA_APK_PLUGIN_URL . 'assets/css/admin-import-export.css', array('arewa-apk-admin'), $import_export_css_version);
            
            $import_export_js_path = AREWA_APK_PLUGIN_DIR . 'assets/js/admin-import-export.js';
            $import_export_js_version = AREWA_APK_VERSION;
            if (file_exists($import_export_js_path)) {
                $import_export_js_version .= '.' . filemtime($import_export_js_path);
            }
            wp_enqueue_script('arewa-apk-admin-import-export', AREWA_APK_PLUGIN_URL . 'assets/js/admin-import-export.js', array('jquery'), $import_export_js_version, true);
            
            wp_localize_script('arewa-apk-admin-import-export', 'arewaAPKAdminImportExport', array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'strings' => array(
                    'exporting' => __('Exporting...', 'arewa-apk-apps-manager'),
                    'importing' => __('Importing...', 'arewa-apk-apps-manager'),
                    'importingLong' => __('Importing... This may take a few minutes.', 'arewa-apk-apps-manager'),
                    'downloadExportFile' => __('Download Export File', 'arewa-apk-apps-manager'),
                    'exportError' => __('An error occurred during export.', 'arewa-apk-apps-manager'),
                    'importError' => __('An error occurred during import.', 'arewa-apk-apps-manager'),
                    'importTimeout' => __('Import timeout. The file may be too large. Please try again or contact support.', 'arewa-apk-apps-manager'),
                    'selectAppsToExport' => __('Please select at least one app to export.', 'arewa-apk-apps-manager'),
                    'imported' => __('Imported:', 'arewa-apk-apps-manager'),
                    'skipped' => __('Skipped:', 'arewa-apk-apps-manager'),
                    'errors' => __('Errors:', 'arewa-apk-apps-manager'),
                )
            ));
        }
        
        // Enqueue admin add-edit-app JS
        if (in_array($current_page, array('arewa-apk-add-app', 'arewa-apk-edit-app'))) {
            $add_edit_js_path = AREWA_APK_PLUGIN_DIR . 'assets/js/admin-add-edit-app.js';
            $add_edit_js_version = AREWA_APK_VERSION;
            if (file_exists($add_edit_js_path)) {
                $add_edit_js_version .= '.' . filemtime($add_edit_js_path);
            }
            wp_enqueue_script('arewa-apk-admin-add-edit-app', AREWA_APK_PLUGIN_URL . 'assets/js/admin-add-edit-app.js', array('jquery'), $add_edit_js_version, true);
            
            wp_localize_script('arewa-apk-admin-add-edit-app', 'arewaAPKAdminAddEdit', array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'strings' => array(
                    'appNameRequired' => __('App Name is required.', 'arewa-apk-apps-manager'),
                    'appNameMaxLength' => __('App Name must be 33 characters or less.', 'arewa-apk-apps-manager'),
                    'shortDescRequired' => __('Short Description is required.', 'arewa-apk-apps-manager'),
                    'appVersionRequired' => __('App Version is required.', 'arewa-apk-apps-manager'),
                    'appIconRequired' => __('App Icon is required.', 'arewa-apk-apps-manager'),
                    'featureGraphicRequired' => __('Feature Graphic is required.', 'arewa-apk-apps-manager'),
                    'screenshotsRequired' => __('At least one App Screenshot is required.', 'arewa-apk-apps-manager'),
                    'apkUrlRequired' => __('APK Download URL is required.', 'arewa-apk-apps-manager'),
                    'aboutRequired' => __('About this app is required.', 'arewa-apk-apps-manager'),
                    'categoryRequired' => __('Please select a category for your app.', 'arewa-apk-apps-manager'),
                    'maxTagsReached' => __('You can select a maximum of 10 tags.', 'arewa-apk-apps-manager'),
                    'selected' => __('selected', 'arewa-apk-apps-manager'),
                )
            ));
        }
        
        // Enqueue admin JavaScript
        $admin_js_path = AREWA_APK_PLUGIN_DIR . 'assets/js/admin.js';
        $admin_js_version = AREWA_APK_VERSION;
        if (file_exists($admin_js_path)) {
            $admin_js_version .= '.' . filemtime($admin_js_path);
        }
        wp_enqueue_script('arewa-apk-admin', AREWA_APK_PLUGIN_URL . 'assets/js/admin.js', array('jquery', 'media-upload', 'media-views'), $admin_js_version, true);
        
        // Localize script for admin AJAX
        wp_localize_script('arewa-apk-admin', 'arewaAPKAdmin', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('arewapap_apk_admin_nonce'),
            'strings' => array(
                'replySaved' => __('Reply saved successfully!', 'arewa-apk-apps-manager'),
                'replyDeleted' => __('Reply deleted successfully!', 'arewa-apk-apps-manager'),
                'error' => __('An error occurred. Please try again.', 'arewa-apk-apps-manager'),
            )
        ));
    }
    
    public function render_admin_page() {
        include AREWA_APK_PLUGIN_DIR . 'includes/admin/settings-page.php';
    }
    
    public function render_comments_page() {
        include AREWA_APK_PLUGIN_DIR . 'includes/admin/comments-page.php';
    }
    
    /**
     * Load Import/Export handlers
     */
    private function load_import_export_handlers() {
        if (!class_exists('Arewa_APK_Import_Export_AJAX')) {
            require_once AREWA_APK_PLUGIN_DIR . 'includes/admin/import-export/helpers-media.php';
            require_once AREWA_APK_PLUGIN_DIR . 'includes/admin/import-export/export.php';
            require_once AREWA_APK_PLUGIN_DIR . 'includes/admin/import-export/import.php';
            require_once AREWA_APK_PLUGIN_DIR . 'includes/admin/import-export/ajax-import.php';
            
            // Initialize AJAX handlers
            if (class_exists('Arewa_APK_Import_Export_AJAX')) {
                Arewa_APK_Import_Export_AJAX::init();
            }
        }
    }
    
    public function render_import_export_page() {
        // Ensure Import/Export modules are loaded
        $this->load_import_export_handlers();
        
        include AREWA_APK_PLUGIN_DIR . 'includes/admin/import-export/page.php';
    }
    
    /**
     * Render Add New App page (custom, not Gutenberg)
     */
    public function render_add_app_page() {
        // Check permissions
        if (!current_user_can('edit_posts')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'arewa-apk-apps-manager'));
        }
        
        // Handle form submission - must be before any output
        if (isset($_POST['arewa_apk_save_app'])) {
            // Verify nonce
            if (!check_admin_referer('arewa_apk_save_app', 'arewa_apk_app_nonce')) {
                wp_die(esc_html__('Security check failed. Please try again.', 'arewa-apk-apps-manager'));
            }
            
            $save_and_continue = isset($_POST['save_and_continue']);
            $app_id = $this->save_app_from_form(0, $save_and_continue);
            
            // Only redirect if save was successful (app_id > 0)
            if ($app_id && $app_id > 0) {
                // Ensure no output has been sent
                if (!headers_sent()) {
                    if ($save_and_continue) {
                        // Redirect to edit page with the new app ID
                        wp_safe_redirect(admin_url('admin.php?page=arewa-apk-edit-app&app_id=' . $app_id . '&saved=1'));
                        exit;
                    } else {
                        // Redirect to apps list (edit.php for custom post type)
                        wp_safe_redirect(admin_url('edit.php?post_type=arewa_apk_app&app_created=1'));
                        exit;
                    }
                } else {
                    // Headers already sent, use JavaScript redirect as fallback
                    // Note: This is a fallback when headers are already sent, so we must use echo
                    $redirect_url = $save_and_continue 
                        ? admin_url('admin.php?page=arewa-apk-edit-app&app_id=' . $app_id . '&saved=1')
                        : admin_url('edit.php?post_type=arewa_apk_app&app_created=1');
                    $script = 'window.location.href = ' . wp_json_encode($redirect_url) . ';';
                    echo '<script type="text/javascript">' . "\n";
                    echo esc_js($script) . "\n";
                    echo '</script>' . "\n";
                    exit;
                }
            }
            // If app_id is false/0, errors were shown, stay on page to display them
        }
        
        include AREWA_APK_PLUGIN_DIR . 'includes/admin/add-edit-app-page.php';
    }
    
    /**
     * Render Edit App page (custom, not Gutenberg)
     */
    public function render_edit_app_page() {
        // Check permissions
        if (!current_user_can('edit_posts')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'arewa-apk-apps-manager'));
        }
        
        // Get app ID
        $app_id = isset($_GET['app_id']) ? absint($_GET['app_id']) : 0;
        
        if (!$app_id) {
            wp_die(esc_html__('Invalid app ID.', 'arewa-apk-apps-manager'));
        }
        
        $app = get_post($app_id);
        if (!$app || $app->post_type !== 'arewa_apk_app') {
            wp_die(esc_html__('App not found.', 'arewa-apk-apps-manager'));
        }
        
        // Handle form submission
        if (isset($_POST['arewa_apk_save_app']) && check_admin_referer('arewa_apk_save_app', 'arewa_apk_app_nonce')) {
            $save_and_continue = isset($_POST['save_and_continue']);
            $this->save_app_from_form($app_id, $save_and_continue);
            
            // Show success message for both buttons
            // This avoids white screen issues with redirects
            add_action('admin_notices', function() use ($save_and_continue) {
                $message = $save_and_continue 
                    ? __('App saved successfully!', 'arewa-apk-apps-manager')
                    : __('App updated successfully!', 'arewa-apk-apps-manager');
                echo '<div class="notice notice-success is-dismissible"><p>' . esc_html($message) . '</p></div>';
            });
        }
        
        include AREWA_APK_PLUGIN_DIR . 'includes/admin/add-edit-app-page.php';
    }
    
    /**
     * Save app from custom form
     */
    private function save_app_from_form($app_id = 0, $save_and_continue = false) {
        // Get app_id from POST if not provided (for edit mode)
        if ($app_id == 0 && isset($_POST['app_id'])) {
            $app_id = absint($_POST['app_id']);
        }
        
        // Sanitize and validate input
        $app_name = isset($_POST['app_name']) ? sanitize_text_field($_POST['app_name']) : '';
        // Limit App Name to 33 characters
        if (strlen($app_name) > 33) {
            $app_name = substr($app_name, 0, 33);
        }
        $app_version = isset($_POST['arewa_apk_version']) ? sanitize_text_field($_POST['arewa_apk_version']) : '';
        $app_icon = isset($_POST['arewa_apk_icon']) ? esc_url_raw($_POST['arewa_apk_icon']) : '';
        $feature_graphic = isset($_POST['arewa_apk_feature_graphic']) ? esc_url_raw($_POST['arewa_apk_feature_graphic']) : '';
        $apk_url = isset($_POST['arewa_apk_url']) ? esc_url_raw($_POST['arewa_apk_url']) : '';
        $short_description = isset($_POST['arewa_apk_short_description']) ? sanitize_text_field($_POST['arewa_apk_short_description']) : '';
        // Limit to 100 characters
        if (strlen($short_description) > 100) {
            $short_description = substr($short_description, 0, 100);
        }
        $about_app = isset($_POST['arewa_apk_about']) ? wp_kses_post($_POST['arewa_apk_about']) : '';
        $comments_per_page = isset($_POST['arewa_apk_comments_per_page']) ? absint($_POST['arewa_apk_comments_per_page']) : 10;
        $disable_seo = isset($_POST['arewa_apk_disable_seo']) ? sanitize_text_field($_POST['arewa_apk_disable_seo']) : 'no';
        
        // Handle screenshots
        $app_screenshots = array();
        if (isset($_POST['arewa_apk_screenshots']) && is_array($_POST['arewa_apk_screenshots'])) {
            foreach ($_POST['arewa_apk_screenshots'] as $screenshot) {
                $url = esc_url_raw($screenshot);
                if (!empty($url)) {
                    $app_screenshots[] = $url;
                }
            }
        }
        
        // Validate all required fields
        $errors = array();
        
        if (empty($app_name)) {
            $errors[] = __('App name is required.', 'arewa-apk-apps-manager');
        }
        
        if (empty($short_description)) {
            $errors[] = __('Short description is required.', 'arewa-apk-apps-manager');
        }
        
        if (empty($app_version)) {
            $errors[] = __('App version is required.', 'arewa-apk-apps-manager');
        }
        
        if (empty($app_icon)) {
            $errors[] = __('App icon is required.', 'arewa-apk-apps-manager');
        }
        
        if (empty($feature_graphic)) {
            $errors[] = __('Feature graphic is required.', 'arewa-apk-apps-manager');
        }
        
        if (empty($app_screenshots)) {
            $errors[] = __('At least one app screenshot is required.', 'arewa-apk-apps-manager');
        }
        
        if (empty($apk_url)) {
            $errors[] = __('APK download URL is required.', 'arewa-apk-apps-manager');
        }
        
        // Check if about_app has actual content (not just empty tags)
        $about_app_stripped = wp_strip_all_tags($about_app);
        $about_app_stripped = trim($about_app_stripped);
        if (empty($about_app_stripped)) {
            $errors[] = __('About this app is required.', 'arewa-apk-apps-manager');
        }
        
        // Validate category before creating post
        if (!isset($_POST['arewa_apk_category']) || empty($_POST['arewa_apk_category'])) {
            $errors[] = __('App category is required.', 'arewa-apk-apps-manager');
        }
        
        if (!empty($errors)) {
            add_action('admin_notices', function() use ($errors) {
                echo '<div class="notice notice-error"><p><strong>' . esc_html__('Please fix the following errors:', 'arewa-apk-apps-manager') . '</strong></p><ul>';
                foreach ($errors as $error) {
                    echo '<li>' . esc_html($error) . '</li>';
                }
                echo '</ul></div>';
            });
            return false;
        }
        
        // Prepare post data
        $post_data = array(
            'post_title' => $app_name,
            'post_type' => 'arewa_apk_app',
            'post_status' => isset($_POST['post_status']) ? sanitize_text_field($_POST['post_status']) : 'publish',
        );
        
        // Handle post date if provided
        if (isset($_POST['post_date']) && !empty($_POST['post_date'])) {
            $post_date = sanitize_text_field($_POST['post_date']);
            $post_time = isset($_POST['post_time']) ? sanitize_text_field($_POST['post_time']) : current_time('H:i:s');
            
            // Combine date and time
            $post_date_string = $post_date . ' ' . $post_time . ':00';
            $post_date_gmt = get_gmt_from_date($post_date_string);
            
            $post_data['post_date'] = $post_date_string;
            $post_data['post_date_gmt'] = $post_date_gmt;
            
            // Get current post status to preserve it if date is changed
            $current_post = $app_id > 0 ? get_post($app_id) : null;
            $current_status = $current_post ? $current_post->post_status : 'publish';
            
            // Check if date is in the past or present
            $post_timestamp = strtotime($post_date_string);
            $current_timestamp = current_time('timestamp');
            
            // Always respect the explicitly set status from the form
            // If user selected "Published", keep it as "Published" even if date is in future
            if (isset($post_data['post_status']) && $post_data['post_status'] === 'publish') {
                // User explicitly wants it published - keep it as publish regardless of date
                $post_data['post_status'] = 'publish';
            } elseif (!isset($post_data['post_status']) || empty($post_data['post_status'])) {
                // Status was not explicitly set, preserve current status
                if ($post_timestamp <= $current_timestamp) {
                    // Date is in past or present - keep as publish if it was published
                    if ($current_status === 'publish' || $current_status === 'future') {
                        $post_data['post_status'] = 'publish';
                    } else {
                        $post_data['post_status'] = $current_status;
                    }
                } else {
                    // Date is in the future - preserve current status (don't auto-change to future)
                    $post_data['post_status'] = $current_status;
                }
            } else {
                // Status was explicitly set - but if date is past and status is future, change to publish
                if ($post_timestamp <= $current_timestamp && $post_data['post_status'] === 'future') {
                    $post_data['post_status'] = 'publish';
                }
            }
        }
        
        if ($app_id > 0) {
            $post_data['ID'] = $app_id;
            $result = wp_update_post($post_data, true); // true = wp_error on failure
        } else {
            $result = wp_insert_post($post_data, true); // true = wp_error on failure
            if (!is_wp_error($result)) {
                $app_id = $result;
            }
        }
        
        if (is_wp_error($result)) {
            add_action('admin_notices', function() use ($result) {
                echo '<div class="notice notice-error"><p>' . esc_html($result->get_error_message()) . '</p></div>';
            });
            return false;
        }
        
        // Verify that post was created/updated successfully
        if (empty($app_id) || $app_id === 0) {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error"><p>' . esc_html__('Failed to create app. Please try again.', 'arewa-apk-apps-manager') . '</p></div>';
            });
            return false;
        }
        
        // Force status to 'publish' if user explicitly selected it (WordPress may have changed it to 'future')
        if (isset($_POST['post_status']) && $_POST['post_status'] === 'publish') {
            $saved_post = get_post($app_id);
            if ($saved_post && $saved_post->post_status === 'future') {
                // WordPress changed it to future, but user wants it published - force it back
                wp_update_post(array(
                    'ID' => $app_id,
                    'post_status' => 'publish'
                ), true);
            }
        }
        
        // Save meta fields
        update_post_meta($app_id, '_arewa_apk_version', $app_version);
        update_post_meta($app_id, '_arewa_apk_icon', $app_icon);
        update_post_meta($app_id, '_arewa_apk_feature_graphic', $feature_graphic);
        update_post_meta($app_id, '_arewa_apk_screenshots', $app_screenshots);
        update_post_meta($app_id, '_arewa_apk_url', $apk_url);
        update_post_meta($app_id, '_arewa_apk_short_description', $short_description);
        update_post_meta($app_id, '_arewa_apk_about', $about_app);
        update_post_meta($app_id, '_arewa_apk_comments_per_page', $comments_per_page);
        update_post_meta($app_id, '_arewa_apk_disable_seo', $disable_seo);
        
        // Handle category (only one allowed) - already validated above
        if (isset($_POST['arewa_apk_category']) && !empty($_POST['arewa_apk_category'])) {
            $category_id = absint($_POST['arewa_apk_category']);
            wp_set_object_terms($app_id, array($category_id), 'arewa_apk_category');
        }
        
        // Handle tags (up to 10 allowed)
        if (isset($_POST['arewa_apk_tags']) && is_array($_POST['arewa_apk_tags'])) {
            $tags = array_map('absint', $_POST['arewa_apk_tags']);
            // Limit to 10 tags
            if (count($tags) > 10) {
                $tags = array_slice($tags, 0, 10);
            }
            wp_set_object_terms($app_id, $tags, 'arewa_apk_tag');
        } else {
            // If no tags selected, remove all tags
            wp_set_object_terms($app_id, array(), 'arewa_apk_tag');
        }
        
        // Return the app_id so calling function can handle redirect
        return $app_id;
    }
    
    public function render_shortcode($atts) {
        $atts = shortcode_atts(array(
            'id' => '', // Post ID of the app
            'slug' => '', // Post slug of the app
            'display' => 'single', // 'single' or 'list' or 'archive' or 'carousel'
            'show_icon' => 'yes',
            'show_screenshot' => 'yes',
            'show_version' => 'yes',
            'show_comments' => 'yes',
            'category' => '', // Category slug for filtering apps in list mode
            'posts_per_page' => '', // Number of posts per page for list mode
            'limit' => '', // Number of posts to show in carousel mode (alias for posts_per_page)
            'title' => '', // Custom title for list/carousel mode
            'description' => '', // Custom description for list mode
            'has_pagination' => 'yes', // Show pagination: 'yes' or 'no'
        ), $atts, 'arewa_apk_app');
        
        // If 'limit' is provided, use it as posts_per_page (for carousel compatibility)
        if (!empty($atts['limit']) && empty($atts['posts_per_page'])) {
            $atts['posts_per_page'] = $atts['limit'];
        }
        
        // If display is 'list' or 'archive', show the archive/list
        if ($atts['display'] === 'list' || $atts['display'] === 'archive') {
            ob_start();
            include AREWA_APK_PLUGIN_DIR . 'includes/frontend/archive-display.php';
            return ob_get_clean();
        }
        
        // If display is 'carousel', show the carousel
        if ($atts['display'] === 'carousel') {
            ob_start();
            include AREWA_APK_PLUGIN_DIR . 'includes/frontend/carousel-display.php';
            return ob_get_clean();
        }
        
        // Otherwise, show single app (default behavior)
        // Determine which app to display
        $app_id = null;
        
        // First check if we're on a single app page (highest priority)
        if (is_singular('arewa_apk_app')) {
            global $post;
            if ($post && $post->post_type === 'arewa_apk_app') {
                $app_id = $post->ID;
            }
        }
        
        // Then check if ID is provided in shortcode
        if (!$app_id && !empty($atts['id'])) {
            $app_id = absint($atts['id']);
        }
        
        // Then check if slug is provided
        if (!$app_id && !empty($atts['slug'])) {
            $app = get_page_by_path($atts['slug'], OBJECT, 'arewa_apk_app');
            if ($app) {
                $app_id = $app->ID;
            }
        }
        
        // If no app ID found, try to use legacy settings (backward compatibility)
        if (!$app_id) {
            // Use legacy settings mode
            $app_id = 0;
        }
        
        // Make variables available to the included template
        $app_id = $app_id ? $app_id : 0;
        
        ob_start();
        include AREWA_APK_PLUGIN_DIR . 'includes/frontend/display.php';
        return ob_get_clean();
    }
    
    public function ajax_submit_comment() {
        try {
            // Check if comments are enabled globally
            $plugin_settings = self::get_plugin_settings();
            if ($plugin_settings['enable_comments'] !== 'yes') {
                wp_send_json_error(array('message' => __('Comments are currently disabled.', 'arewa-apk-apps-manager')));
                return;
            }
            
            // Verify nonce
            if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'arewapap_apk_nonce')) {
                wp_send_json_error(array('message' => __('Security check failed. Please refresh the page and try again.', 'arewa-apk-apps-manager')));
                return;
            }
            
            // Sanitize input
            $author_name = isset($_POST['author_name']) ? sanitize_text_field($_POST['author_name']) : '';
            $author_email = isset($_POST['author_email']) ? sanitize_email($_POST['author_email']) : '';
            $comment_text = isset($_POST['comment_text']) ? sanitize_textarea_field($_POST['comment_text']) : '';
            $rating = isset($_POST['rating']) ? absint($_POST['rating']) : 0;
            $author_avatar = '';
            
            // Validate required fields
        if (empty($author_name) || empty($author_email) || empty($comment_text)) {
            wp_send_json_error(array('message' => __('All fields are required.', 'arewa-apk-apps-manager')));
                return;
        }
        
            // Validate email
        if (!is_email($author_email)) {
            wp_send_json_error(array('message' => __('Invalid email address.', 'arewa-apk-apps-manager')));
                return;
            }
            
            // Handle avatar upload (optional) - only if allowed in settings
            $allow_avatar_upload = $plugin_settings['allow_avatar_upload'] ?? 'yes';
            if ($allow_avatar_upload === 'yes' && !empty($_FILES['author_avatar']) && isset($_FILES['author_avatar']['error']) && $_FILES['author_avatar']['error'] === UPLOAD_ERR_OK) {
                // Validate file type
                $allowed_types = array('image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp');
                $file_type = wp_check_filetype($_FILES['author_avatar']['name']);
                $mime_type = isset($_FILES['author_avatar']['type']) ? $_FILES['author_avatar']['type'] : '';
                
                // Check if file type is valid
                $is_valid_type = in_array($mime_type, $allowed_types) || (isset($file_type['type']) && in_array($file_type['type'], $allowed_types));
                
                if ($is_valid_type) {
                    // Validate file size (max 2MB)
                    $max_size = 2 * 1024 * 1024; // 2MB
                    $file_size = isset($_FILES['author_avatar']['size']) ? $_FILES['author_avatar']['size'] : 0;
                    $is_valid_size = $file_size > 0 && $file_size <= $max_size;
                    
                    if ($is_valid_size) {
                        require_once(ABSPATH . 'wp-admin/includes/file.php');
                        require_once(ABSPATH . 'wp-admin/includes/image.php');
                        require_once(ABSPATH . 'wp-admin/includes/media.php');
                        
                        $upload_overrides = array(
                            'test_form' => false,
                            'mimes' => array(
                                'jpg|jpeg|jpe' => 'image/jpeg',
                                'png' => 'image/png',
                                'gif' => 'image/gif',
                                'webp' => 'image/webp'
                            )
                        );
                        
                        $uploaded_file = wp_handle_upload($_FILES['author_avatar'], $upload_overrides);
                        
                        if ($uploaded_file && !isset($uploaded_file['error'])) {
                            $author_avatar = isset($uploaded_file['url']) ? $uploaded_file['url'] : '';
                            
                            // Create attachment if URL is available
                            if (!empty($author_avatar) && isset($uploaded_file['file'])) {
                                $attachment = array(
                                    'post_mime_type' => isset($uploaded_file['type']) ? $uploaded_file['type'] : 'image/jpeg',
                                    'post_title' => sanitize_file_name(basename($uploaded_file['file'])),
                                    'post_content' => '',
                                    'post_status' => 'inherit'
                                );
                                
                                $attach_id = wp_insert_attachment($attachment, $uploaded_file['file']);
                                if (!is_wp_error($attach_id)) {
                                    $attach_data = wp_generate_attachment_metadata($attach_id, $uploaded_file['file']);
                                    wp_update_attachment_metadata($attach_id, $attach_data);
                                }
                            }
                        }
                        // If upload fails, just continue without avatar - don't block comment
                    }
                    // If size is too large, just continue without avatar - don't block comment
                }
                // If type is invalid, just continue without avatar - don't block comment
            }
            
            // Get app_id from POST (if provided)
            $app_id = isset($_POST['app_id']) ? absint($_POST['app_id']) : 0;
            
            // Insert comment into database
        global $wpdb;
        $table_name = $wpdb->prefix . 'arewa_apk_comments';
        
            // Ensure table exists - this also adds the author_avatar column if needed
            $this->create_comment_tables();
            
            // Prepare data for insertion - include avatar field (it should exist after create_comment_tables)
            $data = array(
                'app_id' => $app_id,
                'author_name' => $author_name,
                'author_email' => $author_email,
                'author_avatar' => $author_avatar ? $author_avatar : '',
                'comment_text' => $comment_text,
                'rating' => $rating,
                'ip_address' => $this->get_client_ip(),
                'approved' => ($plugin_settings['require_approval'] === 'yes') ? 0 : 1,
            );
            
            $format = array('%d', '%s', '%s', '%s', '%s', '%d', '%s', '%d');
            
            // Try to insert comment
            $inserted = $wpdb->insert($table_name, $data, $format);
            
            // If insert failed, try without avatar column (for backward compatibility)
            if ($inserted === false && !empty($wpdb->last_error)) {
                // Check if error is due to missing column
                $error_message = $wpdb->last_error;
                if (strpos($error_message, 'author_avatar') !== false || strpos($error_message, 'Unknown column') !== false) {
                    // Remove avatar from data and try again
                    unset($data['author_avatar']);
                    unset($format[2]); // Remove avatar format
                    $format = array_values($format); // Re-index array
                    
                    // Try insert without avatar
                    $inserted = $wpdb->insert($table_name, $data, $format);
                }
            }
            
            if ($inserted !== false && $inserted > 0) {
                // Clear cache when a new comment is added
                $this->clear_cache();
                
                // Get updated stats
                $total_comments = self::get_comments_count($app_id);
                $average_rating = self::get_average_rating($app_id);
                $rating_count = self::get_rating_count($app_id);
                
                // Set success message based on approval requirement
                $success_message = __('Comment submitted successfully!', 'arewa-apk-apps-manager');
                if ($plugin_settings['require_approval'] === 'yes') {
                    $success_message = __('Comment submitted successfully! It will be visible after admin approval.', 'arewa-apk-apps-manager');
                }
                
                wp_send_json_success(array(
                    'message' => $success_message,
                    'total_comments' => $total_comments,
                    'average_rating' => $average_rating,
                    'rating_count' => $rating_count
                ));
        } else {
                // Get last error for debugging
                $error_message = $wpdb->last_error;
                if (empty($error_message)) {
                    $error_message = __('Failed to submit comment. Please try again.', 'arewa-apk-apps-manager');
                }
                
                // Log detailed error information
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('Arewa APK Comment Insert Error: ' . $error_message);
                    error_log('Arewa APK Comment Data: ' . print_r($data, true));
                    error_log('Arewa APK Format: ' . print_r($format, true));
                    error_log('Arewa APK Table Name: ' . $table_name);
                    error_log('Arewa APK Insert Result: ' . var_export($inserted, true));
                }
                
                // Send error message to user
                wp_send_json_error(array('message' => $error_message));
            }
        } catch (Exception $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Arewa APK Comment Error: ' . $e->getMessage());
                error_log('Arewa APK Comment Stack Trace: ' . $e->getTraceAsString());
            }
            wp_send_json_error(array('message' => __('An error occurred. Please try again.', 'arewa-apk-apps-manager')));
        }
    }
    
    public function ajax_submit_rating() {
        check_ajax_referer('arewa_apk_nonce', 'nonce');
        // Handle rating submission
        wp_send_json_success();
    }
    
    public function ajax_submit_vote() {
        check_ajax_referer('arewa_apk_nonce', 'nonce');
        
        $comment_id = absint($_POST['comment_id'] ?? 0);
        $vote_type = sanitize_text_field($_POST['vote_type'] ?? '');
        
        if (empty($comment_id) || !in_array($vote_type, array('helpful', 'not_helpful'))) {
            wp_send_json_error(array('message' => __('Invalid vote data.', 'arewa-apk-apps-manager')));
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'arewa_apk_comments';
        
        // Get current vote counts
        // Escape table name for security
        $table_name_escaped = esc_sql($table_name);
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        // Table names cannot use placeholders, but are escaped via esc_sql()
        $comment = $wpdb->get_row($wpdb->prepare(
            "SELECT vote_helpful, vote_not_helpful FROM `{$table_name_escaped}` WHERE id = %d",
            absint($comment_id)
        ));
        
        if (!$comment) {
            wp_send_json_error(array('message' => __('Comment not found.', 'arewa-apk-apps-manager')));
        }
        
        $helpful_count = absint($comment->vote_helpful);
        $not_helpful_count = absint($comment->vote_not_helpful);
        
        // Get user's current vote from cookie
        $cookie_name = 'arewa_vote_' . $comment_id;
        $current_vote = isset($_COOKIE[$cookie_name]) ? sanitize_text_field($_COOKIE[$cookie_name]) : '';
        
        // Determine new vote state
        if ($current_vote === $vote_type) {
            // User clicked the same button - remove vote (toggle off)
            if ($vote_type === 'helpful') {
                $helpful_count = max(0, $helpful_count - 1);
            } else {
                $not_helpful_count = max(0, $not_helpful_count - 1);
            }
            $new_vote = '';
        } else {
            // User clicked different button or first time voting
            // First, remove old vote if exists
            if ($current_vote === 'helpful') {
                $helpful_count = max(0, $helpful_count - 1);
            } else if ($current_vote === 'not_helpful') {
                $not_helpful_count = max(0, $not_helpful_count - 1);
            }
            
            // Then add new vote
            if ($vote_type === 'helpful') {
                $helpful_count = $helpful_count + 1;
            } else {
                $not_helpful_count = $not_helpful_count + 1;
            }
            $new_vote = $vote_type;
        }
        
        // Update database
        $updated = $wpdb->update(
            $table_name,
            array(
                'vote_helpful' => $helpful_count,
                'vote_not_helpful' => $not_helpful_count
            ),
            array('id' => $comment_id),
            array('%d', '%d'),
            array('%d')
        );
        
        if ($updated !== false) {
            // Set cookie to remember user's vote (30 days)
            // Note: setcookie() only works if headers haven't been sent yet
            // For AJAX requests, we'll return the cookie value and let JavaScript set it
            $cookie_expires = time() + (30 * 24 * 60 * 60);
            
            // Try to set cookie via PHP (will work if headers not sent)
            if (!headers_sent()) {
                if (!empty($new_vote)) {
                    setcookie($cookie_name, $new_vote, $cookie_expires, '/', '', is_ssl(), false);
                } else {
                    // Remove cookie if vote is removed
                    setcookie($cookie_name, '', time() - 3600, '/', '', is_ssl(), false);
                }
            }
            
            wp_send_json_success(array(
                'helpful_count' => $helpful_count,
                'not_helpful_count' => $not_helpful_count,
                'current_vote' => $new_vote,
                'cookie_name' => $cookie_name,
                'cookie_value' => $new_vote,
                'cookie_expires' => $cookie_expires
            ));
        } else {
            wp_send_json_error(array('message' => __('Failed to update vote.', 'arewa-apk-apps-manager')));
        }
    }
    
    private function get_client_ip() {
        $ip_keys = array('HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR');
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                $server_value = sanitize_text_field(wp_unslash($_SERVER[$key]));
                foreach (explode(',', $server_value) as $ip) {
                    $ip = trim($ip);
                    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                        return $ip;
                    }
                }
            }
        }
        return isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '';
    }
    
    public function register_elementor_widget($widgets_manager) {
        if (!did_action('elementor/loaded')) {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-warning"><p>' . esc_html__('Elementor is required for Arewa APK App Elementor widget to work.', 'arewa-apk-apps-manager') . '</p></div>';
            });
            return;
        }
        
        require_once AREWA_APK_PLUGIN_DIR . 'includes/elementor/widget.php';
        $widgets_manager->register(new \Arewa_APK_Elementor_Widget());
    }
    
    public static function get_settings() {
        return get_option('arewa_apk_settings', array());
    }
    
    /**
     * Get global plugin settings
     */
    public static function get_plugin_settings() {
        $defaults = array(
            'enable_comments' => 'yes',
            'allow_avatar_upload' => 'yes',
            'comments_per_page' => 10,
            'require_approval' => 'no',
        );
        return get_option('arewa_apk_plugin_settings', $defaults);
    }
    
    /**
     * Sanitize plugin settings
     */
    public function sanitize_plugin_settings($input) {
        $sanitized = array();
        $sanitized['enable_comments'] = isset($input['enable_comments']) && $input['enable_comments'] === 'yes' ? 'yes' : 'no';
        $sanitized['allow_avatar_upload'] = isset($input['allow_avatar_upload']) && $input['allow_avatar_upload'] === 'yes' ? 'yes' : 'no';
        $sanitized['comments_per_page'] = absint($input['comments_per_page'] ?? 10);
        $sanitized['require_approval'] = isset($input['require_approval']) && $input['require_approval'] === 'yes' ? 'yes' : 'no';
        
        // Archive list display settings
        $sanitized['show_download_button'] = isset($input['show_download_button']) && $input['show_download_button'] === 'yes' ? 'yes' : 'no';
        $sanitized['show_short_description'] = isset($input['show_short_description']) && $input['show_short_description'] === 'yes' ? 'yes' : 'no';
        $sanitized['show_categories'] = isset($input['show_categories']) && $input['show_categories'] === 'yes' ? 'yes' : 'no';
        $sanitized['show_version'] = isset($input['show_version']) && $input['show_version'] === 'yes' ? 'yes' : 'no';
        $sanitized['show_last_updated'] = isset($input['show_last_updated']) && $input['show_last_updated'] === 'yes' ? 'yes' : 'no';
        $sanitized['show_download_count'] = isset($input['show_download_count']) && $input['show_download_count'] === 'yes' ? 'yes' : 'no';
        $sanitized['show_rating'] = isset($input['show_rating']) && $input['show_rating'] === 'yes' ? 'yes' : 'no';
        $sanitized['show_excerpt'] = isset($input['show_excerpt']) && $input['show_excerpt'] === 'yes' ? 'yes' : 'no';
        $sanitized['posts_per_page'] = absint($input['posts_per_page'] ?? get_option('posts_per_page', 10));
        
        return $sanitized;
    }
    
    public static function get_comments($page = 1, $per_page = 10, $sort_by = 'date', $order = 'desc', $app_id = 0) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'arewa_apk_comments';
        $offset = ($page - 1) * $per_page;
        
        // Validate and sanitize inputs
        $allowed_sort_by = array('date', 'rating', 'votes', 'helpful', 'not_helpful');
        $allowed_order = array('asc', 'desc');
        
        $sort_by = sanitize_key($sort_by);
        if (!in_array($sort_by, $allowed_sort_by, true)) {
            $sort_by = 'date';
        }
        
        $order = sanitize_key($order);
        if (!in_array($order, $allowed_order, true)) {
            $order = 'desc';
        }
        
        // Sanitize numeric values
        $per_page_int = absint($per_page);
        $offset_int = absint($offset);
        $app_id_int = absint($app_id);
        $page_int = absint($page);
        
        // Whitelist of allowed column names for ORDER BY
        $allowed_columns = array(
            'comment_date',
            'rating',
            'vote_helpful',
            'vote_not_helpful'
        );
        
        // Build ORDER BY clause using whitelist
        $order_direction = ('ASC' === strtoupper($order)) ? 'ASC' : 'DESC';
        $order_by_clause = '';
        
        switch ($sort_by) {
            case 'date':
                $order_by_clause = 'comment_date ' . $order_direction;
                break;
            case 'rating':
                $order_by_clause = 'rating ' . $order_direction . ', comment_date DESC';
                break;
            case 'votes':
                // Total votes expression - safe because it's hardcoded
                $order_by_clause = '(vote_helpful + vote_not_helpful) ' . $order_direction . ', comment_date DESC';
                break;
            case 'helpful':
                $order_by_clause = 'vote_helpful ' . $order_direction . ', comment_date DESC';
                break;
            case 'not_helpful':
                $order_by_clause = 'vote_not_helpful ' . $order_direction . ', comment_date DESC';
                break;
            default:
                $order_by_clause = 'comment_date DESC';
                break;
        }
        
        // Validate ORDER BY clause contains only allowed columns
        // Extract column names from ORDER BY clause
        $order_by_parts = preg_split('/\s*,\s*/', $order_by_clause);
        $valid_order_by_parts = array();
        
        foreach ($order_by_parts as $part) {
            $part = trim($part);
            // Handle expressions like "(vote_helpful + vote_not_helpful) DESC"
            if (preg_match('/^\(([^)]+)\)\s+(ASC|DESC)$/i', $part, $matches)) {
                $expr = trim($matches[1]);
                $dir = strtoupper(trim($matches[2]));
                // Validate expression contains only allowed columns
                if (preg_match('/^vote_helpful\s*\+\s*vote_not_helpful$/i', $expr)) {
                    $valid_order_by_parts[] = '(' . $expr . ') ' . $dir;
                }
            } elseif (preg_match('/^([a-z_]+)\s+(ASC|DESC)$/i', $part, $matches)) {
                $col = trim($matches[1]);
                $dir = strtoupper(trim($matches[2]));
                // Validate column is in whitelist
                if (in_array($col, $allowed_columns, true)) {
                    $valid_order_by_parts[] = $col . ' ' . $dir;
                }
            } elseif (preg_match('/^([a-z_]+)$/i', $part, $matches)) {
                $col = trim($matches[1]);
                // Validate column is in whitelist
                if (in_array($col, $allowed_columns, true)) {
                    $valid_order_by_parts[] = $col;
                }
            }
        }
        
        // If validation failed, use default
        if (empty($valid_order_by_parts)) {
            $valid_order_by_parts = array('comment_date DESC');
        }
        
        $order_by_safe = implode(', ', $valid_order_by_parts);
        
        // Build WHERE clause with placeholders
        $where_clause = 'WHERE approved = %d';
        $where_values = array(1);
        
        // Add app_id filter
        if ($app_id_int > 0) {
            $where_clause .= ' AND app_id = %d';
            $where_values[] = $app_id_int;
        } else {
            $where_clause .= ' AND app_id = %d';
            $where_values[] = 0;
        }
        
        // Build complete query with ORDER BY (ORDER BY cannot use placeholders, but it's validated from whitelist)
        // Table name is safe because it's built from $wpdb->prefix
        // Use backticks for table name to prevent issues with reserved words
        $table_name_safe = $wpdb->_escape($table_name);
        
        // Build query - ORDER BY is validated from whitelist, all other values use placeholders
        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        // ORDER BY clause is validated from whitelist, table name is escaped, all values use placeholders
        $query = "SELECT * FROM `{$table_name_safe}` {$where_clause} ORDER BY {$order_by_safe} LIMIT %d OFFSET %d";
        // phpcs:enable
        $where_values[] = $per_page_int;
        $where_values[] = $offset_int;
        
        // Prepare query with placeholders
        // Query string is built from validated whitelist values (table name escaped, ORDER BY validated),
        // all dynamic values use placeholders in $where_values array
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- $query contains validated whitelist values and placeholders, will be prepared by $wpdb->prepare()
        // phpcs:ignore PluginCheck.Security.DirectDB.UnescapedDBParameter -- $query is built from validated whitelist, all user input uses placeholders
        $prepared_query = $wpdb->prepare($query, $where_values);
        
        // Execute query
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query is prepared above with $wpdb->prepare() using placeholders
        // phpcs:ignore PluginCheck.Security.DirectDB.UnescapedDBParameter -- $prepared_query is the result of $wpdb->prepare(), all values are safely escaped
        $comments = $wpdb->get_results($prepared_query);
        
        return $comments;
    }
    
    public static function get_comments_count($app_id = 0) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'arewa_apk_comments';
        // Escape table name for security
        $table_name_escaped = esc_sql($table_name);
        
        // Count all approved comments - no user permission check needed
        // Use prepared statement for safety
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        // Table names cannot use placeholders, but are escaped via esc_sql()
        if ($app_id > 0) {
            $count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM `{$table_name_escaped}` WHERE approved = %d AND app_id = %d",
                1,
                absint($app_id)
            ));
        } else {
            // For legacy mode, only count comments with app_id = 0
            $count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM `{$table_name_escaped}` WHERE approved = %d AND app_id = %d",
                1,
                0
            ));
        }
        return absint($count);
    }
    
    public static function get_average_rating($app_id = 0) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'arewa_apk_comments';
        // Escape table name for security
        $table_name_escaped = esc_sql($table_name);
        
        // Get average rating from all approved comments with ratings
        // Use prepared statement for safety
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        // Table names cannot use placeholders, but are escaped via esc_sql()
        if ($app_id > 0) {
            $avg_rating = $wpdb->get_var($wpdb->prepare(
                "SELECT AVG(rating) FROM `{$table_name_escaped}` WHERE approved = %d AND rating > 0 AND app_id = %d",
                1,
                absint($app_id)
            ));
        } else {
            // For legacy mode, only count comments with app_id = 0
            $avg_rating = $wpdb->get_var($wpdb->prepare(
                "SELECT AVG(rating) FROM `{$table_name_escaped}` WHERE approved = %d AND rating > 0 AND app_id = %d",
                1,
                0
            ));
        }
        return $avg_rating ? round(floatval($avg_rating), 2) : 0;
    }
    
    public static function get_rating_count($app_id = 0) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'arewa_apk_comments';
        // Escape table name for security
        $table_name_escaped = esc_sql($table_name);
        
        // Count all approved comments with ratings
        // Use prepared statement for safety
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        // Table names cannot use placeholders, but are escaped via esc_sql()
        if ($app_id > 0) {
            $count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM `{$table_name_escaped}` WHERE approved = %d AND rating > 0 AND app_id = %d",
                1,
                absint($app_id)
            ));
        } else {
            // For legacy mode, only count comments with app_id = 0
            $count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM `{$table_name_escaped}` WHERE approved = %d AND rating > 0 AND app_id = %d",
                1,
                0
            ));
        }
        return absint($count);
    }
    
    public function ajax_admin_reply() {
        // Verify user is admin
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'arewapap_apk_admin_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed. Please refresh the page and try again.', 'arewa-apk-apps-manager')));
            return;
        }
        
        try {
            $comment_id = isset($_POST['comment_id']) ? absint($_POST['comment_id']) : 0;
            $reply_text = isset($_POST['reply_text']) ? sanitize_textarea_field($_POST['reply_text']) : '';
            $reply_url = isset($_POST['reply_url']) ? esc_url_raw($_POST['reply_url']) : '';
            $reply_image = '';
            $reply_image_url = isset($_POST['reply_image_url']) ? esc_url_raw($_POST['reply_image_url']) : '';
            
            if (empty($comment_id)) {
                wp_send_json_error(array('message' => __('Invalid comment ID.', 'arewa-apk-apps-manager')));
                return;
            }
            
            // Handle image: first check if file is uploaded, otherwise use URL from media library
            if (!empty($_FILES['reply_image']) && isset($_FILES['reply_image']['error']) && $_FILES['reply_image']['error'] === UPLOAD_ERR_OK) {
                // Validate file type
                $allowed_types = array('image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp');
                $file_type = wp_check_filetype($_FILES['reply_image']['name']);
                $mime_type = isset($_FILES['reply_image']['type']) ? $_FILES['reply_image']['type'] : '';
                
                $is_valid_type = in_array($mime_type, $allowed_types) || (isset($file_type['type']) && in_array($file_type['type'], $allowed_types));
                
                if ($is_valid_type) {
                    $max_size = 2 * 1024 * 1024; // 2MB
                    $file_size = isset($_FILES['reply_image']['size']) ? $_FILES['reply_image']['size'] : 0;
                    $is_valid_size = $file_size > 0 && $file_size <= $max_size;
                    
                    if ($is_valid_size) {
                        require_once(ABSPATH . 'wp-admin/includes/file.php');
                        require_once(ABSPATH . 'wp-admin/includes/image.php');
                        require_once(ABSPATH . 'wp-admin/includes/media.php');
                        
                        $upload_overrides = array(
                            'test_form' => false,
                            'mimes' => array(
                                'jpg|jpeg|jpe' => 'image/jpeg',
                                'png' => 'image/png',
                                'gif' => 'image/gif',
                                'webp' => 'image/webp'
                            )
                        );
                        
                        $uploaded_file = wp_handle_upload($_FILES['reply_image'], $upload_overrides);
                        
                        if ($uploaded_file && !isset($uploaded_file['error'])) {
                            $reply_image = isset($uploaded_file['url']) ? $uploaded_file['url'] : '';
                            
                            // Create attachment
                            if (!empty($reply_image) && isset($uploaded_file['file'])) {
                                $attachment = array(
                                    'post_mime_type' => isset($uploaded_file['type']) ? $uploaded_file['type'] : 'image/jpeg',
                                    'post_title' => sanitize_file_name(basename($uploaded_file['file'])),
                                    'post_content' => '',
                                    'post_status' => 'inherit'
                                );
                                
                                $attach_id = wp_insert_attachment($attachment, $uploaded_file['file']);
                                if (!is_wp_error($attach_id)) {
                                    $attach_data = wp_generate_attachment_metadata($attach_id, $uploaded_file['file']);
                                    wp_update_attachment_metadata($attach_id, $attach_data);
                                }
                            }
                        }
                    }
                }
            } elseif (!empty($reply_image_url)) {
                // Use image URL from media library if no file was uploaded
                $reply_image = $reply_image_url;
            }
            
            // Update comment with admin reply
            global $wpdb;
            $table_name = $wpdb->prefix . 'arewa_apk_comments';
            
            // Ensure table exists and has required columns
            $this->create_comment_tables();
            
            // Check if columns exist and prepare data
            $table_name_escaped = esc_sql($table_name);
            
            // Get all columns to check structure
            $all_columns = $wpdb->get_col("DESCRIBE `{$table_name_escaped}`");
            
            // If get_col fails, try alternative method
            if (empty($all_columns) || !is_array($all_columns)) {
                $columns_result = $wpdb->get_results("SHOW COLUMNS FROM `{$table_name_escaped}`", ARRAY_A);
                $all_columns = array();
                if (is_array($columns_result)) {
                    foreach ($columns_result as $col) {
                        if (isset($col['Field'])) {
                            $all_columns[] = $col['Field'];
                        }
                    }
                }
            }
            
            // If still empty, log error
            if (empty($all_columns)) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('Arewa APK Admin Reply: Could not retrieve columns from table ' . $table_name);
                }
                wp_send_json_error(array('message' => __('Could not access database table. Please check the error logs.', 'arewa-apk-apps-manager')));
                return;
            }
            
            // Ensure all required columns exist
            $required_columns = array(
                'admin_reply_text' => 'text DEFAULT NULL',
                'admin_reply_image' => 'varchar(255) DEFAULT NULL',
                'admin_reply_url' => 'varchar(255) DEFAULT NULL',
                'admin_reply_date' => 'datetime DEFAULT NULL'
            );
            
            // Check and add missing columns
            $columns_added = false;
            foreach ($required_columns as $col_name => $col_def) {
                if (!in_array($col_name, $all_columns)) {
                    if (defined('WP_DEBUG') && WP_DEBUG) {
                        error_log('Arewa APK Admin Reply: Column ' . $col_name . ' not found, adding...');
                    }
                    
                    // Suppress errors while adding column
                    // Escape column name for security (col_def is from whitelist, safe)
                    $col_name_escaped = esc_sql($col_name);
                    // $col_def comes from a whitelist array, so it's safe
                    // But we'll validate it contains only safe characters
                    if (preg_match('/^[a-zA-Z0-9_()\s,]+$/i', $col_def)) {
                        $wpdb->hide_errors();
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                        $add_result = $wpdb->query("ALTER TABLE `{$table_name_escaped}` ADD COLUMN `{$col_name_escaped}` {$col_def}");
                        $add_error = $wpdb->last_error;
                        $wpdb->show_errors();
                    } else {
                        if (defined('WP_DEBUG') && WP_DEBUG) {
                            error_log('Arewa APK Admin Reply: Invalid column definition for ' . $col_name);
                        }
                        $add_result = false;
                        $add_error = 'Invalid column definition';
                    }
                    
                    if ($add_result !== false && empty($add_error)) {
                        $all_columns[] = $col_name;
                        $columns_added = true;
                        if (defined('WP_DEBUG') && WP_DEBUG) {
                            error_log('Arewa APK Admin Reply: Successfully added column ' . $col_name);
                        }
                    } else {
                        if (defined('WP_DEBUG') && WP_DEBUG) {
                            error_log('Arewa APK Admin Reply: Failed to add column ' . $col_name . '. Error: ' . $add_error);
                        }
                    }
                    
                    // Clear error before next iteration
                    $wpdb->last_error = '';
                }
            }
            
            // Re-check columns if any were added
            if ($columns_added) {
                $all_columns = $wpdb->get_col("DESCRIBE `{$table_name_escaped}`");
                $wpdb->last_error = ''; // Clear any errors from DESCRIBE
            }
            
            // Verify all required columns exist
            $missing_cols = array();
            foreach (array_keys($required_columns) as $col_name) {
                if (!in_array($col_name, $all_columns)) {
                    $missing_cols[] = $col_name;
                }
            }
            
            if (!empty($missing_cols)) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('Arewa APK Admin Reply: Missing columns after add attempt: ' . implode(', ', $missing_cols));
                    error_log('Arewa APK Admin Reply: Available columns: ' . implode(', ', $all_columns));
                }
                wp_send_json_error(array('message' => __('Database columns could not be created. Please check the error logs and try again.', 'arewa-apk-apps-manager')));
                return;
            }
            
            // Use all_columns for verification
            $columns = $all_columns;
            
            // Prepare data for update - always include all fields
            $data = array(
                'admin_reply_text' => $reply_text,
                'admin_reply_image' => $reply_image,
                'admin_reply_url' => $reply_url,
                'admin_reply_date' => current_time('mysql')
            );
            
            // Log data being sent
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Arewa APK Admin Reply: Updating comment ' . $comment_id);
                error_log('Arewa APK Admin Reply: Data: ' . print_r($data, true));
                error_log('Arewa APK Admin Reply: Available columns: ' . print_r($columns, true));
            }
            
            // Check if comment exists first
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            // Table names cannot use placeholders, but are escaped via esc_sql()
            $comment_exists = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM `{$table_name_escaped}` WHERE id = %d",
                $comment_id
            ));
            
            if ($comment_exists == 0) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('Arewa APK Admin Reply: Comment ' . $comment_id . ' not found');
                }
                wp_send_json_error(array('message' => __('Comment not found.', 'arewa-apk-apps-manager')));
                return;
            }
            
            // Clear any previous errors before update
            $wpdb->last_error = '';
            
            // Prepare data array with proper null handling
            // Convert empty strings to null for proper database handling
            $update_data = array(
                'admin_reply_text' => !empty($reply_text) ? $reply_text : null,
                'admin_reply_image' => !empty($reply_image) ? $reply_image : null,
                'admin_reply_url' => !empty($reply_url) ? $reply_url : null,
                'admin_reply_date' => current_time('mysql')
            );
            
            // Use $wpdb->update for cleaner code
            // Format: %s for strings, %d for integers
            $updated = $wpdb->update(
                $table_name,
                $update_data,
                array('id' => $comment_id),
                array('%s', '%s', '%s', '%s'), // Format for all fields (text, image, url, date)
                array('%d') // Format for WHERE clause (id)
            );
            
            // Check for errors
            $db_error = $wpdb->last_error;
            
            // $wpdb->update returns:
            // - The number of rows affected (0 or more) on success
            // - false on error
            if ($updated === false) {
                // Error occurred
                $error_message = !empty($db_error) ? $db_error : __('Database update failed. Please check the error logs.', 'arewa-apk-apps-manager');
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('Arewa APK Admin Reply Update Error: ' . $error_message);
                    error_log('Arewa APK Admin Reply Update Result: ' . var_export($updated, true));
                    error_log('Arewa APK Admin Reply Data: ' . print_r($update_data, true));
                    error_log('Arewa APK Admin Reply Comment ID: ' . $comment_id);
                    error_log('Arewa APK Admin Reply Table: ' . $table_name);
                    error_log('Arewa APK Admin Reply Columns: ' . print_r($columns, true));
                    error_log('Arewa APK Admin Reply Last Query: ' . $wpdb->last_query);
                }
                
                wp_send_json_error(array('message' => $error_message));
                return;
            } elseif (!empty($db_error)) {
                // Database error occurred but update might have succeeded
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('Arewa APK Admin Reply Database Error: ' . $db_error);
                    error_log('Arewa APK Admin Reply Update Result: ' . var_export($updated, true));
                    error_log('Arewa APK Admin Reply Last Query: ' . $wpdb->last_query);
                }
                
                // Check if update actually succeeded despite error
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                // Table names cannot use placeholders, but are escaped via esc_sql()
                $verify_comment = $wpdb->get_row($wpdb->prepare(
                    "SELECT admin_reply_text, admin_reply_image, admin_reply_url, admin_reply_date FROM `{$table_name_escaped}` WHERE id = %d",
                    $comment_id
                ));
                
                // Verify if update succeeded by checking if comment has reply data
                if ($verify_comment && (
                    (!empty($reply_text) && $verify_comment->admin_reply_text) ||
                    (!empty($reply_url) && $verify_comment->admin_reply_url) ||
                    (!empty($reply_image) && $verify_comment->admin_reply_image) ||
                    $verify_comment->admin_reply_date
                )) {
                    // Update succeeded despite error message
                    if (defined('WP_DEBUG') && WP_DEBUG) {
                        error_log('Arewa APK Admin Reply: Update verified successful despite error message');
                    }
                    $reply_date = !empty($verify_comment->admin_reply_date) ? $verify_comment->admin_reply_date : current_time('mysql');
                    
                    wp_send_json_success(array(
                        'message' => __('Reply saved successfully!', 'arewa-apk-apps-manager'),
                        'reply_text' => !empty($verify_comment->admin_reply_text) ? $verify_comment->admin_reply_text : $reply_text,
                        'reply_image' => !empty($verify_comment->admin_reply_image) ? $verify_comment->admin_reply_image : $reply_image,
                        'reply_url' => !empty($verify_comment->admin_reply_url) ? $verify_comment->admin_reply_url : $reply_url,
                        'reply_date' => $reply_date
                    ));
                    return;
                } else {
                    // Update failed
                    wp_send_json_error(array('message' => __('Database error: ', 'arewa-apk-apps-manager') . $db_error));
                    return;
                }
            }
            
            // Success - $updated is the number of rows affected (0 or more)
            // 0 means no rows were changed (same values), which is acceptable
            $reply_date = $update_data['admin_reply_date'];
            
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Arewa APK Admin Reply: Successfully saved reply for comment ' . $comment_id . ', rows affected: ' . $updated);
            }
            
            wp_send_json_success(array(
                'message' => __('Reply saved successfully!', 'arewa-apk-apps-manager'),
                'reply_text' => $reply_text,
                'reply_image' => $reply_image,
                'reply_url' => $reply_url,
                'reply_date' => $reply_date
            ));
        } catch (Exception $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Arewa APK Admin Reply Error: ' . $e->getMessage());
            }
            wp_send_json_error(array('message' => __('An error occurred. Please try again.', 'arewa-apk-apps-manager')));
        }
    }
    
    public function ajax_delete_admin_reply() {
        // Verify user is admin
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'arewapap_apk_admin_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'arewa-apk-apps-manager')));
            return;
        }
        
        $comment_id = isset($_POST['comment_id']) ? absint($_POST['comment_id']) : 0;
        
        if (empty($comment_id)) {
            wp_send_json_error(array('message' => __('Invalid comment ID.', 'arewa-apk-apps-manager')));
            return;
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'arewa_apk_comments';
        
        $updated = $wpdb->update(
            $table_name,
            array(
                'admin_reply_text' => '',
                'admin_reply_image' => '',
                'admin_reply_url' => '',
                'admin_reply_date' => null,
            ),
            array('id' => $comment_id),
            array('%s', '%s', '%s', '%s'),
            array('%d')
        );
        
        if ($updated !== false) {
            wp_send_json_success(array('message' => __('Reply deleted successfully!', 'arewa-apk-apps-manager')));
        } else {
            wp_send_json_error(array('message' => __('Failed to delete reply.', 'arewa-apk-apps-manager')));
        }
    }
    
    public function ajax_filter_comments() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'arewa_apk_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Get settings for comments per page FIRST
        $settings = self::get_settings();
        $comments_per_page = absint($settings['comments_per_page'] ?? 10);
        if ($comments_per_page < 1) {
            $comments_per_page = 10;
        }
        
        $page = isset($_POST['page']) ? absint($_POST['page']) : 1;
        $sort_by = isset($_POST['sort_by']) ? sanitize_text_field($_POST['sort_by']) : 'date';
        $order = isset($_POST['order']) ? sanitize_text_field($_POST['order']) : 'desc';
        $app_id = isset($_POST['app_id']) ? absint($_POST['app_id']) : 0;
        
        // Validate inputs
        if ($page < 1) {
            $page = 1;
        }
        
        // Always use the per_page from settings, not from POST
        $per_page = $comments_per_page;
        
        // Get total comments count first
        $total_comments = self::get_comments_count($app_id);
        $total_pages = $per_page > 0 ? ceil($total_comments / $per_page) : 1;
        
        // Validate page number doesn't exceed total pages
        if ($page > $total_pages && $total_pages > 0) {
            $page = $total_pages;
        }
        
        // Get filtered comments
        $comments = self::get_comments($page, $per_page, $sort_by, $order, $app_id);
        
        // Prepare comments HTML
        ob_start();
        
        if (empty($comments)) {
            echo '<p class="arewa-no-comments">' . esc_html__('No comments yet. Be the first to review!', 'arewa-apk-apps-manager') . '</p>';
        } else {
            foreach ($comments as $comment) {
                // Get first letter of name for avatar fallback
                $first_letter = mb_substr(trim($comment->author_name), 0, 1);
                $first_letter = mb_strtoupper($first_letter);
                $author_avatar = !empty($comment->author_avatar) ? esc_url($comment->author_avatar) : '';
                
                // Generate color based on first letter
                $colors = array(
                    'A' => '#667eea', 'B' => '#764ba2', 'C' => '#f093fb', 'D' => '#4facfe',
                    'E' => '#43e97b', 'F' => '#fa709a', 'G' => '#fee140', 'H' => '#30cfd0',
                    'I' => '#a8edea', 'J' => '#fed6e3', 'K' => '#ff9a9e', 'L' => '#fecfef',
                    'M' => '#fecfef', 'N' => '#ffecd2', 'O' => '#fcb69f', 'P' => '#ff8a80',
                    'Q' => '#ffab91', 'R' => '#ffccbc', 'S' => '#ffe0b2', 'T' => '#fff9c4',
                    'U' => '#c5e1a5', 'V' => '#a5d6a7', 'W' => '#80cbc4', 'X' => '#b2dfdb',
                    'Y' => '#b3e5fc', 'Z' => '#81d4fa'
                );
                $avatar_color = isset($colors[$first_letter]) ? $colors[$first_letter] : '#667eea';
                
                // Hide email for security
                $email_parts = explode('@', $comment->author_email);
                if (count($email_parts) === 2) {
                    $email_local = $email_parts[0];
                    $email_domain = $email_parts[1];
                    $hidden_email = mb_substr($email_local, 0, 1) . '***@' . $email_domain;
                } else {
                    $hidden_email = '***@' . (isset($email_parts[1]) ? $email_parts[1] : 'gmail.com');
                }
                ?>
                <div class="arewa-comment-item">
                    <div class="arewa-comment-header">
                        <div class="arewa-comment-author-info">
                            <div class="arewa-comment-avatar" <?php if (!$author_avatar): ?>style="background: <?php echo esc_attr($avatar_color); ?>;"<?php endif; ?>>
                                <?php if ($author_avatar): ?>
                                    <img src="<?php echo esc_url($author_avatar); ?>" alt="<?php echo esc_attr($comment->author_name); ?>" />
                                <?php else: ?>
                                    <div class="arewa-comment-avatar-initial"><?php echo esc_html($first_letter); ?></div>
                                <?php endif; ?>
                            </div>
                            <div class="arewa-comment-author">
                                <strong><?php echo esc_html($comment->author_name); ?></strong>
                                <span class="arewa-comment-email"><?php echo esc_html($hidden_email); ?></span>
                            </div>
                        </div>
                        <?php if ($comment->rating > 0): ?>
                            <div class="arewa-comment-rating">
                                <?php for ($i = 1; $i <= 5; $i++): ?>
                                    <?php if ($i <= $comment->rating): ?>
                                        <span class="arewa-star arewa-star-filled" aria-label="<?php
                                        // translators: %d: Star rating number (1-5)
                                        echo esc_attr(sprintf(__('%d out of 5 stars', 'arewa-apk-apps-manager'), $i));
                                        ?>">★</span>
                                    <?php else: ?>
                                        <span class="arewa-star arewa-star-empty" aria-label="<?php
                                        // translators: %d: Star rating number (1-5)
                                        echo esc_attr(sprintf(__('%d out of 5 stars', 'arewa-apk-apps-manager'), $i));
                                        ?>">☆</span>
                                    <?php endif; ?>
                                <?php endfor; ?>
                                <span class="arewa-rating-text"><?php echo esc_html($comment->rating); ?>/5</span>
                            </div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="arewa-comment-date">
                        <?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($comment->comment_date))); ?>
                    </div>
                    
                    <div class="arewa-comment-text">
                        <?php echo nl2br(esc_html($comment->comment_text)); ?>
                    </div>
                    
                    <?php if (!empty($comment->admin_reply_text) || !empty($comment->admin_reply_image) || !empty($comment->admin_reply_url)): ?>
                        <div class="arewa-admin-reply">
                            <div class="arewa-admin-reply-header">
                                <span class="arewa-admin-reply-badge"><?php esc_html_e('Admin Reply', 'arewa-apk-apps-manager'); ?></span>
                                <?php if (!empty($comment->admin_reply_date)): ?>
                                    <span class="arewa-admin-reply-date"><?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($comment->admin_reply_date))); ?></span>
                                <?php endif; ?>
                            </div>
                            <div class="arewa-admin-reply-content">
                                <?php if (!empty($comment->admin_reply_text)): ?>
                                    <div class="arewa-admin-reply-text">
                                        <?php echo nl2br(esc_html($comment->admin_reply_text)); ?>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if (!empty($comment->admin_reply_image)): ?>
                                    <div class="arewa-admin-reply-image">
                                        <img src="<?php echo esc_url($comment->admin_reply_image); ?>" alt="<?php esc_attr_e('Admin reply image', 'arewa-apk-apps-manager'); ?>" />
                                    </div>
                                <?php endif; ?>
                                
                                <?php if (!empty($comment->admin_reply_url)): ?>
                                    <div class="arewa-admin-reply-url">
                                        <a href="<?php echo esc_url($comment->admin_reply_url); ?>" target="_blank" rel="noopener noreferrer" class="arewa-admin-reply-link">
                                            <span class="dashicons dashicons-external"></span>
                                            <?php echo esc_html($comment->admin_reply_url); ?>
                                        </a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endif; ?>
                    
                    <div class="arewa-comment-votes">
                        <button type="button" class="arewa-vote-btn arewa-vote-helpful" data-comment-id="<?php echo esc_attr($comment->id); ?>" data-vote-type="helpful">
                            <span class="arewa-vote-icon">👍</span>
                            <span class="vote-count"><?php echo absint($comment->vote_helpful); ?></span>
                            <span class="vote-text"><?php esc_html_e('Helpful', 'arewa-apk-apps-manager'); ?></span>
                        </button>
                        
                        <button type="button" class="arewa-vote-btn arewa-vote-not-helpful" data-comment-id="<?php echo esc_attr($comment->id); ?>" data-vote-type="not_helpful">
                            <span class="arewa-vote-icon">👎</span>
                            <span class="vote-count"><?php echo absint($comment->vote_not_helpful); ?></span>
                            <span class="vote-text"><?php esc_html_e('Not Helpful', 'arewa-apk-apps-manager'); ?></span>
                        </button>
                    </div>
                </div>
                <?php
            }
        }
        
        $comments_html = ob_get_clean();
        
        // Prepare pagination HTML
        $pagination_html = '';
        if ($total_pages > 1) {
            ob_start();
            // Build base URL - only include comment_page, not sort_by/order in URL
            // Use WordPress function to get current URL and remove unwanted params
            $current_url = remove_query_arg(array('comment_page', 'sort_by', 'order'));
            $page_args = array(
                'base' => add_query_arg('comment_page', '%#%', $current_url),
                'format' => '',
                'prev_text' => '&laquo; ' . __('Previous', 'arewa-apk-apps-manager'),
                'next_text' => __('Next', 'arewa-apk-apps-manager') . ' &raquo;',
                'total' => $total_pages,
                'current' => $page,
                'type' => 'list'
            );
            $page_links = paginate_links($page_args);
            if ($page_links) {
                // Don't wrap in div - the container already exists
                if ($page_links) {
                    echo wp_kses_post($page_links);
                }
                $pagination_html = ob_get_clean();
            } else {
                ob_end_clean();
                // If paginate_links returns false/empty, create fallback pagination
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('Arewa APK: paginate_links returned empty for page ' . $page . ' of ' . $total_pages);
                }
                // Fallback: create simple pagination manually
                $pagination_html = '<ul class="page-numbers">';
                if ($page > 1) {
                    $prev_url = add_query_arg('comment_page', $page - 1, remove_query_arg(array('comment_page', 'sort_by', 'order')));
                    $pagination_html .= '<li><a class="prev page-numbers" href="' . esc_url($prev_url) . '">&laquo; ' . __('Previous', 'arewa-apk-apps-manager') . '</a></li>';
                }
                for ($i = 1; $i <= $total_pages; $i++) {
                    if ($i == $page) {
                        $pagination_html .= '<li><span class="page-numbers current">' . $i . '</span></li>';
                    } else {
                        $page_url = add_query_arg('comment_page', $i, remove_query_arg(array('comment_page', 'sort_by', 'order')));
                        $pagination_html .= '<li><a class="page-numbers" href="' . esc_url($page_url) . '">' . $i . '</a></li>';
                    }
                }
                if ($page < $total_pages) {
                    $next_url = add_query_arg('comment_page', $page + 1, remove_query_arg(array('comment_page', 'sort_by', 'order')));
                    $pagination_html .= '<li><a class="next page-numbers" href="' . esc_url($next_url) . '">' . __('Next', 'arewa-apk-apps-manager') . ' &raquo;</a></li>';
                }
                $pagination_html .= '</ul>';
            }
        }
        
        // Get average rating and rating count for stats
        $average_rating = self::get_average_rating($app_id);
        $rating_count = self::get_rating_count($app_id);
        
        wp_send_json_success(array(
            'comments_html' => $comments_html,
            'pagination_html' => $pagination_html,
            'total_comments' => $total_comments,
            'total_pages' => $total_pages,
            'current_page' => $page,
            'per_page' => $per_page,
            'sort_by' => $sort_by,
            'order' => $order,
            'average_rating' => $average_rating,
            'rating_count' => $rating_count
        ));
    }
    
    public function ajax_track_download() {
        // Get app_id from request
        $app_id = isset($_POST['app_id']) ? absint($_POST['app_id']) : 0;
        
        if ($app_id > 0) {
            // Track download count for specific app (CPT)
            $download_count = absint(get_post_meta($app_id, '_arewa_apk_download_count', true) ?: 0);
            $download_count = $download_count + 1;
            update_post_meta($app_id, '_arewa_apk_download_count', $download_count);
        } else {
            // Legacy: Track download count globally
            $download_count = get_option('arewa_apk_download_count', 0);
            $download_count = absint($download_count) + 1;
            update_option('arewa_apk_download_count', $download_count);
        }
        
        wp_send_json_success(array('count' => $download_count));
    }
    
    public static function get_download_count() {
        return absint(get_option('arewa_apk_download_count', 0));
    }
    
    /**
     * Export settings to JSON file
     */
    public function ajax_export_settings() {
        // Verify user is admin
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'arewapap_export_settings')) {
            wp_send_json_error(array('message' => __('Security check failed. Please refresh the page and try again.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Get all settings (legacy)
        $settings = self::get_settings();
        $plugin_settings = self::get_plugin_settings();
        $download_count = self::get_download_count();
        
        // Get all apps (CPT posts)
        $apps = get_posts(array(
            'post_type' => 'arewa_apk_app',
            'posts_per_page' => -1,
            'post_status' => 'any',
        ));
        
        $apps_data = array();
        foreach ($apps as $app) {
            $app_data = array(
                'id' => $app->ID,
                'title' => $app->post_title,
                'status' => $app->post_status,
                'slug' => $app->post_name,
                'excerpt' => $app->post_excerpt,
                'version' => get_post_meta($app->ID, '_arewa_apk_version', true),
                'icon' => get_post_meta($app->ID, '_arewa_apk_icon', true),
                'screenshots' => get_post_meta($app->ID, '_arewa_apk_screenshots', true),
                'apk_url' => get_post_meta($app->ID, '_arewa_apk_url', true),
                'about_app' => get_post_meta($app->ID, '_arewa_apk_about', true),
                'comments_per_page' => get_post_meta($app->ID, '_arewa_apk_comments_per_page', true),
                'disable_seo' => get_post_meta($app->ID, '_arewa_apk_disable_seo', true),
                'download_count' => absint(get_post_meta($app->ID, '_arewa_apk_download_count', true) ?: 0),
            );
            
            // Get categories
            $categories = wp_get_object_terms($app->ID, 'arewa_apk_category', array('fields' => 'slugs'));
            if (!is_wp_error($categories)) {
                $app_data['categories'] = $categories;
            }
            
            $apps_data[] = $app_data;
        }
        
        // Prepare export data
        $export_data = array(
            'version' => AREWA_APK_VERSION,
            'export_date' => current_time('mysql'),
            'site_url' => home_url(),
            'legacy_settings' => $settings,
            'plugin_settings' => $plugin_settings,
            'apps' => $apps_data,
            'total_apps' => count($apps_data),
            'download_count' => $download_count,
        );
        
        // Create JSON file - use only JSON_PRETTY_PRINT for security (no flags that prevent escaping)
        $json_data = wp_json_encode($export_data, JSON_PRETTY_PRINT);
        
        // Generate filename
        $filename = 'arewa-apk-all-apps-' . gmdate('Y-m-d-His') . '.json';
        $upload_dir = wp_upload_dir();
        $file_path = $upload_dir['basedir'] . '/' . $filename;
        $file_url = $upload_dir['baseurl'] . '/' . $filename;
        
        // Write file
        $file_written = @file_put_contents($file_path, $json_data);
        
        if ($file_written !== false) {
            wp_send_json_success(array(
                // translators: %d: Number of apps exported
                'message' => sprintf(esc_html__('All apps exported successfully! (%d apps)', 'arewa-apk-apps-manager'), count($apps_data)),
                'file_url' => $file_url,
                'filename' => $filename,
                'total_apps' => count($apps_data)
            ));
        } else {
            wp_send_json_error(array('message' => __('Failed to create export file. Please check file permissions.', 'arewa-apk-apps-manager')));
        }
    }
    
    /**
     * Import settings from JSON file
     */
    public function ajax_import_settings() {
        // Verify user is admin
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'arewapap_import_settings')) {
            wp_send_json_error(array('message' => __('Security check failed. Please refresh the page and try again.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Check if file was uploaded
        if (!isset($_FILES['import_file']) || $_FILES['import_file']['error'] !== UPLOAD_ERR_OK) {
            wp_send_json_error(array('message' => __('No file uploaded or upload error occurred.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Validate file type
        $file_name = isset($_FILES['import_file']['name']) ? sanitize_file_name(wp_unslash($_FILES['import_file']['name'])) : '';
        $file_ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
        if ($file_ext !== 'json') {
            wp_send_json_error(array('message' => __('Invalid file type. Please upload a JSON file.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Read file contents
        $file_contents = @file_get_contents($_FILES['import_file']['tmp_name']);
        if ($file_contents === false) {
            wp_send_json_error(array('message' => __('Failed to read import file.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Parse JSON
        $import_data = json_decode($file_contents, true);
        if (json_last_error() !== JSON_ERROR_NONE || !is_array($import_data)) {
            wp_send_json_error(array('message' => __('Invalid JSON file. Please check the file format.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Import legacy settings (backward compatibility)
        if (isset($import_data['settings']) && is_array($import_data['settings'])) {
            $sanitized_settings = $this->sanitize_settings($import_data['settings']);
            update_option('arewa_apk_settings', $sanitized_settings);
        } elseif (isset($import_data['legacy_settings']) && is_array($import_data['legacy_settings'])) {
            $sanitized_settings = $this->sanitize_settings($import_data['legacy_settings']);
            update_option('arewa_apk_settings', $sanitized_settings);
        }
        
        // Import plugin settings
        if (isset($import_data['plugin_settings']) && is_array($import_data['plugin_settings'])) {
            $sanitized_plugin_settings = $this->sanitize_plugin_settings($import_data['plugin_settings']);
            update_option('arewa_apk_plugin_settings', $sanitized_plugin_settings);
        }
        
        // Import apps (CPT posts)
        $imported_apps_count = 0;
        if (isset($import_data['apps']) && is_array($import_data['apps'])) {
            foreach ($import_data['apps'] as $app_data) {
                // Prepare post data
                $post_data = array(
                    'post_title' => isset($app_data['title']) ? sanitize_text_field($app_data['title']) : '',
                    'post_type' => 'arewa_apk_app',
                    'post_status' => isset($app_data['status']) ? sanitize_text_field($app_data['status']) : 'draft',
                    'post_excerpt' => isset($app_data['excerpt']) ? sanitize_textarea_field($app_data['excerpt']) : '',
                );
                
                // If slug is provided, try to use it
                if (!empty($app_data['slug'])) {
                    $post_data['post_name'] = sanitize_title($app_data['slug']);
                }
                
                // Check if app with same ID or slug exists
                $existing_id = 0;
                if (!empty($app_data['id'])) {
                    $existing_post = get_post(absint($app_data['id']));
                    if ($existing_post && $existing_post->post_type === 'arewa_apk_app') {
                        $existing_id = $existing_post->ID;
                    }
                }
                
                // If no existing ID found, check by slug
                if (!$existing_id && !empty($app_data['slug'])) {
                    $existing_post = get_page_by_path($app_data['slug'], OBJECT, 'arewa_apk_app');
                    if ($existing_post) {
                        $existing_id = $existing_post->ID;
                    }
                }
                
                if ($existing_id > 0) {
                    $post_data['ID'] = $existing_id;
                    $result = wp_update_post($post_data);
                } else {
                    $result = wp_insert_post($post_data);
                    $existing_id = $result;
                }
                
                if (!is_wp_error($result) && $existing_id > 0) {
                    // Save meta fields
                    if (isset($app_data['version'])) {
                        update_post_meta($existing_id, '_arewa_apk_version', sanitize_text_field($app_data['version']));
                    }
                    if (isset($app_data['icon'])) {
                        update_post_meta($existing_id, '_arewa_apk_icon', esc_url_raw($app_data['icon']));
                    }
                    if (isset($app_data['screenshots']) && is_array($app_data['screenshots'])) {
                        $screenshots = array();
                        foreach ($app_data['screenshots'] as $screenshot) {
                            $screenshots[] = esc_url_raw($screenshot);
                        }
                        update_post_meta($existing_id, '_arewa_apk_screenshots', $screenshots);
                    }
                    if (isset($app_data['apk_url'])) {
                        update_post_meta($existing_id, '_arewa_apk_url', esc_url_raw($app_data['apk_url']));
                    }
                    if (isset($app_data['about_app'])) {
                        update_post_meta($existing_id, '_arewa_apk_about', wp_kses_post($app_data['about_app']));
                    }
                    if (isset($app_data['comments_per_page'])) {
                        update_post_meta($existing_id, '_arewa_apk_comments_per_page', absint($app_data['comments_per_page']));
                    }
                    if (isset($app_data['disable_seo'])) {
                        update_post_meta($existing_id, '_arewa_apk_disable_seo', sanitize_text_field($app_data['disable_seo']));
                    }
                    if (isset($app_data['download_count'])) {
                        update_post_meta($existing_id, '_arewa_apk_download_count', absint($app_data['download_count']));
                    }
                    
                    // Import categories
                    if (isset($app_data['categories']) && is_array($app_data['categories'])) {
                        $category_ids = array();
                        foreach ($app_data['categories'] as $category_slug) {
                            $term = get_term_by('slug', $category_slug, 'arewa_apk_category');
                            if ($term) {
                                $category_ids[] = $term->term_id;
                            }
                        }
                        if (!empty($category_ids)) {
                            wp_set_object_terms($existing_id, $category_ids, 'arewa_apk_category');
                        }
                    }
                    
                    $imported_apps_count++;
                }
            }
        }
        
        // Import download count if available
        if (isset($import_data['download_count']) && is_numeric($import_data['download_count'])) {
            update_option('arewa_apk_download_count', absint($import_data['download_count']));
        }
        
        // Clear cache
        $this->clear_cache();
        
        // translators: %d: Number of apps imported
        $message = sprintf(esc_html__('Import completed successfully! %d apps imported.', 'arewa-apk-apps-manager'), $imported_apps_count);
        
        wp_send_json_success(array(
            'message' => $message,
            'imported_apps' => $imported_apps_count,
            'imported_version' => isset($import_data['version']) ? $import_data['version'] : __('Unknown', 'arewa-apk-apps-manager'),
            'export_date' => isset($import_data['export_date']) ? $import_data['export_date'] : __('Unknown', 'arewa-apk-apps-manager')
        ));
    }
    
    // Import/Export methods have been moved to includes/admin/import-export/
    // See: export.php, import.php, ajax-import.php, helpers-media.php
    // The old methods below are kept for reference but are no longer used
    
    /**
     * @deprecated This method has been moved to Arewa_APK_Import_Export_AJAX::ajax_export_content()
     * @see includes/admin/import-export/ajax-import.php
     */
    public function ajax_export_content() {
        // Method moved to Arewa_APK_Import_Export_AJAX class
        // This stub is kept for backward compatibility
        wp_send_json_error(array('message' => __('This method has been deprecated. Please refresh the page.', 'arewa-apk-apps-manager')));
    }
    
    /**
     * @deprecated This method has been moved to Arewa_APK_Import_Export_AJAX::ajax_import_content()
     * @see includes/admin/import-export/ajax-import.php
     */
    public function ajax_import_content() {
        // Method moved to Arewa_APK_Import_Export_AJAX class
        // This stub is kept for backward compatibility
        wp_send_json_error(array('message' => __('This method has been deprecated. Please refresh the page.', 'arewa-apk-apps-manager')));
    }
    
    /**
     * @deprecated Helper methods have been moved to Arewa_APK_Media_Helper class
     * @see includes/admin/import-export/helpers-media.php
     */
    private function get_attachment_path_from_url($url) {
        // Method moved to Arewa_APK_Media_Helper class
        if (class_exists('Arewa_APK_Media_Helper')) {
            return Arewa_APK_Media_Helper::get_attachment_path_from_url($url);
        }
        return false;
    }
    
    /**
     * @deprecated Helper methods have been moved to Arewa_APK_Media_Helper class
     * @see includes/admin/import-export/helpers-media.php
     */
    private function import_media_file($file_path, $title = '') {
        // Method moved to Arewa_APK_Media_Helper class
        if (class_exists('Arewa_APK_Media_Helper')) {
            return Arewa_APK_Media_Helper::import_media_file($file_path, $title);
        }
        return false;
    }
    
    /**
     * @deprecated Helper methods have been moved to Arewa_APK_Media_Helper class
     * @see includes/admin/import-export/helpers-media.php
     */
    private function delete_directory($dir) {
        // Method moved to Arewa_APK_Media_Helper class
        if (class_exists('Arewa_APK_Media_Helper')) {
            return Arewa_APK_Media_Helper::delete_directory($dir);
        }
        return false;
    }
    
    // Old import/export code has been moved to:
    // - includes/admin/import-export/export.php
    // - includes/admin/import-export/import.php  
    // - includes/admin/import-export/ajax-import.php
    // - includes/admin/import-export/helpers-media.php
    
    /**
     * Check if a major SEO plugin is active
     * Returns the plugin identifier if found, false otherwise
     * Also checks via is_plugin_active for more reliable detection
     */
    private function is_seo_plugin_active() {
        // Check for Rank Math SEO
        if (defined('RANK_MATH_VERSION') || class_exists('RankMath') || 
            (function_exists('is_plugin_active') && is_plugin_active('seo-by-rank-math/rank-math.php'))) {
            return 'rank_math';
        }
        
        // Check for Yoast SEO
        if (defined('WPSEO_VERSION') || class_exists('WPSEO_Options') || 
            (function_exists('is_plugin_active') && is_plugin_active('wordpress-seo/wp-seo.php'))) {
            return 'yoast';
        }
        
        // Check for All in One SEO
        if (defined('AIOSEO_VERSION') || class_exists('AIOSEO') || 
            (function_exists('is_plugin_active') && is_plugin_active('all-in-one-seo-pack/all_in_one_seo_pack.php'))) {
            return 'aioseo';
        }
        
        // Check for SEOPress
        if (defined('SEOPRESS_VERSION') || function_exists('seopress_get_service') || 
            (function_exists('is_plugin_active') && is_plugin_active('wp-seopress/seopress.php'))) {
            return 'seopress';
        }
        
        // Check for The SEO Framework
        if (defined('THE_SEO_FRAMEWORK_VERSION') || class_exists('The_SEO_Framework') || 
            (function_exists('is_plugin_active') && is_plugin_active('autodescription/autodescription.php'))) {
            return 'tsf';
        }
        
        // Check for Squirrly SEO
        if (defined('SQ_VERSION') || class_exists('SQ_Classes_FrontController') || 
            (function_exists('is_plugin_active') && is_plugin_active('squirrly-seo/squirrly.php'))) {
            return 'squirrly';
        }
        
        // Check for SEOPressor
        if (defined('SEOPRESSOR_VERSION') || 
            (function_exists('is_plugin_active') && is_plugin_active('seo-pressor/seo-pressor.php'))) {
            return 'seopressor';
        }
        
        return false;
    }
    
    /**
     * Get SEO plugin name for display
     */
    public function get_seo_plugin_name() {
        $plugin = $this->is_seo_plugin_active();
        $names = array(
            'rank_math' => 'Rank Math SEO',
            'yoast' => 'Yoast SEO',
            'aioseo' => 'All in One SEO',
            'seopress' => 'SEOPress',
            'tsf' => 'The SEO Framework',
            'squirrly' => 'Squirrly SEO',
            'seopressor' => 'SEOPressor',
        );
        return isset($names[$plugin]) ? $names[$plugin] : false;
    }
    
    /**
     * Add SEO meta tags and structured data when shortcode is present on the page
     * This function adds Schema.org structured data, Open Graph, and Twitter Cards
     * Compatible with major SEO plugins - only adds SoftwareApplication schema and avoids conflicts
     */
    public function add_seo_meta_tags() {
        // Check if SEO is disabled in settings
        $settings = self::get_settings();
        $disable_seo = isset($settings['disable_seo']) && $settings['disable_seo'] === 'yes';
        if ($disable_seo) {
            return;
        }
        
        global $post;
        
        // Check if we're on a singular page/post
        if (!is_singular()) {
            return;
        }
        
        if (!is_a($post, 'WP_Post')) {
            return;
        }
        
        $app_id = 0;
        $app_name = '';
        $app_version = '';
        $app_icon = '';
        $app_screenshots = array();
        $apk_url = '';
        $short_description = '';
        $about_app = '';
        $disable_seo_app = 'no';
        
        // First, check if we're on a single app page (Custom Post Type)
        // This takes priority over shortcode detection
        if (is_singular('arewa_apk_app') && $post->post_type === 'arewa_apk_app') {
            $app_id = $post->ID;
            $app_name = $post->post_title;
            $app_version = get_post_meta($app_id, '_arewa_apk_version', true) ?: '';
            $app_icon = get_post_meta($app_id, '_arewa_apk_icon', true);
            $app_screenshots = get_post_meta($app_id, '_arewa_apk_screenshots', true);
            if (!is_array($app_screenshots)) {
                $app_screenshots = array();
            }
            $apk_url = get_post_meta($app_id, '_arewa_apk_url', true);
            $short_description = get_post_meta($app_id, '_arewa_apk_short_description', true);
            $about_app = get_post_meta($app_id, '_arewa_apk_about', true);
            $disable_seo_app = get_post_meta($app_id, '_arewa_apk_disable_seo', true) ?: 'no';
            
            // Check if SEO is disabled for this specific app
            if ($disable_seo_app === 'yes') {
                return;
            }
        } else {
            // Check if shortcode exists in post content (legacy mode)
            // Only check for shortcode if we're NOT on a single app page
            $has_shortcode = false;
            if (!empty($post->post_content) && has_shortcode($post->post_content, 'arewa_apk_app')) {
                $has_shortcode = true;
            }
            
            // Also check if shortcode is in excerpt (some themes use this)
            if (!$has_shortcode && !empty($post->post_excerpt) && has_shortcode($post->post_excerpt, 'arewa_apk_app')) {
                $has_shortcode = true;
            }
            
            // If shortcode not found, don't add SEO data
            if (!$has_shortcode) {
                return;
            }
            
            // Use legacy settings
            $app_name = $settings['app_name'] ?? '';
            $app_version = $settings['app_version'] ?? '';
            $app_icon = $settings['app_icon'] ?? '';
            $app_screenshots = $settings['app_screenshots'] ?? array();
            $apk_url = $settings['apk_url'] ?? '';
        }
        
        // Skip if no app name
        if (empty($app_name)) {
            return;
        }
        
        // Check if a major SEO plugin is active
        $seo_plugin = $this->is_seo_plugin_active();
        
        // Get rating data for specific app or global
        $average_rating = self::get_average_rating($app_id);
        $rating_count = self::get_rating_count($app_id);
        $download_count = $app_id > 0 ? absint(get_post_meta($app_id, '_arewa_apk_download_count', true) ?: 0) : self::get_download_count();
        
        // Get current page URL
        $page_url = get_permalink();
        $site_name = get_bloginfo('name');
        
        // Get page description (use short description, post excerpt, or content)
        $page_description = '';
        if ($app_id > 0) {
            // For single app pages, prefer short description
            if (!empty($short_description)) {
                $page_description = $short_description;
            } elseif (!empty($post->post_excerpt)) {
                $page_description = wp_strip_all_tags($post->post_excerpt);
            } elseif (!empty($about_app)) {
                $page_description = wp_trim_words(wp_strip_all_tags($about_app), 30);
            } elseif (!empty($post->post_content)) {
                // Remove shortcode from content before creating description
                $content_without_shortcode = preg_replace('/\[arewa_apk_app[^\]]*\]/i', '', $post->post_content);
                $page_description = wp_trim_words(wp_strip_all_tags($content_without_shortcode), 30);
            }
        } else {
            // Legacy mode - use post excerpt or content
            if (!empty($post->post_excerpt)) {
                $page_description = wp_strip_all_tags($post->post_excerpt);
            } elseif (!empty($post->post_content)) {
                // Remove shortcode from content before creating description
                $content_without_shortcode = preg_replace('/\[arewa_apk_app[^\]]*\]/i', '', $post->post_content);
                $page_description = wp_trim_words(wp_strip_all_tags($content_without_shortcode), 30);
            }
        }
        
        if (empty($page_description)) {
            // translators: %1$s: App name, %2$s: App version, %3$s: Download count
            $page_description = sprintf(esc_html__('Download %1$s - Version %2$s. %3$s downloads.', 'arewa-apk-apps-manager'), esc_html($app_name), esc_html($app_version), esc_html(number_format_i18n($download_count)));
        }
        
        // If a major SEO plugin is active, only add SoftwareApplication schema
        // Don't add Open Graph/Twitter tags to avoid conflicts with SEO plugin
        if ($seo_plugin) {
            // Only add Schema.org SoftwareApplication - let SEO plugin handle the rest
            echo "\n<!-- Arewa APK App Schema (SoftwareApplication only - SEO plugin handles other meta tags) -->\n";
        } else {
            // No SEO plugin detected - add all meta tags
            // Get first screenshot for Open Graph (prefer screenshot over icon)
            $og_image = '';
            if (!empty($app_screenshots) && is_array($app_screenshots) && !empty($app_screenshots[0])) {
                $og_image = $app_screenshots[0];
            } elseif (!empty($app_icon)) {
                $og_image = $app_icon;
            }
            
            // Open Graph Meta Tags
            echo "\n<!-- Arewa APK App SEO Meta Tags -->\n";
            echo '<meta property="og:type" content="website" />' . "\n";
            echo '<meta property="og:title" content="' . esc_attr($app_name . ' - ' . $site_name) . '" />' . "\n";
            echo '<meta property="og:description" content="' . esc_attr($page_description) . '" />' . "\n";
            echo '<meta property="og:url" content="' . esc_url($page_url) . '" />' . "\n";
            if (!empty($og_image)) {
                echo '<meta property="og:image" content="' . esc_url($og_image) . '" />' . "\n";
                echo '<meta property="og:image:width" content="1200" />' . "\n";
                echo '<meta property="og:image:height" content="630" />' . "\n";
            }
            echo '<meta property="og:site_name" content="' . esc_attr($site_name) . '" />' . "\n";
            
            // Twitter Card Meta Tags
            echo '<meta name="twitter:card" content="summary_large_image" />' . "\n";
            echo '<meta name="twitter:title" content="' . esc_attr($app_name . ' - ' . $site_name) . '" />' . "\n";
            echo '<meta name="twitter:description" content="' . esc_attr($page_description) . '" />' . "\n";
            if (!empty($og_image)) {
                echo '<meta name="twitter:image" content="' . esc_url($og_image) . '" />' . "\n";
            }
            
            // Additional Meta Tags
            echo '<meta name="application-name" content="' . esc_attr($app_name) . '" />' . "\n";
            if (!empty($app_version)) {
                echo '<meta name="application-version" content="' . esc_attr($app_version) . '" />' . "\n";
            }
            echo '<meta name="mobile-web-app-capable" content="yes" />' . "\n";
        }
        
        // Schema.org Structured Data (JSON-LD) for SoftwareApplication
        $schema = array(
            '@context' => 'https://schema.org',
            '@type' => 'SoftwareApplication',
            'name' => $app_name,
            'applicationCategory' => 'MobileApplication',
            'operatingSystem' => 'Android',
            'offers' => array(
                '@type' => 'Offer',
                'price' => '0',
                'priceCurrency' => 'USD'
            )
        );
        
        // Add version
        if (!empty($app_version)) {
            $schema['softwareVersion'] = $app_version;
        }
        
        // Add description
        if (!empty($page_description)) {
            $schema['description'] = $page_description;
        }
        
        // Add image/icon
        if (!empty($app_icon)) {
            $schema['image'] = $app_icon;
        }
        
        // Add URL
        if (!empty($page_url)) {
            $schema['url'] = $page_url;
        }
        
        // Add download URL
        if (!empty($apk_url)) {
            $schema['downloadUrl'] = $apk_url;
        }
        
        // Add screenshots
        if (!empty($app_screenshots) && is_array($app_screenshots)) {
            $screenshots = array();
            foreach ($app_screenshots as $screenshot) {
                if (!empty($screenshot)) {
                    $screenshots[] = $screenshot;
                }
            }
            if (!empty($screenshots)) {
                $schema['screenshot'] = $screenshots;
            }
        }
        
        // Add aggregate rating if we have ratings (always add if available, even if 0)
        // Google recommends including aggregateRating when available
        if ($rating_count > 0 && $average_rating > 0) {
            $schema['aggregateRating'] = array(
                '@type' => 'AggregateRating',
                'ratingValue' => floatval($average_rating),
                'ratingCount' => intval($rating_count),
                'bestRating' => 5,
                'worstRating' => 1
            );
        } elseif ($rating_count === 0 && $app_id > 0) {
            // For single app pages, if there are no ratings yet, we could add a default
            // But it's better to only add when there are actual ratings
            // So we'll leave it out if there are no ratings
        }
        
        // Add download count if available
        if ($download_count > 0) {
            $schema['downloadCount'] = intval($download_count);
        }
        
        // Output JSON-LD schema (always add SoftwareApplication schema)
        // JSON-LD must be output directly with proper type attribute
        // Note: This function is already called in wp_head hook, so we can echo directly
        // Use wp_json_encode without flags that prevent escaping for security
        $schema_json = wp_json_encode($schema, JSON_PRETTY_PRINT);
        // Output JSON-LD schema - wp_json_encode() already escapes for safe output
        echo '<script type="application/ld+json">' . "\n";
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- wp_json_encode() properly escapes JSON for safe output in script tags. This is the correct and safe way to output JSON-LD schema per WordPress standards.
        echo $schema_json . "\n";
        echo '</script>' . "\n";
        
        if ($seo_plugin) {
            echo "<!-- End Arewa APK App Schema (SoftwareApplication) -->\n\n";
            // Note: Your SEO plugin (" . ucfirst(str_replace('_', ' ', $seo_plugin)) . ") handles Article schema and other meta tags.
            // This plugin only adds SoftwareApplication schema to complement your SEO plugin.
        } else {
            echo "<!-- End Arewa APK App SEO Meta Tags -->\n\n";
        }
    }
    
    /**
     * Verify and restore data integrity after plugin reinstallation
     * This ensures all data is preserved even after uninstall/reinstall
     * Called on plugins_loaded hook to verify data exists
     */
    public function verify_data_integrity() {
        global $wpdb;
        
        // Verify settings exist
        $settings = get_option('arewa_apk_settings', false);
        if (false === $settings) {
            // Settings don't exist, but don't overwrite if they were just deleted
            // This is handled in activate() function
            return;
        }
        
        // Verify download count exists
        $download_count = get_option('arewa_apk_download_count', false);
        if (false === $download_count) {
            // Download count doesn't exist, set to 0 but preserve if it exists
            add_option('arewa_apk_download_count', 0);
        }
        
        // Verify comments table exists and has correct structure
        $table_name = $wpdb->prefix . 'arewa_apk_comments';
        $table_name_escaped = esc_sql($table_name);
        $table_check = $wpdb->get_var("SHOW TABLES LIKE '{$table_name_escaped}'");
        
        if ($table_check !== $table_name) {
            // Table doesn't exist, create it (but this should be handled in activate())
            $this->create_comment_tables();
        }
    }
    
    /**
     * Clear cache when a comment is added/updated
     * Supports popular WordPress caching plugins
     */
    private function clear_cache() {
        // Clear WordPress object cache
        wp_cache_flush();
        
        // WP Super Cache
        if (function_exists('wp_cache_clear_cache')) {
            wp_cache_clear_cache();
        }
        
        // W3 Total Cache
        if (function_exists('w3tc_flush_all')) {
            w3tc_flush_all();
        }
        
        // WP Rocket
        if (function_exists('rocket_clean_domain')) {
            rocket_clean_domain();
        }
        
        // LiteSpeed Cache
        if (class_exists('LiteSpeed_Cache_API')) {
            LiteSpeed_Cache_API::purge_all();
        }
        
        // WP Fastest Cache
        if (isset($GLOBALS['wp_fastest_cache']) && method_exists($GLOBALS['wp_fastest_cache'], 'deleteCache')) {
            $GLOBALS['wp_fastest_cache']->deleteCache(true);
        }
        
        // Cache Enabler
        if (function_exists('ce_clear_cache')) {
            ce_clear_cache();
        }
        
        // Comet Cache
        if (class_exists('comet_cache')) {
            comet_cache::clear();
        }
        
        // Hyper Cache
        if (function_exists('hyper_cache_clean')) {
            hyper_cache_clean();
        }
        
        // WP-Optimize Cache
        if (function_exists('wpo_cache_flush')) {
            wpo_cache_flush();
        }
        
        // Autoptimize Cache
        if (function_exists('autoptimize_cache_clear')) {
            autoptimize_cache_clear();
        }
        
        // Clear transients related to comments
        delete_transient('arewa_apk_comments_count');
        delete_transient('arewa_apk_average_rating');
        
        // Action hook for other plugins to hook into
        do_action('arewa_apk_comment_cache_cleared');
    }
    
    /**
     * Add data to custom columns for Quick Edit
     */
    public function quick_edit_column_data($column, $post_id) {
        // This is called for each column, but we'll handle the data in the Quick Edit box
    }
    
    /**
     * Add custom fields to Quick Edit box
     */
    public function quick_edit_custom_box($column_name, $post_type) {
        if ($post_type !== 'arewa_apk_app') {
            return;
        }
        
        // Only add once per Quick Edit box (on the first column)
        static $printed = false;
        if ($column_name !== 'title') {
            return;
        }
        
        // Only print once per page load
        if ($printed) {
            return;
        }
        $printed = true;
        
        wp_nonce_field('arewa_quick_edit', 'arewa_quick_edit_nonce');
        
        ?>
        <fieldset class="inline-edit-col-right inline-edit-arewa-apk-app">
            <div class="inline-edit-col">
                <div class="inline-edit-group wp-clearfix">
                    <label class="inline-edit-status alignleft">
                        <span class="title"><?php esc_html_e('Status', 'arewa-apk-apps-manager'); ?></span>
                        <select name="_status">
                            <option value="draft"><?php esc_html_e('Draft', 'arewa-apk-apps-manager'); ?></option>
                            <option value="publish"><?php esc_html_e('Published', 'arewa-apk-apps-manager'); ?></option>
                            <option value="future"><?php esc_html_e('Scheduled', 'arewa-apk-apps-manager'); ?></option>
                        </select>
                    </label>
                </div>
                
                <div class="inline-edit-group wp-clearfix">
                    <label class="alignleft">
                        <span class="title"><?php esc_html_e('Published Date', 'arewa-apk-apps-manager'); ?></span>
                        <span class="input-text-wrap">
                            <input type="date" name="post_date" class="post_date" value="" />
                            <input type="time" name="post_time" class="post_time" value="" style="margin-left: 5px;" />
                        </span>
                    </label>
                </div>
            </div>
        </fieldset>
        <?php
        // Reset static for next Quick Edit
        $printNonce = true;
    }
    
    /**
     * Add custom fields to Bulk Edit box
     */
    public function bulk_edit_custom_box($column_name, $post_type) {
        if ($post_type !== 'arewa_apk_app') {
            return;
        }
        
        static $printNonce = true;
        if ($printNonce) {
            $printNonce = false;
            wp_nonce_field('arewa_bulk_edit', 'arewa_bulk_edit_nonce');
        }
        
        ?>
        <fieldset class="inline-edit-col-right inline-edit-arewa-apk-app">
            <div class="inline-edit-col">
                <div class="inline-edit-group wp-clearfix">
                    <label class="inline-edit-status alignleft">
                        <span class="title"><?php esc_html_e('Status', 'arewa-apk-apps-manager'); ?></span>
                        <select name="_status">
                            <option value="-1"><?php esc_html_e('— No Change —', 'arewa-apk-apps-manager'); ?></option>
                            <option value="draft"><?php esc_html_e('Draft', 'arewa-apk-apps-manager'); ?></option>
                            <option value="publish"><?php esc_html_e('Published', 'arewa-apk-apps-manager'); ?></option>
                        </select>
                    </label>
                </div>
            </div>
        </fieldset>
        <?php
    }
    
    /**
     * Save Quick Edit data
     */
    public function arewa_apk_save_quick_edit_data($post_id, $post) {
        // Check if this is an autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return $post_id;
        }
        
        // Check post type
        if ($post->post_type !== 'arewa_apk_app') {
            return $post_id;
        }
        
        // Check permissions
        if (!current_user_can('edit_post', $post_id)) {
            return $post_id;
        }
        
        // Check nonce for Quick Edit
        if (isset($_POST['arewa_quick_edit_nonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['arewa_quick_edit_nonce'])), 'arewa_quick_edit')) {
            $post_data = array('ID' => $post_id);
            
            // Update status if provided
            if (isset($_POST['_status']) && $_POST['_status'] !== '-1') {
                $post_data['post_status'] = sanitize_text_field($_POST['_status']);
            }
            
            // Update date if provided
            if (isset($_POST['post_date']) && !empty($_POST['post_date'])) {
                $post_date = sanitize_text_field($_POST['post_date']);
                $post_time = isset($_POST['post_time']) ? sanitize_text_field($_POST['post_time']) : current_time('H:i:s');
                
                $post_date_string = $post_date . ' ' . $post_time . ':00';
                $post_date_gmt = get_gmt_from_date($post_date_string);
                
                $post_data['post_date'] = $post_date_string;
                $post_data['post_date_gmt'] = $post_date_gmt;
                
                // Get current post to preserve status if not explicitly changed
                $current_post = get_post($post_id);
                $current_status = $current_post ? $current_post->post_status : 'publish';
                
                // Check if date is in the past or present
                $post_timestamp = strtotime($post_date_string);
                $current_timestamp = current_time('timestamp');
                
                // If status was not explicitly changed in Quick Edit
                if (!isset($post_data['post_status'])) {
                    // If date is in past/present, keep current status (or set to publish if it was publish)
                    if ($post_timestamp <= $current_timestamp) {
                        // Date is in past or present - keep as publish if it was published
                        if ($current_status === 'publish' || $current_status === 'future') {
                            $post_data['post_status'] = 'publish';
                        } else {
                            $post_data['post_status'] = $current_status;
                        }
                    } else {
                        // Date is in the future - only change to future if it was already published
                        // Otherwise keep current status
                        if ($current_status === 'publish') {
                            $post_data['post_status'] = 'future';
                        } else {
                            $post_data['post_status'] = $current_status;
                        }
                    }
                } else {
                    // Status was explicitly changed - but if date is past and status is future, change to publish
                    if ($post_timestamp <= $current_timestamp && $post_data['post_status'] === 'future') {
                        $post_data['post_status'] = 'publish';
                    }
                }
            }
            
            // Only update if there are changes
            if (count($post_data) > 1) {
                wp_update_post($post_data);
            }
        }
        
        return $post_id;
    }
    
    /**
     * Save Bulk Edit data via AJAX
     */
    public function arewa_apk_save_bulk_edit_data() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'arewa_bulk_edit')) {
            wp_die(esc_html__('Security check failed.', 'arewa-apk-apps-manager'));
        }
        
        // Check permissions
        if (!current_user_can('edit_posts')) {
            wp_die(esc_html__('You do not have permission to edit posts.', 'arewa-apk-apps-manager'));
        }
        
        $post_ids = isset($_POST['post_ids']) ? array_map('intval', $_POST['post_ids']) : array();
        
        if (empty($post_ids)) {
            wp_send_json_error(__('No posts selected.', 'arewa-apk-apps-manager'));
        }
        
        $status = isset($_POST['_status']) ? sanitize_text_field($_POST['_status']) : '';
        
        foreach ($post_ids as $post_id) {
            if (!current_user_can('edit_post', $post_id)) {
                continue;
            }
            
            $post_data = array('ID' => $post_id);
            
            if ($status !== '' && $status !== '-1') {
                $post_data['post_status'] = $status;
            }
            
            if (count($post_data) > 1) {
                wp_update_post($post_data);
            }
        }
        
        wp_send_json_success(__('Posts updated successfully.', 'arewa-apk-apps-manager'));
    }
    
    /**
     * Prevent WordPress from auto-changing status to 'future' when user wants 'publish'
     */
    public function prevent_auto_future_status($data, $postarr) {
        // Only for our custom post type
        if (!isset($data['post_type']) || $data['post_type'] !== 'arewa_apk_app') {
            return $data;
        }
        
        // If user explicitly set status to 'publish' in the form, keep it as 'publish'
        if (isset($_POST['post_status']) && $_POST['post_status'] === 'publish') {
            // Check if date is in the future
            if (isset($data['post_date']) && !empty($data['post_date'])) {
                $post_timestamp = strtotime($data['post_date']);
                $current_timestamp = current_time('timestamp');
                
                // If date is in future but user wants it published, keep it as publish
                if ($post_timestamp > $current_timestamp) {
                    $data['post_status'] = 'publish';
                }
            } else {
                // Even without date check, if user wants publish, keep it
                $data['post_status'] = 'publish';
            }
        }
        
        return $data;
    }
}

// Register activation and deactivation hooks BEFORE instantiating the class
register_activation_hook(__FILE__, array('Arewa_APK_App', 'activate'));
register_deactivation_hook(__FILE__, array('Arewa_APK_App', 'deactivate'));

// Initialize the plugin
Arewa_APK_App::get_instance();

