/**
 * Arewa APK App - Admin JavaScript
 */

(function($) {
    'use strict';
    
    // Store frame globally to reuse it
    var screenshotsFrame = null;
    
    // Short description character counter
    function initShortDescriptionCounter() {
        var shortDescField = $('#arewa_apk_short_description');
        var shortDescCount = $('#arewa-short-desc-count, #arewa-short-desc-count-meta');
        
        if (shortDescField.length && shortDescCount.length) {
            function updateShortDescCount() {
                var currentLength = shortDescField.val().length;
                var maxLength = 100;
                shortDescCount.text('(' + currentLength + '/' + maxLength + ')');
                
                if (currentLength > maxLength) {
                    shortDescCount.css('color', '#d63638');
                    // Truncate to max length
                    shortDescField.val(shortDescField.val().substring(0, maxLength));
                    updateShortDescCount();
                } else if (currentLength >= maxLength - 10) {
                    shortDescCount.css('color', '#d63638');
                } else {
                    shortDescCount.css('color', '#0073aa');
                }
            }
            
            // Initialize count
            updateShortDescCount();
            
            // Update on input
            shortDescField.on('input', function() {
                updateShortDescCount();
            });
            
            // Prevent exceeding max length
            shortDescField.on('keypress', function(e) {
                if ($(this).val().length >= 100 && e.which !== 8 && e.which !== 0 && e.which !== 46) {
                    e.preventDefault();
                    return false;
                }
            });
        }
    }
    
    // Function to initialize media uploader
    function initArewaMediaUploader() {
        // Check if wp.media is available
        if (typeof wp === 'undefined' || typeof wp.media === 'undefined') {
            console.log('Arewa APK App: Waiting for wp.media to be available...');
            // Retry after a short delay
            setTimeout(initArewaMediaUploader, 100);
            return;
        }
        
        console.log('Arewa APK App: Initializing media uploader...');
        
        // Media Uploader for Image Fields (single)
        $(document).off('click', '.arewa-upload-button').on('click', '.arewa-upload-button', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            console.log('Arewa APK App: Single image upload button clicked');
            
            var button = $(this);
            var targetField = button.data('target');
            var previewId = targetField + '_preview';
            var preview = $('#' + previewId);
            
            // Create media frame
            var frame = wp.media({
                title: 'Select Image',
                button: {
                    text: 'Use this image'
                },
                multiple: false
            });
            
            // Handle selection
            frame.on('select', function() {
                var attachment = frame.state().get('selection').first().toJSON();
                var imageUrl = attachment.url;
                
                console.log('Arewa APK App: Image selected:', imageUrl);
                
                // Update input field
                $('#' + targetField).val(imageUrl);
                
                // Update preview
                if (preview.length) {
                    preview.html('<img src="' + imageUrl + '" alt="Preview" style="max-width: 300px; height: auto; border: 1px solid #ddd; padding: 5px; border-radius: 4px;" />');
                }
                
                // Show remove button
                $('.arewa-remove-button[data-target="' + targetField + '"]').show();
            });
            
            // Open frame
            frame.open();
        });
        
        // Media Uploader for Multiple Screenshots
        
        $(document).off('click', '.arewa-upload-multiple-button').on('click', '.arewa-upload-multiple-button', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            console.log('Arewa APK App: Multiple screenshots button clicked');
            
            var button = $(this);
            var targetId = button.data('target');
            // Try to find screenshots list by data-target or default IDs
            var screenshotsList = $('#' + targetId + '_list');
            if (!screenshotsList.length) {
                // Fallback to default IDs
                screenshotsList = $('#arewa_apk_screenshots_list, #app_screenshots_list');
            }
            
            // Check if screenshotsList exists
            if (!screenshotsList.length) {
                console.error('Arewa APK App: Screenshots list container not found. Target:', targetId);
                console.error('Arewa APK App: Available containers:', $('[id*="screenshot"]').map(function() { return this.id; }).get());
                alert('Error: Screenshots container not found. Please refresh the page.');
                return;
            }
            
            // Check if wp.media is available
            if (typeof wp === 'undefined' || typeof wp.media === 'undefined') {
                console.error('Arewa APK App: wp.media is not available');
                alert('Error: Media library is not available. Please refresh the page.');
                return;
            }
            
            console.log('Arewa APK App: Opening media frame for multiple selection');
            
            try {
                // If frame already exists, reuse it
                if (screenshotsFrame) {
                    console.log('Arewa APK App: Reusing existing frame');
                    screenshotsFrame.open();
                    return;
                }
                
                console.log('Arewa APK App: Creating new media frame');
                
                // Create media frame with multiple selection
                screenshotsFrame = wp.media({
                    title: 'Select Screenshots',
                    button: {
                        text: 'Use selected images'
                    },
                    multiple: true,
                    library: {
                        type: 'image'
                    }
                });
                
                // Handle selection
                screenshotsFrame.on('select', function() {
                    try {
                        var attachments = screenshotsFrame.state().get('selection').toJSON();
                        
                        console.log('Arewa APK App: Selected', attachments.length, 'images');
                        
                        if (attachments.length === 0) {
                            console.log('Arewa APK App: No images selected');
                            return;
                        }
                        
                        attachments.forEach(function(attachment) {
                            var imageUrl = attachment.url;
                            var index = screenshotsList.find('.arewa-screenshot-item').length;
                            
                            // Check if this image is already in the list
                            var existing = screenshotsList.find('input[value="' + imageUrl + '"]');
                            if (existing.length > 0) {
                                console.log('Arewa APK App: Image already exists, skipping:', imageUrl);
                                return; // Skip if already exists
                            }
                            
                            // Determine the correct input name based on context
                            var inputName = 'arewa_apk_screenshots[]';
                            // Check if we're in legacy settings page
                            if ($('#app_screenshots_list').length > 0 && screenshotsList.attr('id') === 'app_screenshots_list') {
                                inputName = 'app_screenshots[]';
                            }
                            
                            // Create new screenshot item
                            var itemHtml = '<div class="arewa-screenshot-item" data-index="' + index + '" style="display: inline-block; margin: 5px; position: relative;">' +
                                '<input type="hidden" name="' + inputName + '" value="' + imageUrl + '" />' +
                                '<img src="' + imageUrl + '" alt="Screenshot ' + (index + 1) + '" style="max-width: 150px; height: auto; border: 1px solid #ddd; padding: 5px; border-radius: 4px; display: block;" />' +
                                '<button type="button" class="button arewa-remove-screenshot" style="margin-top: 5px; width: 100%;">Remove</button>' +
                                '</div>';
                            
                            screenshotsList.append(itemHtml);
                            console.log('Arewa APK App: Added screenshot:', imageUrl);
                        });
                    } catch (error) {
                        console.error('Arewa APK App: Error handling selection:', error);
                    }
                });
                
                // Open frame
                console.log('Arewa APK App: Opening media frame...');
                screenshotsFrame.open();
                console.log('Arewa APK App: Media frame opened successfully');
            } catch (error) {
                console.error('Arewa APK App: Error creating/opening media frame:', error);
                alert('Error opening media library: ' + error.message);
            }
        });
        
        // Remove Screenshot
        $(document).off('click', '.arewa-remove-screenshot').on('click', '.arewa-remove-screenshot', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            console.log('Arewa APK App: Remove screenshot clicked');
            
            var item = $(this).closest('.arewa-screenshot-item');
            item.fadeOut(300, function() {
                $(this).remove();
                console.log('Arewa APK App: Screenshot removed');
            });
        });
        
        // Remove Image (for single image fields)
        $(document).off('click', '.arewa-remove-button').on('click', '.arewa-remove-button', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            var button = $(this);
            var targetField = button.data('target');
            var previewId = targetField + '_preview';
            
            // Clear input
            $('#' + targetField).val('');
            
            // Clear preview
            $('#' + previewId).empty();
            
            // Hide remove button
            button.hide();
        });
        
        // Update preview when URL is manually entered (support both legacy and CPT fields)
        $(document).off('change blur', '#app_icon, #app_screenshot, #arewa_apk_icon').on('change blur', '#app_icon, #app_screenshot, #arewa_apk_icon', function() {
            var fieldId = $(this).attr('id');
            var imageUrl = $(this).val();
            var previewId = fieldId + '_preview';
            var preview = $('#' + previewId);
            var removeButton = $('.arewa-remove-button[data-target="' + fieldId + '"]');
            
            if (imageUrl) {
                if (preview.length) {
                    preview.html('<img src="' + imageUrl + '" alt="Preview" style="max-width: 300px; height: auto; border: 1px solid #ddd; padding: 5px; border-radius: 4px;" />');
                }
                removeButton.show();
            } else {
                preview.empty();
                removeButton.hide();
            }
        });
        
        console.log('Arewa APK App: Media uploader initialized successfully');
    }
    
    // Admin Reply Functions
    function initAdminReply() {
        // Toggle reply form
        $(document).on('click', '.arewa-reply-btn', function(e) {
            e.preventDefault();
            var commentId = $(this).data('comment-id');
            var replyRow = $('#arewa-reply-row-' + commentId);
            
            if (replyRow.is(':visible')) {
                replyRow.slideUp();
            } else {
                replyRow.slideDown();
            }
        });
        
        // Cancel reply
        $(document).on('click', '.arewa-cancel-reply-btn', function(e) {
            e.preventDefault();
            var commentId = $(this).data('comment-id');
            $('#arewa-reply-row-' + commentId).slideUp();
        });
        
        // Select image for reply
        $(document).on('click', '.arewa-select-image-btn', function(e) {
            e.preventDefault();
            var commentId = $(this).data('comment-id');
            var button = $(this);
            var fileInput = $('#reply_image_' + commentId);
            var form = $('#arewa-reply-row-' + commentId).find('.arewa-admin-reply-form');
            var previewContainer = $('#arewa-reply-row-' + commentId).find('.arewa-reply-image-preview');
            var previewImg = previewContainer.find('img');
            
            // Check if wp.media is available
            if (typeof wp === 'undefined' || typeof wp.media === 'undefined') {
                alert('Media library is not available. Please use file upload instead.');
                fileInput.click();
                return;
            }
            
            // Create media frame
            var frame = wp.media({
                title: 'Select Reply Image',
                button: {
                    text: 'Use this image'
                },
                multiple: false,
                library: {
                    type: 'image'
                }
            });
            
            // Handle selection
            frame.on('select', function() {
                var attachment = frame.state().get('selection').first().toJSON();
                var imageUrl = attachment.url;
                
                // Update preview
                previewImg.attr('src', imageUrl);
                previewContainer.show();
                
                // Store the URL in a hidden input field so we can send it to the server
                var hiddenInput = form.find('input[name="reply_image_url"]');
                if (!hiddenInput.length) {
                    // Create hidden input if it doesn't exist
                    form.append('<input type="hidden" name="reply_image_url" value="" />');
                    hiddenInput = form.find('input[name="reply_image_url"]');
                }
                hiddenInput.val(imageUrl);
                
                // Clear file input since we're using URL from media library
                fileInput.val('');
                
                console.log('Image selected from media library:', imageUrl);
            });
            
            // Open frame
            frame.open();
        });
        
        // File input change (for direct file upload)
        $(document).on('change', 'input[name="reply_image"]', function(e) {
            var file = this.files[0];
            if (!file) return;
            
            var commentId = $(this).attr('id').replace('reply_image_', '');
            var form = $('#arewa-reply-row-' + commentId).find('.arewa-admin-reply-form');
            var previewContainer = $('#arewa-reply-row-' + commentId).find('.arewa-reply-image-preview');
            var previewImg = previewContainer.find('img');
            var hiddenInput = form.find('input[name="reply_image_url"]');
            
            // Validate file type
            if (!file.type.match('image.*')) {
                alert('Please select an image file.');
                $(this).val('');
                return;
            }
            
            // Validate file size (max 2MB)
            if (file.size > 2 * 1024 * 1024) {
                alert('File size exceeds 2MB limit.');
                $(this).val('');
                return;
            }
            
            // Clear hidden URL input since we're using file upload
            if (hiddenInput.length) {
                hiddenInput.val('');
            }
            
            // Show preview
            var reader = new FileReader();
            reader.onload = function(e) {
                previewImg.attr('src', e.target.result);
                previewContainer.show();
            };
            reader.readAsDataURL(file);
        });
        
        // Remove image
        $(document).on('click', '.arewa-remove-image-btn', function(e) {
            e.preventDefault();
            var commentId = $(this).data('comment-id');
            var form = $('#arewa-reply-row-' + commentId).find('.arewa-admin-reply-form');
            var fileInput = $('#reply_image_' + commentId);
            var previewContainer = $('#arewa-reply-row-' + commentId).find('.arewa-reply-image-preview');
            var previewImg = previewContainer.find('img');
            var hiddenInput = form.find('input[name="reply_image_url"]');
            
            fileInput.val('');
            if (hiddenInput.length) {
                hiddenInput.val('');
            }
            previewImg.attr('src', '');
            previewContainer.hide();
        });
        
        // Submit reply form - handle both form submit and button click
        $(document).on('submit', '.arewa-admin-reply-form', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            console.log('Reply form submitted');
            handleReplySubmit($(this));
        });
        
        // Also handle button click as backup (in case form submit doesn't fire)
        $(document).on('click', '.arewa-save-reply-btn', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            var btn = $(this);
            var form = btn.closest('.arewa-admin-reply-form');
            
            if (form.length === 0) {
                console.error('Form not found for save button');
                return;
            }
            
            console.log('Save Reply button clicked');
            // Trigger form submit (which will be caught by the submit handler)
            form.trigger('submit');
        });
        
        // Function to handle reply submission
        function handleReplySubmit(form) {
            var commentId = form.data('comment-id');
            var submitBtn = form.find('.arewa-save-reply-btn');
            var messageDiv = form.find('.arewa-reply-message');
            
            // Check if arewaAPKAdmin is available
            if (typeof arewaAPKAdmin === 'undefined') {
                console.error('arewaAPKAdmin is not defined!');
                console.error('Available globals:', Object.keys(window).filter(k => k.indexOf('arewa') >= 0));
                messageDiv.removeClass('success').addClass('error').text('Configuration error. Please refresh the page.').show();
                return;
            }
            
            console.log('arewaAPKAdmin:', arewaAPKAdmin);
            console.log('Comment ID:', commentId);
            
            // Get form data - safely handle undefined/null values
            var replyTextElement = form.find('textarea[name="reply_text"]');
            var replyText = replyTextElement.length > 0 ? (replyTextElement.val() || '').trim() : '';
            
            var replyUrlElement = form.find('input[name="reply_url"]');
            var replyUrl = replyUrlElement.length > 0 ? (replyUrlElement.val() || '').trim() : '';
            
            var replyImage = form.find('input[name="reply_image"]')[0];
            
            var replyImageUrlElement = form.find('input[name="reply_image_url"]');
            var replyImageUrl = replyImageUrlElement.length > 0 ? (replyImageUrlElement.val() || '').trim() : '';
            
            console.log('Form data:', {
                replyText: replyText,
                replyUrl: replyUrl,
                hasImageFile: replyImage && replyImage.files && replyImage.files.length > 0,
                replyImageUrl: replyImageUrl
            });
            
            // Validate - at least one field must be provided
            if (!replyText && !replyUrl && (!replyImage || !replyImage.files || !replyImage.files[0]) && !replyImageUrl) {
                messageDiv.removeClass('success').addClass('error').text('Please provide at least one field (text, URL, or image).').show();
                return;
            }
            
            // Prepare form data
            var formData = new FormData();
            formData.append('action', 'arewapap_admin_reply');
            formData.append('nonce', arewaAPKAdmin.nonce);
            formData.append('comment_id', commentId);
            formData.append('reply_text', replyText);
            formData.append('reply_url', replyUrl);
            
            // Add image URL if selected from media library
            if (replyImageUrl) {
                formData.append('reply_image_url', replyImageUrl);
                console.log('Adding image URL:', replyImageUrl);
            }
            
            // Add image file if selected (takes priority over URL)
            if (replyImage && replyImage.files && replyImage.files.length > 0) {
                formData.append('reply_image', replyImage.files[0]);
                console.log('Adding image file:', replyImage.files[0].name);
            }
            
            // Disable submit button
            submitBtn.prop('disabled', true).text('Saving...');
            messageDiv.hide();
            
            console.log('Submitting AJAX request to:', arewaAPKAdmin.ajaxurl);
            
            // Submit via AJAX
            $.ajax({
                url: arewaAPKAdmin.ajaxurl,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                dataType: 'json',
                timeout: 30000, // 30 second timeout
                success: function(response) {
                    console.log('Reply AJAX Success Response:', response);
                    
                    if (response && response.success) {
                        var successMsg = response.data && response.data.message ? response.data.message : (arewaAPKAdmin.strings && arewaAPKAdmin.strings.replySaved ? arewaAPKAdmin.strings.replySaved : 'Reply saved successfully!');
                        messageDiv.removeClass('error').addClass('success').text(successMsg).show();
                        
                        // Reload page after a short delay to show updated reply
                        setTimeout(function() {
                            location.reload();
                        }, 1500);
                    } else {
                        var errorMsg = (response && response.data && response.data.message) ? response.data.message : (arewaAPKAdmin.strings && arewaAPKAdmin.strings.error ? arewaAPKAdmin.strings.error : 'An error occurred. Please try again.');
                        console.error('Reply AJAX Error Response:', response);
                        messageDiv.removeClass('success').addClass('error').text(errorMsg).show();
                        submitBtn.prop('disabled', false).text('Save Reply');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX Error:', status, error);
                    console.error('Status Code:', xhr.status);
                    console.error('Response Text:', xhr.responseText);
                    console.error('Response Headers:', xhr.getAllResponseHeaders());
                    
                    var errorMsg = arewaAPKAdmin.strings.error || 'An error occurred. Please try again.';
                    
                    // Try to parse error response
                    try {
                        if (xhr.responseText) {
                            var response = JSON.parse(xhr.responseText);
                            if (response && response.data && response.data.message) {
                                errorMsg = response.data.message;
                            } else if (response && response.message) {
                                errorMsg = response.message;
                            }
                        }
                    } catch (e) {
                        console.error('Error parsing response:', e);
                        // If response is not JSON, use status-based messages
                        if (xhr.status === 0) {
                            errorMsg = 'Network error. Please check your connection and try again.';
                        } else if (xhr.status === 403) {
                            errorMsg = 'Security check failed. Please refresh the page and try again.';
                        } else if (xhr.status === 500) {
                            errorMsg = 'Server error. Please try again later or contact support.';
                        } else if (xhr.status === 404) {
                            errorMsg = 'Page not found. Please refresh the page and try again.';
                        } else if (xhr.status === 400) {
                            errorMsg = 'Invalid request. Please check your input and try again.';
                        } else if (status === 'timeout') {
                            errorMsg = 'Request timeout. Please try again.';
                        } else if (status === 'parsererror') {
                            errorMsg = 'Server response error. Please try again.';
                        }
                    }
                    
                    messageDiv.removeClass('success').addClass('error').text(errorMsg).show();
                    submitBtn.prop('disabled', false).text('Save Reply');
                }
            });
        } // End handleReplySubmit function
        
        // Delete reply
        $(document).on('click', '.arewa-delete-reply-btn', function(e) {
            e.preventDefault();
            
            if (!confirm('Are you sure you want to delete this reply?')) {
                return;
            }
            
            var button = $(this);
            var commentId = button.data('comment-id');
            
            button.prop('disabled', true).text('Deleting...');
            
            // Submit via AJAX
            $.ajax({
                url: arewaAPKAdmin.ajaxurl,
                type: 'POST',
                data: {
                    action: 'arewapap_delete_admin_reply',
                    nonce: arewaAPKAdmin.nonce,
                    comment_id: commentId
                },
                dataType: 'json',
                success: function(response) {
                    if (response && response.success) {
                        // Reload page to show updated state
                        location.reload();
                    } else {
                        var errorMsg = (response && response.data && response.data.message) ? response.data.message : arewaAPKAdmin.strings.error;
                        alert(errorMsg);
                        button.prop('disabled', false).text('Delete Reply');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX Error:', status, error);
                    alert(arewaAPKAdmin.strings.error);
                    button.prop('disabled', false).text('Delete Reply');
                }
            });
        });
    }
    
    // Quick Edit functionality for apps list
    function initQuickEdit() {
        // Populate Quick Edit fields when opened
        var $inlineEditor = $('.inline-edit-row');
        
        // Use WordPress's built-in quick edit event
        $(document).on('click', '.editinline', function() {
            var $row = $(this).closest('tr');
            var postId = $row.attr('id').replace('post-', '');
            var $postRow = $row.prev('tr');
            
            // Wait for inline editor to be visible
            setTimeout(function() {
                // Get post status
                var statusText = $postRow.find('.post-state').text().trim();
                var status = 'publish';
                if (statusText.toLowerCase().indexOf('draft') !== -1) {
                    status = 'draft';
                }
                $('select[name="_status"]', '.inline-edit-row').val(status);
                
                // Get post date from the row
                var dateColumn = $postRow.find('.date');
                var dateText = dateColumn.text().trim();
                
                // Try to extract date - WordPress format is usually "YYYY/MM/DD at HH:MM"
                var dateMatch = dateText.match(/(\d{4})\/(\d{1,2})\/(\d{1,2})/);
                var timeMatch = dateText.match(/(\d{1,2}):(\d{2})/);
                
                if (dateMatch) {
                    var year = dateMatch[1];
                    var month = ('0' + dateMatch[2]).slice(-2);
                    var day = ('0' + dateMatch[3]).slice(-2);
                    var dateStr = year + '-' + month + '-' + day;
                    $('input[name="post_date"]', '.inline-edit-row').val(dateStr);
                }
                
                if (timeMatch) {
                    var hours = ('0' + timeMatch[1]).slice(-2);
                    var minutes = ('0' + timeMatch[2]).slice(-2);
                    var timeStr = hours + ':' + minutes;
                    $('input[name="post_time"]', '.inline-edit-row').val(timeStr);
                }
            }, 100);
        });
    }
    
    // Initialize when document is ready
    $(document).ready(function() {
        console.log('Arewa APK App: Document ready, initializing...');
        initArewaMediaUploader();
        initAdminReply();
        initShortDescriptionCounter();
        initQuickEdit();
    });
    
})(jQuery);
