(function($) {
    'use strict';
    
    function initCarousel() {
        $('.arewa-apk-carousel-container').each(function() {
            var $container = $(this);
            var $track = $container.find('.arewa-apk-carousel-track');
            var $slides = $container.find('.arewa-apk-carousel-slides');
            var $prevBtn = $container.find('.arewa-apk-carousel-prev');
            var $nextBtn = $container.find('.arewa-apk-carousel-next');
            var $slideItems = $container.find('.arewa-apk-carousel-slide');
            var currentIndex = 0;
            var totalSlides = $slideItems.length;
            
            if (totalSlides === 0) {
                return; // No slides, exit
            }
            
            // Get slides to show based on screen size
            function getSlidesToShow() {
                var windowWidth = $(window).width();
                if (windowWidth <= 768) {
                    return 1;
                } else if (windowWidth <= 1024) {
                    return 2;
                } else {
                    return 3;
                }
            }
            
            // Get slide width
            function getSlideWidth() {
                if ($slideItems.length === 0) return 0;
                var $firstSlide = $slideItems.first();
                return $firstSlide.outerWidth(true);
            }
            
            // Update carousel position with smooth animation
            function updateCarousel() {
                var slidesToShow = getSlidesToShow();
                var slideWidth = getSlideWidth();
                
                if (slideWidth === 0) {
                    // Slides not ready yet, try again
                    setTimeout(updateCarousel, 100);
                    return;
                }
                
                var maxIndex = Math.max(0, totalSlides - slidesToShow);
                
                // Clamp index
                if (currentIndex < 0) {
                    currentIndex = 0;
                }
                if (currentIndex > maxIndex) {
                    currentIndex = maxIndex;
                }
                
                var translateX = currentIndex * slideWidth;
                $slides.css('transform', 'translateX(-' + translateX + 'px)');
                
                // Update button states
                if (currentIndex <= 0) {
                    $prevBtn.css({
                        'opacity': '0',
                        'pointer-events': 'none',
                        'cursor': 'default'
                    });
                } else {
                    $prevBtn.css({
                        'opacity': '0.9',
                        'pointer-events': 'auto',
                        'cursor': 'pointer'
                    });
                }
                
                if (currentIndex >= maxIndex || maxIndex === 0 || totalSlides <= slidesToShow) {
                    $nextBtn.css({
                        'opacity': '0',
                        'pointer-events': 'none',
                        'cursor': 'default'
                    });
                } else {
                    $nextBtn.css({
                        'opacity': '0.9',
                        'pointer-events': 'auto',
                        'cursor': 'pointer'
                    });
                }
            }
            
            // Previous button handler
            function handlePrevClick(e) {
                if (e) {
                    e.preventDefault();
                    e.stopPropagation();
                    e.stopImmediatePropagation();
                }
                
                var slidesToShow = getSlidesToShow();
                var maxIndex = Math.max(0, totalSlides - slidesToShow);
                
                if (currentIndex > 0) {
                    currentIndex--;
                    updateCarousel();
                }
                
                return false;
            }
            
            // Next button handler
            function handleNextClick(e) {
                if (e) {
                    e.preventDefault();
                    e.stopPropagation();
                    e.stopImmediatePropagation();
                }
                
                var slidesToShow = getSlidesToShow();
                var maxIndex = Math.max(0, totalSlides - slidesToShow);
                
                if (currentIndex < maxIndex && totalSlides > slidesToShow) {
                    currentIndex++;
                    updateCarousel();
                }
                
                return false;
            }
            
            // Attach event handlers - use both jQuery and direct event listeners
            $prevBtn.off('click').on('click', handlePrevClick);
            $nextBtn.off('click').on('click', handleNextClick);
            
            // Also attach direct event listeners as backup
            $prevBtn[0] && $prevBtn[0].addEventListener('click', handlePrevClick, true);
            $nextBtn[0] && $nextBtn[0].addEventListener('click', handleNextClick, true);
            
            // Touch/swipe support for mobile
            var touchStartX = 0;
            var touchEndX = 0;
            var isDragging = false;
            
            $track.on('touchstart', function(e) {
                touchStartX = e.originalEvent.touches[0].clientX;
                isDragging = true;
            });
            
            $track.on('touchmove', function(e) {
                if (isDragging) {
                    e.preventDefault();
                }
            });
            
            $track.on('touchend', function(e) {
                if (!isDragging) return;
                touchEndX = e.originalEvent.changedTouches[0].clientX;
                handleSwipe();
                isDragging = false;
            });
            
            function handleSwipe() {
                var swipeThreshold = 50;
                var diff = touchStartX - touchEndX;
                
                if (Math.abs(diff) > swipeThreshold) {
                    if (diff > 0) {
                        // Swipe left - next
                        handleNextClick();
                    } else {
                        // Swipe right - previous
                        handlePrevClick();
                    }
                }
            }
            
            // Initialize
            updateCarousel();
            
            // Handle resize with debounce
            var resizeTimer;
            $(window).on('resize', function() {
                clearTimeout(resizeTimer);
                resizeTimer = setTimeout(function() {
                    currentIndex = 0;
                    updateCarousel();
                }, 250);
            });
            
            // Re-initialize after a short delay to ensure DOM is ready
            setTimeout(function() {
                updateCarousel();
            }, 100);
        });
    }
    
    // Initialize when document is ready
    $(document).ready(function() {
        initCarousel();
    });
    
    // Also initialize after a delay to catch dynamically loaded content
    setTimeout(function() {
        initCarousel();
    }, 500);
    
})(jQuery);
