/**
 * Arewa APK App - Frontend JavaScript
 */

(function($) {
    'use strict';
    
    $(document).ready(function() {
        
        // Star Rating Interaction
        $('.arewa-rating-input .star-label').on('click', function() {
            var rating = $(this).data('rating');
            if (rating === 0) {
                // Clear rating button clicked
                $('.arewa-rating-input input[type="radio"]').prop('checked', false);
                $('#rating-0').prop('checked', true);
                // Update visual state - all stars gray
                $('.arewa-rating-input .star-label').each(function() {
                    var labelRating = $(this).data('rating');
                    if (labelRating > 0) {
                        $(this).find('.arewa-star-input').css('color', '#ddd');
                    }
                });
                return;
            }
            
            var input = $('#rating-' + rating);
            var currentRating = $('.arewa-rating-input input[type="radio"]:checked').val();
            
            // If clicking the same rating that's already selected, clear it
            if (currentRating == rating && rating > 0) {
                // Clear rating - uncheck all and set to 0
                $('.arewa-rating-input input[type="radio"]').prop('checked', false);
                $('#rating-0').prop('checked', true);
                // Update visual state - all stars gray
                $('.arewa-rating-input .star-label').each(function() {
                    var labelRating = $(this).data('rating');
                    if (labelRating > 0) {
                        $(this).find('.arewa-star-input').css('color', '#ddd');
                    }
                });
                return;
            }
            
            // Set the radio button
            input.prop('checked', true);
            $('#rating-0').prop('checked', false);
            
            // Update visual state
            $('.arewa-rating-input .star-label').each(function() {
                var labelRating = $(this).data('rating');
                if (labelRating > 0) {
                    if (labelRating <= rating) {
                        $(this).find('.arewa-star-input').css('color', '#00ff45');
                    } else {
                        $(this).find('.arewa-star-input').css('color', '#ddd');
                    }
                }
            });
        });
        
        // Initialize default rating (5 stars) on page load
        function initializeDefaultRating() {
            var defaultRating = 5;
            $('#rating-' + defaultRating).prop('checked', true);
            $('.arewa-rating-input .star-label').each(function() {
                var labelRating = $(this).data('rating');
                if (labelRating > 0) {
                    if (labelRating <= defaultRating) {
                        $(this).find('.arewa-star-input').css('color', '#00ff45');
                    } else {
                        $(this).find('.arewa-star-input').css('color', '#ddd');
                    }
                }
            });
        }
        
        // Initialize default rating
        initializeDefaultRating();
        
        // Clear rating button handler
        $('.arewa-clear-rating').on('click', function(e) {
            e.preventDefault();
            // Uncheck all ratings
            $('.arewa-rating-input input[type="radio"]').prop('checked', false);
            $('#rating-0').prop('checked', true);
            // Update visual state - all stars gray
            $('.arewa-rating-input .star-label').each(function() {
                var labelRating = $(this).data('rating');
                if (labelRating > 0) {
                    $(this).find('.arewa-star-input').css('color', '#ddd');
                }
            });
        });
        
        // Hover effect for stars
        $('.arewa-rating-input .star-label').on('mouseenter', function() {
            var rating = $(this).data('rating');
            if (rating > 0) {
                $('.arewa-rating-input .star-label').each(function() {
                    var labelRating = $(this).data('rating');
                    if (labelRating > 0 && labelRating <= rating) {
                        $(this).find('.arewa-star-input').css('color', '#00ff45');
                    }
                });
            }
        });
        
        $('.arewa-rating-input').on('mouseleave', function() {
            var checkedRating = $('.arewa-rating-input input[type="radio"]:checked').val();
            if (checkedRating && checkedRating > 0) {
                $('.arewa-rating-input .star-label').each(function() {
                    var labelRating = $(this).data('rating');
                    if (labelRating > 0) {
                        if (labelRating <= checkedRating) {
                            $(this).find('.arewa-star-input').css('color', '#00ff45');
                        } else {
                            $(this).find('.arewa-star-input').css('color', '#ddd');
                        }
                    }
                });
            } else {
                $('.arewa-rating-input .star-label').each(function() {
                    var labelRating = $(this).data('rating');
                    if (labelRating > 0) {
                        $(this).find('.arewa-star-input').css('color', '#ddd');
                    }
                });
            }
        });
        
        // Initialize avatar upload functionality when document is ready
        function initAvatarUpload() {
            var fileInput = $('#arewa-comment-avatar');
            var uploadBtn = $('#arewa-avatar-upload-btn');
            var previewContainer = $('#arewa-avatar-preview');
            var previewImg = $('#arewa-avatar-preview-img');
            var removeBtn = $('#arewa-avatar-remove');
            
            // Check if elements exist
            if (!fileInput.length || !uploadBtn.length) {
                console.log('Avatar upload elements not found');
                return;
            }
            
            // Initially hide preview and show upload button
            if (previewContainer.length) {
                previewContainer.hide().removeClass('show active');
            }
            if (uploadBtn.length) {
                uploadBtn.show().css({
                    'display': 'inline-flex',
                    'visibility': 'visible',
                    'opacity': '1'
                });
            }
            
            // Handle file input change
            fileInput.on('change', function(e) {
                e.stopPropagation();
                
                var input = this;
                var file = input.files && input.files[0];
                
                if (!file) {
                    console.log('No file selected');
                    return;
                }
                
                console.log('File selected:', file.name, file.type, file.size);
                
                // Validate file type
                var allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
                if (!allowedTypes.includes(file.type)) {
                    alert(arewaAPK.strings.invalidFileType || 'Please select a valid image file (JPEG, PNG, GIF, or WebP).');
                    $(input).val('');
                    return;
                }
                
                // Validate file size
                var maxSize = arewaAPK.maxUploadSize || (2 * 1024 * 1024);
                if (file.size > maxSize) {
                    var maxSizeMB = (maxSize / (1024 * 1024)).toFixed(1);
                    alert(arewaAPK.strings.fileTooLarge || 'File size exceeds ' + maxSizeMB + 'MB limit. Please select a smaller image.');
                    $(input).val('');
                    return;
                }
                
                // Read file and show preview
                var reader = new FileReader();
                
                reader.onload = function(e) {
                    var imageUrl = e.target.result;
                    console.log('File loaded, showing preview');
                    
                    // Update preview image
                    if (previewImg.length) {
                        previewImg.attr('src', imageUrl);
                    }
                    
                    // Show preview container
                    if (previewContainer.length) {
                        previewContainer.css({
                            'display': 'block',
                            'visibility': 'visible',
                            'opacity': '1'
                        }).addClass('show active');
                    }
                    
                    // Hide upload button
                    if (uploadBtn.length) {
                        uploadBtn.css({
                            'display': 'none',
                            'visibility': 'hidden',
                            'opacity': '0'
                        }).hide();
                    }
                    
                    console.log('Preview shown successfully');
                };
                
                reader.onerror = function(e) {
                    console.error('Error reading file:', e);
                    alert('Error reading file. Please try again.');
                    $(input).val('');
                };
                
                reader.readAsDataURL(file);
            });
        }
        
        // Initialize avatar upload
        initAvatarUpload();
        
        // Remove avatar preview (use event delegation)
        $(document).on('click', '#arewa-avatar-remove', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            console.log('Removing avatar preview');
            
            var fileInput = $('#arewa-comment-avatar');
            var previewContainer = $('#arewa-avatar-preview');
            var previewImg = $('#arewa-avatar-preview-img');
            var uploadBtn = $('#arewa-avatar-upload-btn');
            
            // Clear file input
            if (fileInput.length) {
                fileInput.val('');
            }
            
            // Hide preview
            if (previewContainer.length) {
                previewContainer.css({
                    'display': 'none',
                    'visibility': 'hidden',
                    'opacity': '0'
                }).removeClass('show active').hide();
            }
            
            // Clear preview image src
            if (previewImg.length) {
                previewImg.attr('src', '');
            }
            
            // Show upload button
            if (uploadBtn.length) {
                uploadBtn.css({
                    'display': 'inline-flex',
                    'visibility': 'visible',
                    'opacity': '1'
                }).show();
            }
            
            console.log('Avatar preview removed');
        });
        
        // Handle click on upload button to trigger file input
        $(document).on('click', '#arewa-avatar-upload-btn', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            console.log('Upload button clicked');
            
            // Trigger click on file input
            var fileInput = $('#arewa-comment-avatar');
            if (fileInput.length && fileInput[0]) {
                fileInput[0].click();
            } else {
                console.error('File input not found');
            }
        });
        
        // Comment Form Submission
        $('#arewa-apk-comment-form').on('submit', function(e) {
            e.preventDefault();
            
            var form = $(this);
            var submitBtn = form.find('.arewa-submit-btn');
            var messageDiv = $('#arewa-comment-message');
            
            // Validate form
            var authorName = $('#arewa-comment-author').val().trim();
            var authorEmail = $('#arewa-comment-email').val().trim();
            var commentText = $('#arewa-comment-text').val().trim();
            
            if (!authorName || !authorEmail || !commentText) {
                messageDiv.removeClass('success').addClass('error').text(arewaAPK.strings.required).show();
                return;
            }
            
            if (!isValidEmail(authorEmail)) {
                messageDiv.removeClass('success').addClass('error').text(arewaAPK.strings.invalidEmail || 'Invalid email address.').show();
                return;
            }
            
            // Disable submit button
            submitBtn.prop('disabled', true).text(arewaAPK.strings.submitting || 'Submitting...');
            messageDiv.hide();
            
            // Validate that we have the required data
            if (!arewaAPK || !arewaAPK.nonce || !arewaAPK.ajaxurl) {
                console.error('Missing arewaAPK data:', arewaAPK);
                messageDiv.removeClass('success').addClass('error').text('Configuration error. Please refresh the page.').show();
                submitBtn.prop('disabled', false).text(arewaAPK.strings.submit || 'Submit Comment');
                return;
            }
            
            // Prepare form data with file upload
            var formData = new FormData();
            // Get app_id from form data attribute
            var appId = form.data('app-id') || 0;
            
            formData.append('action', 'arewapap_submit_comment');
            formData.append('nonce', arewaAPK.nonce);
            formData.append('app_id', appId);
            formData.append('author_name', authorName);
            formData.append('author_email', authorEmail);
            formData.append('comment_text', commentText);
            
            // Get rating value
            var ratingValue = $('input[name="rating"]:checked').val();
            // If no rating is selected (shouldn't happen with default 5, but just in case), use 5 as default
            if (!ratingValue || ratingValue === 'undefined') {
                ratingValue = '5';
            }
            // If rating is 0 (user cleared it), send 0
            // Otherwise send the selected rating (1-5)
            formData.append('rating', ratingValue);
            
            // Add avatar file if selected
            var avatarFileInput = $('#arewa-comment-avatar')[0];
            if (avatarFileInput && avatarFileInput.files && avatarFileInput.files.length > 0) {
                var avatarFile = avatarFileInput.files[0];
                formData.append('author_avatar', avatarFile);
            }
            
            console.log('Submitting comment:', {
                author_name: authorName,
                author_email: authorEmail,
                rating: ratingValue,
                has_avatar: avatarFileInput && avatarFileInput.files && avatarFileInput.files.length > 0
            });
            
            // Submit via AJAX with file upload
            $.ajax({
                url: arewaAPK.ajaxurl,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                dataType: 'json',
                timeout: 30000, // 30 second timeout
                success: function(response) {
                    console.log('Response received:', response);
                    
                    if (response && response.success) {
                        messageDiv.removeClass('error').addClass('success').text(response.data && response.data.message ? response.data.message : arewaAPK.strings.success).show();
                        
                        // Update reviews stats if available
                        if (response.data && (response.data.total_comments !== undefined || response.data.average_rating !== undefined)) {
                            updateReviewsStats(response.data);
                        }
                        
                        // Reset form
                        form[0].reset();
                        
                        // Reset avatar preview
                        $('#arewa-avatar-preview').hide().removeClass('show active');
                        $('#arewa-avatar-preview-img').attr('src', '');
                        $('#arewa-avatar-upload-btn').show().css({
                            'display': 'inline-flex',
                            'visibility': 'visible',
                            'opacity': '1'
                        });
                        
                        // Reset rating stars to default (5 stars)
                        $('.arewa-rating-input input[type="radio"]').prop('checked', false);
                        $('#rating-5').prop('checked', true);
                        $('.arewa-rating-input .star-label').each(function() {
                            var labelRating = $(this).data('rating');
                            if (labelRating > 0) {
                                if (labelRating <= 5) {
                                    $(this).find('.arewa-star-input').css('color', '#00ff45');
                                } else {
                                    $(this).find('.arewa-star-input').css('color', '#ddd');
                                }
                            }
                        });
                        
                        // Refresh comments list via AJAX instead of reloading page
                        // This ensures cache is bypassed and new comment appears immediately
                        setTimeout(function() {
                            refreshCommentsList();
                        }, 500);
                    } else {
                        var errorMsg = (response && response.data && response.data.message) ? response.data.message : arewaAPK.strings.error;
                        console.error('Error response:', response);
                        messageDiv.removeClass('success').addClass('error').text(errorMsg).show();
                        submitBtn.prop('disabled', false).text(arewaAPK.strings.submit || 'Submit Comment');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX Error:', status, error);
                    console.error('Status Code:', xhr.status);
                    console.error('Response Text:', xhr.responseText);
                    console.error('Response Headers:', xhr.getAllResponseHeaders());
                    
                    var errorMsg = arewaAPK.strings.error || 'An error occurred. Please try again.';
                    
                    // Try to parse error response
                    try {
                        if (xhr.responseText) {
                            var response = JSON.parse(xhr.responseText);
                            if (response && response.data && response.data.message) {
                                errorMsg = response.data.message;
                            } else if (response && response.message) {
                                errorMsg = response.message;
                            }
                        }
                    } catch (e) {
                        console.error('Error parsing response:', e);
                        // If response is not JSON, use status-based messages
                        if (xhr.status === 0) {
                            errorMsg = 'Network error. Please check your connection and try again.';
                        } else if (xhr.status === 403) {
                            errorMsg = 'Security check failed. Please refresh the page and try again.';
                        } else if (xhr.status === 500) {
                            errorMsg = 'Server error. Please try again later or contact support.';
                        } else if (xhr.status === 404) {
                            errorMsg = 'Page not found. Please refresh the page and try again.';
                        } else if (xhr.status === 400) {
                            errorMsg = 'Invalid request. Please check your input and try again.';
                        } else if (status === 'timeout') {
                            errorMsg = 'Request timeout. Please try again.';
                        } else if (status === 'parsererror') {
                            errorMsg = 'Server response error. Please try again.';
                        }
                    }
                    
                    messageDiv.removeClass('success').addClass('error').text(errorMsg).show();
                    submitBtn.prop('disabled', false).text(arewaAPK.strings.submit || 'Submit Comment');
                }
            });
        });
        
        // Vote Button Handler - Improved with toggle functionality
        $(document).on('click', '.arewa-vote-btn', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            var button = $(this);
            var commentId = button.data('comment-id');
            var voteType = button.data('vote-type');
            var commentVotes = button.closest('.arewa-comment-votes');
            var helpfulBtn = commentVotes.find('.arewa-vote-helpful');
            var notHelpfulBtn = commentVotes.find('.arewa-vote-not-helpful');
            var helpfulCount = helpfulBtn.find('.vote-count');
            var notHelpfulCount = notHelpfulBtn.find('.vote-count');
            
            // Check if user already voted this type
            var isVoted = button.hasClass('arewa-voted');
            
            // Disable buttons temporarily
            helpfulBtn.prop('disabled', true);
            notHelpfulBtn.prop('disabled', true);
            
            // Submit vote with toggle action
            $.ajax({
                url: arewaAPK.ajaxurl,
                type: 'POST',
                data: {
                    action: 'arewapap_submit_vote',
                    nonce: arewaAPK.nonce,
                    comment_id: commentId,
                    vote_type: voteType
                },
                success: function(response) {
                    if (response.success && response.data) {
                        // Update vote counts
                        helpfulCount.text(response.data.helpful_count || 0);
                        notHelpfulCount.text(response.data.not_helpful_count || 0);
                        
                        // Update cookie immediately in browser
                        var cookieName = response.data.cookie_name || ('arewa_vote_' + commentId);
                        if (response.data.current_vote) {
                            var expires = new Date();
                            expires.setTime(expires.getTime() + (30 * 24 * 60 * 60 * 1000)); // 30 days
                            document.cookie = cookieName + '=' + response.data.current_vote + '; path=/; expires=' + expires.toUTCString() + '; SameSite=Lax';
                        } else {
                            // Remove cookie
                            document.cookie = cookieName + '=; path=/; expires=Thu, 01 Jan 1970 00:00:00 UTC; SameSite=Lax';
                        }
                        
                        // Update button states based on current vote
                        if (response.data.current_vote === 'helpful') {
                            // User voted helpful
                            helpfulBtn.addClass('arewa-voted');
                            notHelpfulBtn.removeClass('arewa-voted');
                            helpfulBtn.css({
                                'background': '#667eea',
                                'color': '#fff',
                                'border-color': '#667eea'
                            });
                            notHelpfulBtn.css({
                                'background': '#f8f9fa',
                                'color': '#555',
                                'border-color': '#ddd'
                            });
                        } else if (response.data.current_vote === 'not_helpful') {
                            // User voted not helpful
                            notHelpfulBtn.addClass('arewa-voted');
                            helpfulBtn.removeClass('arewa-voted');
                            notHelpfulBtn.css({
                                'background': '#e74c3c',
                                'color': '#fff',
                                'border-color': '#e74c3c'
                            });
                            helpfulBtn.css({
                                'background': '#f8f9fa',
                                'color': '#555',
                                'border-color': '#ddd'
                            });
                        } else {
                            // User removed vote
                            helpfulBtn.removeClass('arewa-voted');
                            notHelpfulBtn.removeClass('arewa-voted');
                            helpfulBtn.css({
                                'background': '#f8f9fa',
                                'color': '#555',
                                'border-color': '#ddd'
                            });
                            notHelpfulBtn.css({
                                'background': '#f8f9fa',
                                'color': '#555',
                                'border-color': '#ddd'
                            });
                        }
                    }
                    helpfulBtn.prop('disabled', false);
                    notHelpfulBtn.prop('disabled', false);
                },
                error: function() {
                    helpfulBtn.prop('disabled', false);
                    notHelpfulBtn.prop('disabled', false);
                }
            });
        });
        
        // Initialize vote button states from cookies on page load
        function initializeVoteStates() {
            $('.arewa-comment-votes').each(function() {
                var commentVotes = $(this);
                var helpfulBtn = commentVotes.find('.arewa-vote-helpful');
                var notHelpfulBtn = commentVotes.find('.arewa-vote-not-helpful');
                var commentId = helpfulBtn.data('comment-id');
                
                if (!commentId) return;
                
                // Check cookie for user's vote
                var cookieName = 'arewa_vote_' + commentId;
                var cookieValue = getCookie(cookieName);
                
                if (cookieValue === 'helpful') {
                    helpfulBtn.addClass('arewa-voted');
                    helpfulBtn.css({
                        'background': '#667eea',
                        'color': '#fff',
                        'border-color': '#667eea'
                    });
                } else if (cookieValue === 'not_helpful') {
                    notHelpfulBtn.addClass('arewa-voted');
                    notHelpfulBtn.css({
                        'background': '#e74c3c',
                        'color': '#fff',
                        'border-color': '#e74c3c'
                    });
                }
            });
        }
        
        // Helper function to get cookie value
        function getCookie(name) {
            var value = "; " + document.cookie;
            var parts = value.split("; " + name + "=");
            if (parts.length === 2) return parts.pop().split(";").shift();
            return '';
        }
        
        // Initialize vote states on page load
        initializeVoteStates();
        
        // Email validation helper
        function isValidEmail(email) {
            var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return emailRegex.test(email);
        }
        
        // Screenshots Carousel - Global variables for carousel state
        var arewaCarousel = {
            currentSlide: 0,
            totalSlides: 0,
            slidesPerView: 1,
            autoSlideInterval: null,
            wrapper: null,
            carousel: null
        };
        
        function initScreenshotsCarousel() {
            var carousel = $('.arewa-screenshots-carousel');
            if (!carousel.length) {
                console.log('Arewa APK: Carousel container not found');
                return; // Carousel doesn't exist on this page
            }
            
            var wrapper = carousel.find('.arewa-screenshots-carousel-wrapper');
            if (!wrapper.length) {
                console.log('Arewa APK: Carousel wrapper not found');
                return;
            }
            
            arewaCarousel.carousel = carousel;
            arewaCarousel.wrapper = wrapper;
            arewaCarousel.currentSlide = 0;
            var slides = carousel.find('.arewa-screenshot-slide');
            arewaCarousel.totalSlides = slides.length;
            
            // Determine slides per view based on screen size
            function updateSlidesPerView() {
                var width = $(window).width();
                if (width >= 1025) {
                    arewaCarousel.slidesPerView = 6; // Desktop: 6 slides
                } else if (width >= 769) {
                    arewaCarousel.slidesPerView = 4; // Tablet: 4 slides
                } else {
                    arewaCarousel.slidesPerView = 3; // Mobile: 3 slides
                }
            }
            
            updateSlidesPerView();
            $(window).on('resize', function() {
                updateSlidesPerView();
                updateCarouselPosition();
            });
            
            console.log('Arewa APK: Initializing carousel with', arewaCarousel.totalSlides, 'slides, showing', arewaCarousel.slidesPerView, 'per view');
            
            if (arewaCarousel.totalSlides === 0) {
                console.log('Arewa APK: No slides found');
                return; // No slides to show
            }
            
            function updateCarouselPosition() {
                updateSlidesPerView();
                if (!arewaCarousel.wrapper || arewaCarousel.totalSlides === 0) return;
                
                // Calculate translateX based on current slide and slides per view
                var translateX = -(arewaCarousel.currentSlide * (100 / arewaCarousel.slidesPerView));
                arewaCarousel.wrapper.css({
                    'transform': 'translateX(' + translateX + '%)',
                    'transition': 'transform 0.6s cubic-bezier(0.25, 0.46, 0.45, 0.94)'
                });
                
                console.log('Arewa APK: Showing slide', arewaCarousel.currentSlide, 'with', arewaCarousel.slidesPerView, 'slides per view, translateX:', translateX + '%');
            }
            
            // Prevent accidental auto-slide when user just touches/clicks
            var touchStartTime = 0;
            var minTouchTime = 100; // Minimum time in ms to consider it a touch (not a quick tap)
            
            function showSlide(index) {
                if (arewaCarousel.totalSlides === 0) return;
                
                updateSlidesPerView();
                // Calculate max slide index based on slides per view
                var maxIndex = Math.max(0, arewaCarousel.totalSlides - arewaCarousel.slidesPerView);
                
                // Ensure index is within bounds
                if (index < 0) {
                    index = maxIndex;
                } else if (index > maxIndex) {
                    index = 0;
                }
                
                arewaCarousel.currentSlide = index;
                updateCarouselPosition();
            }
            
            function nextSlide() {
                if (arewaCarousel.totalSlides === 0) return;
                updateSlidesPerView();
                var maxIndex = Math.max(0, arewaCarousel.totalSlides - arewaCarousel.slidesPerView);
                var nextIndex = arewaCarousel.currentSlide + 1;
                if (nextIndex > maxIndex) {
                    nextIndex = 0; // Loop back to start
                }
                showSlide(nextIndex);
            }
            
            function prevSlide() {
                if (arewaCarousel.totalSlides === 0) return;
                updateSlidesPerView();
                var maxIndex = Math.max(0, arewaCarousel.totalSlides - arewaCarousel.slidesPerView);
                var prevIndex = arewaCarousel.currentSlide - 1;
                if (prevIndex < 0) {
                    prevIndex = maxIndex; // Loop to end
                }
                showSlide(prevIndex);
            }
            
            function startAutoSlide() {
                if (arewaCarousel.totalSlides > 1) {
                    stopAutoSlide(); // Clear any existing interval
                    console.log('Arewa APK: Starting auto-slide');
                    // Delay auto-slide start to prevent immediate movement after touch
                    setTimeout(function() {
                        if (!isDragging) {
                            arewaCarousel.autoSlideInterval = setInterval(function() {
                                if (!isDragging) {
                                    nextSlide();
                                }
                            }, 5000); // Change slide every 5 seconds (slower)
                        }
                    }, 1000); // Wait 1 second before starting auto-slide
                }
            }
            
            function stopAutoSlide() {
                if (arewaCarousel.autoSlideInterval) {
                    clearInterval(arewaCarousel.autoSlideInterval);
                    arewaCarousel.autoSlideInterval = null;
                }
            }
            
            // Store functions globally for event handlers
            window.arewaNextSlide = nextSlide;
            window.arewaPrevSlide = prevSlide;
            window.arewaStopAutoSlide = stopAutoSlide;
            window.arewaStartAutoSlide = startAutoSlide;
            
            // Initialize position
            updateCarouselPosition();
            
            // Arrows removed - using drag/swipe only
            
            // Start auto-slide if carousel exists and has multiple slides
            if (arewaCarousel.totalSlides > 1) {
                startAutoSlide();
                
                // Pause on hover (desktop only)
                if (window.matchMedia('(min-width: 769px)').matches) {
                    carousel.closest('.arewa-apk-screenshots').on('mouseenter', function() {
                        stopAutoSlide();
                    }).on('mouseleave', function() {
                        startAutoSlide();
                    });
                }
            }
            
            // Enhanced drag/swipe support for both mobile and desktop
            var isDragging = false;
            var hasMoved = false;
            var startX = 0;
            var startY = 0;
            var currentX = 0;
            var currentY = 0;
            var startTranslate = 0;
            var currentTranslate = 0;
            var dragThreshold = 50;
            var minDragDistance = 10; // Minimum distance to consider it a drag
            
            // Touch events for mobile
            carousel.on('touchstart', function(e) {
                stopAutoSlide();
                touchStartTime = Date.now();
                isDragging = true;
                hasMoved = false;
                startX = e.originalEvent.touches[0].clientX;
                startY = e.originalEvent.touches[0].clientY;
                startTranslate = -(arewaCarousel.currentSlide * (100 / arewaCarousel.slidesPerView));
                carousel.css('cursor', 'grabbing');
            });
            
            carousel.on('touchmove', function(e) {
                if (!isDragging) return;
                
                currentX = e.originalEvent.touches[0].clientX;
                currentY = e.originalEvent.touches[0].clientY;
                var diffX = currentX - startX;
                var diffY = currentY - startY;
                
                // Check if user has moved enough to consider it a drag
                if (Math.abs(diffX) > minDragDistance || Math.abs(diffY) > minDragDistance) {
                    hasMoved = true;
                }
                
                // Only process horizontal drags
                if (Math.abs(diffX) > Math.abs(diffY)) {
                    e.preventDefault(); // Prevent scrolling only when dragging horizontally
                    currentTranslate = startTranslate + (diffX / carousel.width() * 100);
                    arewaCarousel.wrapper.css({
                        'transform': 'translateX(' + currentTranslate + '%)',
                        'transition': 'none'
                    });
                }
            });
            
            carousel.on('touchend', function(e) {
                if (!isDragging) return;
                isDragging = false;
                carousel.css('cursor', 'grab');
                
                var touchDuration = Date.now() - touchStartTime;
                
                // If user just touched without moving, don't do anything
                if (!hasMoved || touchDuration < minTouchTime) {
                    hasMoved = false;
                    // Wait a bit before restarting auto-slide to prevent immediate movement
                    setTimeout(function() {
                        startAutoSlide();
                    }, 500);
                    return;
                }
                
                var diffX = currentX - startX;
                var diffY = currentY - startY;
                
                // Only process horizontal swipes
                if (Math.abs(diffX) > Math.abs(diffY) && Math.abs(diffX) > dragThreshold) {
                    if (diffX < 0) {
                        // Swipe left (finger moved left) - show next slide
                        nextSlide();
                    } else {
                        // Swipe right (finger moved right) - show previous slide
                        prevSlide();
                    }
                } else {
                    // Snap back to current position
                    updateCarouselPosition();
                }
                
                hasMoved = false;
                // Wait a bit before restarting auto-slide
                setTimeout(function() {
                    startAutoSlide();
                }, 500);
            });
            
            // Mouse events for desktop
            carousel.on('mousedown', function(e) {
                stopAutoSlide();
                isDragging = true;
                hasMoved = false;
                startX = e.clientX;
                startY = e.clientY;
                startTranslate = -(arewaCarousel.currentSlide * (100 / arewaCarousel.slidesPerView));
                carousel.css('cursor', 'grabbing');
                e.preventDefault();
            });
            
            $(document).on('mousemove', function(e) {
                if (!isDragging) return;
                
                currentX = e.clientX;
                currentY = e.clientY;
                var diffX = currentX - startX;
                var diffY = currentY - startY;
                
                // Check if user has moved enough to consider it a drag
                if (Math.abs(diffX) > minDragDistance || Math.abs(diffY) > minDragDistance) {
                    hasMoved = true;
                }
                
                // Only process horizontal drags
                if (Math.abs(diffX) > Math.abs(diffY)) {
                    currentTranslate = startTranslate + (diffX / carousel.width() * 100);
                    arewaCarousel.wrapper.css({
                        'transform': 'translateX(' + currentTranslate + '%)',
                        'transition': 'none'
                    });
                }
            });
            
            $(document).on('mouseup', function(e) {
                if (!isDragging) return;
                isDragging = false;
                carousel.css('cursor', 'grab');
                
                // Only process drag if user actually moved
                if (!hasMoved) {
                    startAutoSlide();
                    return;
                }
                
                var diffX = currentX - startX;
                var diffY = currentY - startY;
                
                // Only process horizontal drags
                if (Math.abs(diffX) > Math.abs(diffY) && Math.abs(diffX) > dragThreshold) {
                    if (diffX < 0) {
                        // Drag left (mouse moved left) - show next slide
                        nextSlide();
                    } else {
                        // Drag right (mouse moved right) - show previous slide
                        prevSlide();
                    }
                } else {
                    // Snap back to current position
                    updateCarouselPosition();
                }
                
                hasMoved = false;
                startAutoSlide();
            });
            
            // Set initial cursor style
            carousel.css('cursor', 'grab');
            carousel.css('user-select', 'none');
            arewaCarousel.wrapper.css('user-select', 'none');
        }
        
        // Initialize carousel - wait a bit to ensure DOM is ready
        setTimeout(function() {
            initScreenshotsCarousel();
        }, 300);
        
        // Lightbox functionality
        var lightbox = $('#arewa-lightbox');
        var lightboxImage = $('#arewa-lightbox-image');
        var currentLightboxIndex = 0;
        var allScreenshots = [];
        
        // Get all screenshot images
        $('.arewa-screenshot-slide img').each(function() {
            allScreenshots.push($(this).data('full-image'));
        });
        
        // Open lightbox when clicking on screenshot
        $(document).on('click', '.arewa-screenshot-slide img', function() {
            var imageUrl = $(this).data('full-image');
            var imageIndex = parseInt($(this).data('image-index')) || 0;
            
            currentLightboxIndex = imageIndex;
            lightboxImage.attr('src', imageUrl);
            lightbox.addClass('active');
            $('body').css('overflow', 'hidden'); // Prevent background scrolling
        });
        
        // Close lightbox
        $('.arewa-lightbox-close').on('click', function() {
            lightbox.removeClass('active');
            $('body').css('overflow', '');
        });
        
        // Close lightbox when clicking on background
        lightbox.on('click', function(e) {
            if ($(e.target).hasClass('arewa-lightbox')) {
                lightbox.removeClass('active');
                $('body').css('overflow', '');
            }
        });
        
        // Close lightbox with ESC key
        $(document).on('keydown', function(e) {
            if (e.key === 'Escape' && lightbox.hasClass('active')) {
                lightbox.removeClass('active');
                $('body').css('overflow', '');
            }
        });
        
        // Lightbox navigation
        function showLightboxImage(index) {
            if (allScreenshots.length === 0) return;
            
            if (index < 0) {
                index = allScreenshots.length - 1;
            } else if (index >= allScreenshots.length) {
                index = 0;
            }
            
            currentLightboxIndex = index;
            lightboxImage.attr('src', allScreenshots[index]);
        }
        
        $('.arewa-lightbox-next').on('click', function(e) {
            e.stopPropagation();
            showLightboxImage(currentLightboxIndex + 1);
        });
        
        $('.arewa-lightbox-prev').on('click', function(e) {
            e.stopPropagation();
            showLightboxImage(currentLightboxIndex - 1);
        });
        
        // Keyboard navigation in lightbox
        $(document).on('keydown', function(e) {
            if (lightbox.hasClass('active')) {
                if (e.key === 'ArrowRight') {
                    showLightboxImage(currentLightboxIndex + 1);
                } else if (e.key === 'ArrowLeft') {
                    showLightboxImage(currentLightboxIndex - 1);
                }
            }
        });
        
        // Download tracking
        $('.arewa-apk-download-btn, .arewa-apk-archive-download-btn').on('click', function(e) {
            var downloadBtn = $(this);
            var apkUrl = downloadBtn.data('apk-url') || downloadBtn.attr('href');
            var appId = downloadBtn.data('app-id') || 0;
            
            // Track download via AJAX
            $.ajax({
                url: arewaAPK.ajaxurl,
                type: 'POST',
                data: {
                    action: 'arewapap_track_download',
                    app_id: appId,
                    nonce: arewaAPK.nonce
                },
                success: function(response) {
                    if (response.success && response.data.count !== undefined) {
                        // Format download count in abbreviated form (K, M)
                        function formatDownloadCount(count) {
                            if (count >= 1000000) {
                                var formatted = (count / 1000000).toFixed(1);
                                return formatted.replace('.0', '') + 'M';
                            } else if (count >= 1000) {
                                var formatted = (count / 1000).toFixed(1);
                                return formatted.replace('.0', '') + 'K';
                            } else {
                                return count.toString();
                            }
                        }
                        
                        // Update download count display
                        var count = response.data.count;
                        var formattedCount = formatDownloadCount(count);
                        var downloadText = (count === 1 ? 'download' : 'downloads');
                        
                        // Update count in download button
                        var countElement = downloadBtn.find('.arewa-download-count');
                        if (countElement.length) {
                            countElement.text(formattedCount + ' ' + downloadText);
                        } else {
                            // Add count if it doesn't exist
                            downloadBtn.append('<span class="arewa-download-count">' + 
                                formattedCount + ' ' + downloadText + '</span>');
                        }
                        
                        // Update count in archive meta if exists (for archive page)
                        if (appId > 0) {
                            var archiveDownloads = $('.arewa-apk-archive-item[data-app-id="' + appId + '"] .arewa-apk-archive-downloads');
                            if (archiveDownloads.length) {
                                archiveDownloads.text(formattedCount + ' ' + downloadText);
                            }
                        }
                    }
                }
            });
            
            // Allow the download to proceed normally
            // The link will handle the actual download
        });
        
        // Comments filtering
        var commentsFilter = {
            currentPage: 1,
            sortBy: 'date',
            order: 'desc',
            perPage: 10,
            
            init: function() {
                var self = this;
                
                // Get initial values from URL or select
                var urlParams = new URLSearchParams(window.location.search);
                var pageFromUrl = urlParams.get('comment_page');
                if (pageFromUrl) {
                    var parsedPage = parseInt(pageFromUrl);
                    self.currentPage = (parsedPage > 0) ? parsedPage : 1;
                } else {
                    self.currentPage = 1;
                }
                self.sortBy = urlParams.get('sort_by') || $('#arewa-sort-by').val() || 'date';
                self.order = urlParams.get('order') || $('#arewa-sort-by option:selected').data('order') || 'desc';
                
                // Set initial select value
                $('#arewa-sort-by option').each(function() {
                    var $option = $(this);
                    if ($option.val() === self.sortBy && $option.data('order') === self.order) {
                        $option.prop('selected', true);
                    }
                });
                
                // Handle filter select change
                $('#arewa-sort-by').on('change', function() {
                    var $selected = $(this).find('option:selected');
                    self.sortBy = $selected.val();
                    self.order = $selected.data('order') || 'desc';
                    self.currentPage = 1; // Reset to first page when filter changes
                    self.loadComments();
                });
                
                // Handle reset button
                $('#arewa-filter-reset').on('click', function() {
                    self.sortBy = 'date';
                    self.order = 'desc';
                    self.currentPage = 1;
                    $('#arewa-sort-by').val('date');
                    $('#arewa-sort-by option[data-order="desc"]').prop('selected', true);
                    self.loadComments();
                });
                
                // Handle pagination clicks - allow full page reload instead of AJAX
                // This ensures pagination buttons always appear correctly
                $(document).on('click', '.arewa-comments-pagination a, #arewa-comments-pagination a', function(e) {
                    // Don't prevent default - let the browser follow the link normally
                    // This will cause a full page reload with the correct URL parameters
                    // The page will reload and PHP will render the correct pagination
                    return true; // Allow normal link behavior
                });
            },
            
            loadComments: function() {
                var self = this;
                var $commentsList = $('#arewa-comments-list');
                var $commentsLoading = $('#arewa-comments-loading');
                var $commentsPagination = $('#arewa-comments-pagination');
                
                // Validate current page
                if (self.currentPage < 1) {
                    self.currentPage = 1;
                }
                
                // Show loading state
                $commentsLoading.show();
                $commentsList.hide();
                // Don't hide pagination during loading - just keep it visible
                // This prevents flickering and ensures pagination stays visible
                
                // Update URL without reload - only include comment_page, keep sort_by/order in state only
                var currentUrl = new URL(window.location);
                // Remove sort parameters from URL
                currentUrl.searchParams.delete('sort_by');
                currentUrl.searchParams.delete('order');
                // Only set comment_page if not page 1
                if (self.currentPage > 1) {
                    currentUrl.searchParams.set('comment_page', self.currentPage);
                } else {
                    currentUrl.searchParams.delete('comment_page');
                }
                window.history.pushState({page: self.currentPage, sort_by: self.sortBy, order: self.order}, '', currentUrl.toString());
                
                // Load comments via AJAX
                $.ajax({
                    url: arewaAPK.ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'arewapap_filter_comments',
                        nonce: arewaAPK.nonce,
                        app_id: $('#arewa-apk-comment-form').data('app-id') || 0,
                        page: self.currentPage,
                        per_page: self.perPage,
                        sort_by: self.sortBy,
                        order: self.order,
                        timestamp: new Date().getTime() // Cache busting
                    },
                    cache: false,
                    success: function(response) {
                        if (response && response.success) {
                            // Update comments list
                            if (response.data.comments_html) {
                                $commentsList.html(response.data.comments_html);
                            } else {
                                $commentsList.html('<p class="arewa-no-comments">' + (arewaAPK.strings.noComments || 'No comments yet. Be the first to review!') + '</p>');
                            }
                            
                            // Update pagination
                            if (response.data.total_pages && response.data.total_pages > 1) {
                                // We have multiple pages, so pagination should be visible
                                if (response.data.pagination_html && response.data.pagination_html.trim()) {
                                    // Clear and set new pagination HTML
                                    $commentsPagination.empty().html(response.data.pagination_html);
                                    // Force show to ensure visibility
                                    $commentsPagination.show();
                                } else {
                                    // If pagination HTML is missing but we have pages, keep existing or show warning
                                    console.warn('Pagination HTML is empty but total_pages > 1. Keeping existing pagination.');
                                    if ($commentsPagination.children().length === 0) {
                                        // If no existing pagination, show error message
                                        $commentsPagination.html('<p style="color: red;">Pagination error</p>').show();
                                    } else {
                                        $commentsPagination.show();
                                    }
                                }
                            } else {
                                // Only one page or no pages, hide pagination
                                $commentsPagination.empty().hide();
                            }
                            
                            // Update reviews stats if available
                            if (response.data.total_comments !== undefined || response.data.average_rating !== undefined) {
                                updateReviewsStats(response.data);
                            }
                            
                            // Update filter state from response
                            if (response.data.sort_by) {
                                self.sortBy = response.data.sort_by;
                            }
                            if (response.data.order) {
                                self.order = response.data.order;
                            }
                            if (response.data.current_page) {
                                self.currentPage = parseInt(response.data.current_page) || 1;
                            }
                            
                            // Validate page number doesn't exceed total pages
                            if (response.data.total_pages && self.currentPage > response.data.total_pages) {
                                self.currentPage = Math.max(1, response.data.total_pages);
                                // Reload with correct page
                                self.loadComments();
                                return;
                            }
                            
                            // Re-initialize vote buttons for new comments
                            if (typeof initializeVoteStates === 'function') {
                                initializeVoteStates();
                            }
                            
                            // Scroll to comments section smoothly
                            var commentsSection = $('.arewa-apk-comments-section');
                            if (commentsSection.length) {
                                $('html, body').animate({
                                    scrollTop: commentsSection.offset().top - 100
                                }, 500);
                            }
                        } else {
                            console.error('Error loading comments:', response);
                            var errorMsg = (response && response.data && response.data.message) ? response.data.message : (arewaAPK.strings.error || 'Error loading comments');
                            $commentsList.html('<p class="arewa-no-comments">' + errorMsg + '</p>');
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('AJAX Error:', status, error);
                        $commentsList.html('<p class="arewa-no-comments">' + arewaAPK.strings.error + '</p>');
                    },
                    complete: function() {
                        // Hide loading state
                        $commentsLoading.hide();
                        $commentsList.show();
                        // Show pagination if it has content
                        if ($commentsPagination.children().length > 0) {
                            $commentsPagination.show();
                        }
                    }
                });
            }
        };
        
        // Function to update reviews stats
        function updateReviewsStats(data) {
            var $statsContainer = $('.arewa-reviews-stats');
            if ($statsContainer.length === 0) {
                // Create stats container if it doesn't exist
                var $header = $('.arewa-comments-header');
                if ($header.length) {
                    $header.append('<div class="arewa-reviews-stats"></div>');
                    $statsContainer = $('.arewa-reviews-stats');
                } else {
                    return; // Can't update if header doesn't exist
                }
            }
            
            var statsHtml = '';
            
            // Total reviews
            if (data.total_comments !== undefined && data.total_comments > 0) {
                var reviewsText = data.total_comments === 1 ? 'Review' : 'Reviews';
                statsHtml += '<span class="arewa-total-reviews"><strong>' + 
                    parseInt(data.total_comments).toLocaleString() + '</strong> ' + reviewsText + '</span>';
            }
            
            // Average rating
            if (data.average_rating !== undefined && data.average_rating > 0 && data.rating_count !== undefined && data.rating_count > 0) {
                var avgRating = parseFloat(data.average_rating);
                var fullStars = Math.floor(avgRating);
                var hasHalf = (avgRating - fullStars) >= 0.5;
                
                var starsHtml = '';
                for (var i = 1; i <= 5; i++) {
                    if (i <= fullStars) {
                        starsHtml += '<span class="arewa-star arewa-star-filled" style="color: #00ff45;">★</span>';
                    } else if (i === fullStars + 1 && hasHalf) {
                        starsHtml += '<span class="arewa-star arewa-star-half" style="color: #00ff45;">★</span>';
                    } else {
                        starsHtml += '<span class="arewa-star arewa-star-empty" style="color: #ddd;">☆</span>';
                    }
                }
                
                statsHtml += '<span class="arewa-average-rating"><strong>' + 
                    avgRating.toFixed(1) + '</strong><span class="arewa-rating-stars-inline">' + 
                    starsHtml + '</span><span class="arewa-rating-count">(' + 
                    parseInt(data.rating_count).toLocaleString() + ')</span></span>';
            }
            
            if (statsHtml) {
                $statsContainer.html(statsHtml);
            }
        }
        
        // Initialize comments filter if filter elements exist
        if ($('#arewa-sort-by').length) {
            commentsFilter.init();
            
            // Handle browser back/forward navigation
            window.addEventListener('popstate', function(event) {
                if (event.state) {
                    commentsFilter.currentPage = event.state.page || 1;
                    commentsFilter.sortBy = event.state.sort_by || 'date';
                    commentsFilter.order = event.state.order || 'desc';
                    commentsFilter.loadComments();
                } else {
                    // If no state, read from URL
                    var urlParams = new URLSearchParams(window.location.search);
                    var pageFromUrl = urlParams.get('comment_page');
                    if (pageFromUrl) {
                        var parsedPage = parseInt(pageFromUrl);
                        commentsFilter.currentPage = (parsedPage > 0) ? parsedPage : 1;
                    } else {
                        commentsFilter.currentPage = 1;
                    }
                    commentsFilter.sortBy = urlParams.get('sort_by') || 'date';
                    commentsFilter.order = urlParams.get('order') || 'desc';
                    commentsFilter.loadComments();
                }
            });
            
            // Note: We don't reload comments on initial page load because PHP already
            // loads them correctly based on URL parameters. This prevents unnecessary
            // AJAX calls and content flashing. Comments will only be reloaded via AJAX
            // when user interacts with pagination or filters.
        }
        
        // Function to refresh comments list after new comment is added
        // Make it global so it can be called from setTimeout
        window.refreshCommentsList = function() {
            var $commentsList = $('#arewa-comments-list');
            var $commentsPagination = $('#arewa-comments-pagination');
            var $loadingIndicator = $('#arewa-comments-loading');
            var $filterSelect = $('#arewa-sort-by');
            
            // Get current filter values
            var sortBy = 'date';
            var order = 'desc';
            var page = 1;
            
            if ($filterSelect.length > 0) {
                var selectedOption = $filterSelect.find('option:selected');
                sortBy = $filterSelect.val() || 'date';
                order = selectedOption.data('order') || 'desc';
            }
            
            // Show loading indicator
            if ($loadingIndicator.length) {
                $loadingIndicator.show();
            }
            $commentsList.css('opacity', '0.5');
            
            // Load comments via AJAX with cache-busting timestamp
            $.ajax({
                url: arewaAPK.ajaxurl,
                type: 'POST',
                data: {
                    action: 'arewa_filter_comments',
                    nonce: arewaAPK.nonce,
                    app_id: $('#arewa-apk-comment-form').data('app-id') || 0,
                    page: page,
                    per_page: 10,
                    sort_by: sortBy,
                    order: order,
                    timestamp: new Date().getTime() // Add timestamp to bypass cache
                },
                cache: false, // Disable cache for this request
                success: function(response) {
                    if (response && response.success) {
                        // Update comments list
                        $commentsList.html(response.data.comments_html);
                        $commentsList.css('opacity', '1');
                        
                        // Update pagination
                        if (response.data.pagination_html) {
                            $commentsPagination.html(response.data.pagination_html);
                        } else {
                            $commentsPagination.empty();
                        }
                        
                        // Re-initialize vote buttons for new comments
                        if (typeof initializeVoteStates === 'function') {
                            initializeVoteStates();
                        }
                        
                        // Scroll to comments section smoothly
                        var commentsSection = $('.arewa-apk-comments-section');
                        if (commentsSection.length) {
                            $('html, body').animate({
                                scrollTop: commentsSection.offset().top - 50
                            }, 500);
                        }
                    } else {
                        // If AJAX fails, fall back to page reload
                        location.reload();
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Error refreshing comments:', status, error);
                    // If AJAX fails, fall back to page reload
                    location.reload();
                },
                complete: function() {
                    // Hide loading indicator
                    if ($loadingIndicator.length) {
                        $loadingIndicator.hide();
                    }
                    $commentsList.css('opacity', '1');
                }
            });
        };
        
    });
    
})(jQuery);

