<?php
/**
 * AJAX Handlers for Import/Export
 * 
 * @package Arewa_APK_App
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class Arewa_APK_Import_Export_AJAX {
    
    /**
     * Register AJAX handlers
     */
    public static function init() {
        add_action('wp_ajax_arewapap_export_content', array(__CLASS__, 'ajax_export_content'));
        add_action('wp_ajax_arewapap_import_content', array(__CLASS__, 'ajax_import_content'));
    }
    
    /**
     * AJAX handler for exporting apps
     */
    public static function ajax_export_content() {
        // Verify user is admin
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'arewapap_export_content')) {
            wp_send_json_error(array('message' => __('Security check failed. Please refresh the page and try again.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Get selected app IDs - sanitize input
        if (!isset($_POST['app_ids']) || !is_array($_POST['app_ids']) || empty($_POST['app_ids'])) {
            wp_send_json_error(array('message' => __('Please select at least one app to export.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Sanitize app IDs: wp_unslash to remove slashes added by WordPress, then absint each ID
        $app_ids = array_map('absint', array_map('sanitize_text_field', wp_unslash($_POST['app_ids'])));
        
        // Export apps
        $result = Arewa_APK_Export::export_apps($app_ids);
        
        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
            return;
        }
        
        wp_send_json_success(array(
            'message' => sprintf(
                /* translators: %d: Number of apps exported */
                __('%d apps exported successfully!', 'arewa-apk-apps-manager'),
                $result['total_apps']
            ),
            'file_url' => $result['file_url'],
            'filename' => $result['filename'],
            'total_apps' => $result['total_apps']
        ));
    }
    
    /**
     * AJAX handler for importing apps
     */
    public static function ajax_import_content() {
        // Verify user is admin
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission to perform this action.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'arewapap_import_content')) {
            wp_send_json_error(array('message' => __('Security check failed. Please refresh the page and try again.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Check if file was uploaded
        if (!isset($_FILES['import_file']) || !is_array($_FILES['import_file'])) {
            wp_send_json_error(array('message' => __('No file uploaded.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Check for upload errors
        if (!isset($_FILES['import_file']['error']) || $_FILES['import_file']['error'] !== UPLOAD_ERR_OK) {
            $error_message = __('File upload error occurred.', 'arewa-apk-apps-manager');
            if (isset($_FILES['import_file']['error'])) {
                switch ($_FILES['import_file']['error']) {
                    case UPLOAD_ERR_INI_SIZE:
                    case UPLOAD_ERR_FORM_SIZE:
                        $error_message = __('File is too large.', 'arewa-apk-apps-manager');
                        break;
                    case UPLOAD_ERR_PARTIAL:
                        $error_message = __('File was only partially uploaded.', 'arewa-apk-apps-manager');
                        break;
                    case UPLOAD_ERR_NO_FILE:
                        $error_message = __('No file was uploaded.', 'arewa-apk-apps-manager');
                        break;
                }
            }
            wp_send_json_error(array('message' => $error_message));
            return;
        }
        
        // Validate uploaded file for security - must be verified before processing
        $tmp_name = isset($_FILES['import_file']['tmp_name']) ? $_FILES['import_file']['tmp_name'] : '';
        if (empty($tmp_name) || !is_string($tmp_name) || !is_uploaded_file($tmp_name)) {
            wp_send_json_error(array('message' => __('Invalid file upload. Security validation failed.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Validate file size (check against WordPress upload limit)
        $file_size = isset($_FILES['import_file']['size']) ? absint($_FILES['import_file']['size']) : 0;
        $max_upload_size = wp_max_upload_size();
        if ($file_size === 0 || $file_size > $max_upload_size) {
            wp_send_json_error(array(
                'message' => sprintf(
                    /* translators: %s: Maximum upload size */
                    __('File is too large. Maximum size allowed is %s.', 'arewa-apk-apps-manager'),
                    size_format($max_upload_size)
                )
            ));
            return;
        }
        
        // Validate file type - sanitize file name before checking extension
        $file_name = isset($_FILES['import_file']['name']) ? sanitize_file_name(wp_unslash($_FILES['import_file']['name'])) : '';
        if (empty($file_name)) {
            wp_send_json_error(array('message' => __('Invalid file name.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Get file extension safely
        $file_ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
        $allowed_extensions = array('zip');
        if (empty($file_ext) || !in_array($file_ext, $allowed_extensions, true)) {
            wp_send_json_error(array('message' => __('Invalid file type. Please upload a ZIP file.', 'arewa-apk-apps-manager')));
            return;
        }
        
        // Import from ZIP
        $result = Arewa_APK_Import::import_from_zip($tmp_name);
        
        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
            return;
        }
        
        // translators: %d: Number of apps imported
        $message = sprintf(__('Import completed! %d apps imported.', 'arewa-apk-apps-manager'), $result['imported_count']);
        if ($result['skipped_count'] > 0) {
            // translators: %d: Number of apps skipped
            $message .= ' ' . sprintf(__('%d apps skipped (already exist).', 'arewa-apk-apps-manager'), $result['skipped_count']);
        }
        if (!empty($result['errors'])) {
            // translators: %d: Number of errors
            $message .= ' ' . sprintf(__('Errors: %d', 'arewa-apk-apps-manager'), count($result['errors']));
        }
        
        wp_send_json_success(array(
            'message' => $message,
            'imported_count' => $result['imported_count'],
            'skipped_count' => $result['skipped_count'],
            'errors' => $result['errors'],
        ));
    }
}

// Note: init() is called from the main plugin file to avoid multiple registrations

