<?php
/**
 * Export Handler for APK Apps
 * 
 * @package Arewa_APK_App
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class Arewa_APK_Export {
    
    /**
     * Export selected apps to ZIP file
     */
    public static function export_apps($app_ids) {
        global $wpdb;
        
        // Check if ZipArchive is available
        if (!class_exists('ZipArchive')) {
            return new WP_Error('zip_not_available', __('ZIP extension is not available on this server.', 'arewa-apk-apps-manager'));
        }
        
        $upload_dir = wp_upload_dir();
        $temp_dir = $upload_dir['basedir'] . '/arewa-apk-temp-' . time();
        
        // Create temp directory
        if (!wp_mkdir_p($temp_dir)) {
            return new WP_Error('temp_dir_failed', __('Failed to create temporary directory.', 'arewa-apk-apps-manager'));
        }
        
        $media_dir = $temp_dir . '/media';
        if (!wp_mkdir_p($media_dir)) {
            Arewa_APK_Media_Helper::delete_directory($temp_dir);
            return new WP_Error('media_dir_failed', __('Failed to create media directory.', 'arewa-apk-apps-manager'));
        }
        
        $apps_data = array();
        
        foreach ($app_ids as $app_id) {
            $app = get_post($app_id);
            if (!$app || $app->post_type !== 'arewa_apk_app') {
                continue;
            }
            
            $app_data = self::get_app_data($app, $media_dir);
            if ($app_data) {
                $apps_data[] = $app_data;
            }
        }
        
        // Prepare export data (NO plugin settings)
        $export_data = array(
            'version' => AREWA_APK_VERSION,
            'export_date' => current_time('mysql'),
            'export_type' => 'content_only',
            'site_url' => home_url(),
            'apps' => $apps_data,
            'total_apps' => count($apps_data),
        );
        
        // Create JSON file
        $json_data = wp_json_encode($export_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        file_put_contents($temp_dir . '/data.json', $json_data);
        
        // Create ZIP file
        $zip_filename = 'arewa-apk-apps-export-' . gmdate('Y-m-d-His') . '.zip';
        $zip_path = $upload_dir['basedir'] . '/' . $zip_filename;
        $zip_url = $upload_dir['baseurl'] . '/' . $zip_filename;
        
        $zip = new ZipArchive();
        if ($zip->open($zip_path, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) {
            Arewa_APK_Media_Helper::delete_directory($temp_dir);
            return new WP_Error('zip_create_failed', __('Failed to create ZIP file.', 'arewa-apk-apps-manager'));
        }
        
        // Add JSON file
        $zip->addFile($temp_dir . '/data.json', 'data.json');
        
        // Add all media files
        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($media_dir),
            RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ($files as $file) {
            if (!$file->isDir()) {
                $filePath = $file->getRealPath();
                $relativePath = substr($filePath, strlen($temp_dir) + 1);
                $zip->addFile($filePath, $relativePath);
            }
        }
        
        $zip->close();
        
        // Cleanup temp directory
        Arewa_APK_Media_Helper::delete_directory($temp_dir);
        
        return array(
            'file_url' => $zip_url,
            'filename' => $zip_filename,
            'total_apps' => count($apps_data)
        );
    }
    
    /**
     * Get app data for export
     */
    private static function get_app_data($app, $media_dir) {
        global $wpdb;
        
        // Get all app metadata
        $app_data = array(
            'id' => $app->ID,
            'title' => $app->post_title,
            'status' => $app->post_status,
            'slug' => $app->post_name,
            'excerpt' => $app->post_excerpt,
            'content' => $app->post_content,
            'date' => $app->post_date,
            'modified' => $app->post_modified,
            'version' => get_post_meta($app->ID, '_arewa_apk_version', true),
            'icon' => get_post_meta($app->ID, '_arewa_apk_icon', true),
            'feature_graphic' => get_post_meta($app->ID, '_arewa_apk_feature_graphic', true),
            'screenshots' => get_post_meta($app->ID, '_arewa_apk_screenshots', true),
            'apk_url' => get_post_meta($app->ID, '_arewa_apk_url', true),
            'short_description' => get_post_meta($app->ID, '_arewa_apk_short_description', true),
            'about_app' => get_post_meta($app->ID, '_arewa_apk_about', true),
            'comments_per_page' => get_post_meta($app->ID, '_arewa_apk_comments_per_page', true),
            'disable_seo' => get_post_meta($app->ID, '_arewa_apk_disable_seo', true),
            'download_count' => absint(get_post_meta($app->ID, '_arewa_apk_download_count', true) ?: 0),
        );
        
        // Get categories
        $categories = wp_get_object_terms($app->ID, 'arewa_apk_category', array('fields' => 'all'));
        if (!is_wp_error($categories)) {
            $app_data['categories'] = array();
            foreach ($categories as $cat) {
                $app_data['categories'][] = array(
                    'slug' => $cat->slug,
                    'name' => $cat->name,
                    'description' => $cat->description,
                );
            }
        }
        
        // Get tags
        $tags = wp_get_object_terms($app->ID, 'arewa_apk_tag', array('fields' => 'all'));
        if (!is_wp_error($tags)) {
            $app_data['tags'] = array();
            foreach ($tags as $tag) {
                $app_data['tags'][] = array(
                    'slug' => $tag->slug,
                    'name' => $tag->name,
                );
            }
        }
        
        // Get comments
        $comments_table = $wpdb->prefix . 'arewa_apk_comments';
        // Escape table name for security
        $comments_table_escaped = esc_sql($comments_table);
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        // Table names cannot use placeholders, but are escaped via esc_sql()
        $comments = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM `{$comments_table_escaped}` WHERE app_id = %d ORDER BY comment_date ASC",
            absint($app->ID)
        ));
        
        $app_data['comments'] = array();
        foreach ($comments as $comment) {
            $comment_data = array(
                'author_name' => $comment->author_name,
                'author_email' => $comment->author_email,
                'author_avatar' => $comment->author_avatar,
                'comment_text' => $comment->comment_text,
                'rating' => intval($comment->rating),
                'vote_helpful' => intval($comment->vote_helpful),
                'vote_not_helpful' => intval($comment->vote_not_helpful),
                'admin_reply_text' => $comment->admin_reply_text,
                'admin_reply_image' => $comment->admin_reply_image,
                'admin_reply_url' => $comment->admin_reply_url,
                'admin_reply_date' => $comment->admin_reply_date,
                'comment_date' => $comment->comment_date,
                'approved' => intval($comment->approved),
            );
            
            // Export avatar if exists
            if (!empty($comment->author_avatar)) {
                $avatar_path = Arewa_APK_Media_Helper::get_attachment_path_from_url($comment->author_avatar);
                if ($avatar_path && file_exists($avatar_path)) {
                    $avatar_filename = basename($avatar_path);
                    $new_avatar_path = $media_dir . '/avatar-' . $comment->id . '-' . $avatar_filename;
                    if (copy($avatar_path, $new_avatar_path)) {
                        $comment_data['author_avatar_file'] = 'media/avatar-' . $comment->id . '-' . $avatar_filename;
                    }
                }
            }
            
            // Export admin reply image if exists
            if (!empty($comment->admin_reply_image)) {
                $reply_image_path = Arewa_APK_Media_Helper::get_attachment_path_from_url($comment->admin_reply_image);
                if ($reply_image_path && file_exists($reply_image_path)) {
                    $reply_image_filename = basename($reply_image_path);
                    $new_reply_image_path = $media_dir . '/reply-' . $comment->id . '-' . $reply_image_filename;
                    if (copy($reply_image_path, $new_reply_image_path)) {
                        $comment_data['admin_reply_image_file'] = 'media/reply-' . $comment->id . '-' . $reply_image_filename;
                    }
                }
            }
            
            $app_data['comments'][] = $comment_data;
        }
        
        // Export media files
        $media_mapping = array();
        
        // Export icon
        if (!empty($app_data['icon'])) {
            $icon_path = Arewa_APK_Media_Helper::get_attachment_path_from_url($app_data['icon']);
            if ($icon_path && file_exists($icon_path)) {
                $icon_filename = 'icon-' . $app->ID . '-' . basename($icon_path);
                $new_icon_path = $media_dir . '/' . $icon_filename;
                if (copy($icon_path, $new_icon_path)) {
                    $media_mapping['icon'] = 'media/' . $icon_filename;
                }
            }
        }
        
        // Export feature graphic
        if (!empty($app_data['feature_graphic'])) {
            $fg_path = Arewa_APK_Media_Helper::get_attachment_path_from_url($app_data['feature_graphic']);
            if ($fg_path && file_exists($fg_path)) {
                $fg_filename = 'feature-' . $app->ID . '-' . basename($fg_path);
                $new_fg_path = $media_dir . '/' . $fg_filename;
                if (copy($fg_path, $new_fg_path)) {
                    $media_mapping['feature_graphic'] = 'media/' . $fg_filename;
                }
            }
        }
        
        // Export screenshots
        if (!empty($app_data['screenshots']) && is_array($app_data['screenshots'])) {
            $screenshots_mapping = array();
            foreach ($app_data['screenshots'] as $index => $screenshot_url) {
                $screenshot_path = Arewa_APK_Media_Helper::get_attachment_path_from_url($screenshot_url);
                if ($screenshot_path && file_exists($screenshot_path)) {
                    $screenshot_filename = 'screenshot-' . $app->ID . '-' . $index . '-' . basename($screenshot_path);
                    $new_screenshot_path = $media_dir . '/' . $screenshot_filename;
                    if (copy($screenshot_path, $new_screenshot_path)) {
                        $screenshots_mapping[] = 'media/' . $screenshot_filename;
                    }
                }
            }
            if (!empty($screenshots_mapping)) {
                $media_mapping['screenshots'] = $screenshots_mapping;
            }
        }
        
        $app_data['media_mapping'] = $media_mapping;
        
        return $app_data;
    }
}

