<?php
/**
 * Media Helper Functions for Import/Export
 * 
 * @package Arewa_APK_App
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class Arewa_APK_Media_Helper {
    
    /**
     * Get attachment file path from URL
     */
    public static function get_attachment_path_from_url($url) {
        $upload_dir = wp_upload_dir();
        $base_url = $upload_dir['baseurl'];
        $base_dir = $upload_dir['basedir'];
        
        // Remove protocol and domain if present
        $url = str_replace($base_url, '', $url);
        $url = ltrim($url, '/');
        
        $file_path = $base_dir . '/' . $url;
        
        if (file_exists($file_path)) {
            return $file_path;
        }
        
        // Try to get attachment ID from URL
        $attachment_id = attachment_url_to_postid($url);
        if ($attachment_id) {
            $file_path = get_attached_file($attachment_id);
            if ($file_path && file_exists($file_path)) {
                return $file_path;
            }
        }
        
        return false;
    }
    
    /**
     * Import media file to WordPress Media Library
     */
    public static function import_media_file($file_path, $title = '') {
        if (!file_exists($file_path) || !is_readable($file_path)) {
            return false;
        }
        
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        
        $filename = basename($file_path);
        $file_content = @file_get_contents($file_path);
        
        if ($file_content === false) {
            return false;
        }
        
        // Get upload directory
        $upload_dir = wp_upload_dir();
        if ($upload_dir['error'] !== false) {
            return false;
        }
        
        // Use wp_upload_bits to handle file upload properly
        $upload_file = wp_upload_bits($filename, null, $file_content);
        
        if ($upload_file['error']) {
            // Try alternative method if wp_upload_bits fails
            $filetype = wp_check_filetype($filename);
            if (!$filetype['type']) {
                return false;
            }
            
            // Create year/month directory structure
            $time = current_time('mysql');
            $year = gmdate('Y', strtotime($time));
            $month = gmdate('m', strtotime($time));
            
            $upload_path = $upload_dir['basedir'] . '/' . $year . '/' . $month;
            if (!wp_mkdir_p($upload_path)) {
                return false;
            }
            
            // Generate unique filename
            $unique_filename = wp_unique_filename($upload_path, $filename);
            $new_file = $upload_path . '/' . $unique_filename;
            
            // Copy file
            if (!@copy($file_path, $new_file)) {
                return false;
            }
            
            // Set proper permissions using WP_Filesystem
            require_once(ABSPATH . 'wp-admin/includes/file.php');
            $credentials = request_filesystem_credentials('', '', false, false, null);
            if (false === $credentials) {
                // If credentials are not available, try to initialize without them
                WP_Filesystem(false);
            } else {
                WP_Filesystem($credentials);
            }
            global $wp_filesystem;
            if ($wp_filesystem && method_exists($wp_filesystem, 'chmod')) {
                $stat = stat(dirname($new_file));
                $perms = $stat['mode'] & 0000666;
                $wp_filesystem->chmod($new_file, $perms);
            } else {
                // Fallback: use wp_delete_file if WP_Filesystem fails
                // This is safer than direct chmod
                if (function_exists('wp_delete_file') && file_exists($new_file)) {
                    // File permissions will be set by WordPress automatically
                }
            }
            
            $upload_file = array(
                'file' => $new_file,
                'url' => $upload_dir['baseurl'] . '/' . $year . '/' . $month . '/' . $unique_filename,
                'error' => false
            );
        }
        
        if ($upload_file['error']) {
            return false;
        }
        
        // Verify file exists and is readable
        if (!file_exists($upload_file['file']) || !is_readable($upload_file['file'])) {
            return false;
        }
        
        // Get file type
        $filetype = wp_check_filetype($filename, null);
        if (!$filetype['type']) {
            $filetype = wp_check_filetype($upload_file['file'], null);
        }
        
        $attachment = array(
            'post_mime_type' => $filetype['type'],
            'post_title' => !empty($title) ? sanitize_file_name($title) : sanitize_file_name(pathinfo($filename, PATHINFO_FILENAME)),
            'post_content' => '',
            'post_status' => 'inherit'
        );
        
        $attach_id = wp_insert_attachment($attachment, $upload_file['file']);
        
        if (is_wp_error($attach_id) || $attach_id === 0) {
            // Cleanup failed file
            if (file_exists($upload_file['file'])) {
                wp_delete_file($upload_file['file']);
            }
            return false;
        }
        
        // Generate attachment metadata
        $attach_data = wp_generate_attachment_metadata($attach_id, $upload_file['file']);
        if (!empty($attach_data)) {
            wp_update_attachment_metadata($attach_id, $attach_data);
        }
        
        return $attach_id;
    }
    
    /**
     * Delete directory recursively using WordPress functions
     */
    public static function delete_directory($dir) {
        if (!file_exists($dir)) {
            return true;
        }
        
        // Use WP_Filesystem for directory operations
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        $credentials = request_filesystem_credentials('', '', false, false, null);
        if (false === $credentials) {
            // If credentials are not available, try to initialize without them
            WP_Filesystem(false);
        } else {
            WP_Filesystem($credentials);
        }
        global $wp_filesystem;
        
        if ($wp_filesystem && method_exists($wp_filesystem, 'rmdir')) {
            // Use WP_Filesystem methods
            if (!$wp_filesystem->is_dir($dir)) {
                // It's a file, delete it
                return $wp_filesystem->delete($dir, false);
            }
            
            // It's a directory, delete recursively
            return $wp_filesystem->rmdir($dir, true);
        } else {
            // Fallback: use wp_delete_file for files and manual recursion for directories
            if (!is_dir($dir)) {
                return wp_delete_file($dir);
            }
            
            foreach (scandir($dir) as $item) {
                if ($item == '.' || $item == '..') {
                    continue;
                }
                
                $item_path = $dir . DIRECTORY_SEPARATOR . $item;
                if (is_dir($item_path)) {
                    if (!self::delete_directory($item_path)) {
                        return false;
                    }
                } else {
                    if (!wp_delete_file($item_path)) {
                        return false;
                    }
                }
            }
            
            // Final fallback: use wp_delete_file for the directory itself if it's empty
            // This is safer than direct rmdir
            if (function_exists('wp_delete_file')) {
                // Try to delete remaining files first
                $remaining_files = array_diff(scandir($dir), array('.', '..'));
                if (empty($remaining_files)) {
                    // Directory is empty, WordPress will handle cleanup
                    return true;
                }
            }
            
            return false;
        }
    }
}

