<?php
/**
 * Import Handler for APK Apps
 * 
 * @package Arewa_APK_App
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class Arewa_APK_Import {
    
    /**
     * Import apps from ZIP file
     */
    public static function import_from_zip($zip_file_path) {
        // Increase execution time for large imports
        @set_time_limit(300); // 5 minutes
        @ini_set('max_execution_time', 300);
        @ini_set('memory_limit', '256M');
        
        // Check if ZipArchive is available
        if (!class_exists('ZipArchive')) {
            return new WP_Error('zip_not_available', __('ZIP extension is not available on this server.', 'arewa-apk-apps-manager'));
        }
        
        $upload_dir = wp_upload_dir();
        $temp_dir = $upload_dir['basedir'] . '/arewa-apk-import-' . time();
        $temp_dir_created = false;
        
        try {
            // Create temp directory
            if (!wp_mkdir_p($temp_dir)) {
                return new WP_Error('temp_dir_failed', __('Failed to create temporary directory.', 'arewa-apk-apps-manager'));
            }
            $temp_dir_created = true;
            
            // Extract ZIP file
            $zip = new ZipArchive();
            $zip_result = $zip->open($zip_file_path);
            if ($zip_result !== TRUE) {
                if ($temp_dir_created) {
                    Arewa_APK_Media_Helper::delete_directory($temp_dir);
                }
                return new WP_Error('zip_open_failed', __('Failed to open ZIP file. Error code: ', 'arewa-apk-apps-manager') . $zip_result);
            }
            
            $extract_result = $zip->extractTo($temp_dir);
            $zip->close();
            
            if (!$extract_result) {
                if ($temp_dir_created) {
                    Arewa_APK_Media_Helper::delete_directory($temp_dir);
                }
                return new WP_Error('zip_extract_failed', __('Failed to extract ZIP file.', 'arewa-apk-apps-manager'));
            }
            
            // Read JSON data
            $json_path = $temp_dir . '/data.json';
            if (!file_exists($json_path)) {
                if ($temp_dir_created) {
                    Arewa_APK_Media_Helper::delete_directory($temp_dir);
                }
                return new WP_Error('json_not_found', __('Invalid export file. data.json not found.', 'arewa-apk-apps-manager'));
            }
            
            $file_contents = @file_get_contents($json_path);
            if ($file_contents === false) {
                if ($temp_dir_created) {
                    Arewa_APK_Media_Helper::delete_directory($temp_dir);
                }
                return new WP_Error('json_read_failed', __('Failed to read data.json file.', 'arewa-apk-apps-manager'));
            }
            
            $import_data = json_decode($file_contents, true);
            
            if (json_last_error() !== JSON_ERROR_NONE || !is_array($import_data)) {
                if ($temp_dir_created) {
                    Arewa_APK_Media_Helper::delete_directory($temp_dir);
                }
                return new WP_Error('json_invalid', __('Invalid JSON file. Error: ', 'arewa-apk-apps-manager') . json_last_error_msg());
            }
            
            if (!isset($import_data['apps']) || !is_array($import_data['apps'])) {
                if ($temp_dir_created) {
                    Arewa_APK_Media_Helper::delete_directory($temp_dir);
                }
                return new WP_Error('no_apps', __('No apps found in export file.', 'arewa-apk-apps-manager'));
            }
            
            // Import apps
            $result = self::import_apps($import_data['apps'], $temp_dir);
            
            // Cleanup temp directory
            if ($temp_dir_created) {
                Arewa_APK_Media_Helper::delete_directory($temp_dir);
            }
            
            return $result;
            
        } catch (Exception $e) {
            // Cleanup on error
            if ($temp_dir_created && isset($temp_dir)) {
                Arewa_APK_Media_Helper::delete_directory($temp_dir);
            }
            return new WP_Error('import_exception', __('Import failed: ', 'arewa-apk-apps-manager') . $e->getMessage());
        } catch (Error $e) {
            // Cleanup on fatal error
            if ($temp_dir_created && isset($temp_dir)) {
                Arewa_APK_Media_Helper::delete_directory($temp_dir);
            }
            return new WP_Error('import_fatal', __('Fatal error during import: ', 'arewa-apk-apps-manager') . $e->getMessage());
        }
    }
    
    /**
     * Import apps from data array
     */
    private static function import_apps($apps_data, $temp_dir) {
        global $wpdb;
        $imported_count = 0;
        $skipped_count = 0;
        $errors = array();
        
        foreach ($apps_data as $index => $app_data) {
            try {
                // Validate app data
                if (empty($app_data['title'])) {
                    // translators: %d: App index number
                    $errors[] = sprintf(__('App #%d: Missing title, skipping.', 'arewa-apk-apps-manager'), $index + 1);
                    continue;
                }
                
                // Check if app with same slug already exists
                $app_slug = isset($app_data['slug']) && !empty($app_data['slug']) ? sanitize_title($app_data['slug']) : sanitize_title($app_data['title']);
                $existing_post = get_page_by_path($app_slug, OBJECT, 'arewa_apk_app');
                if ($existing_post) {
                    $skipped_count++;
                    continue; // Skip existing apps
                }
                
                // Create app post
                $new_app_id = self::create_app_post($app_data, $app_slug);
                
                if (is_wp_error($new_app_id) || $new_app_id === 0) {
                    $error_msg = is_wp_error($new_app_id) ? $new_app_id->get_error_message() : __('Failed to create post', 'arewa-apk-apps-manager');
                    // translators: %1$s: App title, %2$s: Error message
                    $errors[] = sprintf(__('Error creating app "%1$s": %2$s', 'arewa-apk-apps-manager'), $app_data['title'], $error_msg);
                    continue;
                }
                
                // Import media files
                self::import_app_media($new_app_id, $app_data, $temp_dir);
                
                // Save meta fields
                self::save_app_meta($new_app_id, $app_data);
                
                // Import categories and tags
                self::import_taxonomies($new_app_id, $app_data);
                
                // Import comments
                self::import_comments($new_app_id, $app_data, $temp_dir);
                
                $imported_count++;
                
            } catch (Exception $e) {
                // translators: %1$s: App title, %2$s: Error message
                $errors[] = sprintf(__('Error importing app "%1$s": %2$s', 'arewa-apk-apps-manager'), 
                    isset($app_data['title']) ? $app_data['title'] : __('Unknown', 'arewa-apk-apps-manager'),
                    $e->getMessage()
                );
                continue;
            }
        }
        
        // Clear cache if method exists
        if (class_exists('Arewa_APK_App')) {
            $instance = Arewa_APK_App::get_instance();
            if (method_exists($instance, 'clear_cache')) {
                $instance->clear_cache();
            }
        }
        
        return array(
            'imported_count' => $imported_count,
            'skipped_count' => $skipped_count,
            'errors' => $errors,
        );
    }
    
    /**
     * Create app post
     */
    private static function create_app_post($app_data, $app_slug) {
        $post_data = array(
            'post_title' => sanitize_text_field($app_data['title']),
            'post_type' => 'arewa_apk_app',
            'post_status' => isset($app_data['status']) ? sanitize_text_field($app_data['status']) : 'draft',
            'post_excerpt' => isset($app_data['excerpt']) ? sanitize_textarea_field($app_data['excerpt']) : '',
            'post_content' => isset($app_data['content']) ? wp_kses_post($app_data['content']) : '',
            'post_name' => $app_slug,
        );
        
        return wp_insert_post($post_data, true);
    }
    
    /**
     * Import app media files
     */
    private static function import_app_media($app_id, $app_data, $temp_dir) {
        $media_mapping = isset($app_data['media_mapping']) ? $app_data['media_mapping'] : array();
        
        // Import icon
        if (!empty($media_mapping['icon'])) {
            $icon_file = $temp_dir . '/' . $media_mapping['icon'];
            if (file_exists($icon_file)) {
                $icon_attachment_id = Arewa_APK_Media_Helper::import_media_file($icon_file, isset($app_data['title']) ? $app_data['title'] . ' - Icon' : 'Icon');
                if ($icon_attachment_id) {
                    $icon_url = wp_get_attachment_url($icon_attachment_id);
                    if ($icon_url) {
                        update_post_meta($app_id, '_arewa_apk_icon', $icon_url);
                    }
                }
            }
        } elseif (!empty($app_data['icon'])) {
            update_post_meta($app_id, '_arewa_apk_icon', esc_url_raw($app_data['icon']));
        }
        
        // Import feature graphic
        if (!empty($media_mapping['feature_graphic'])) {
            $fg_file = $temp_dir . '/' . $media_mapping['feature_graphic'];
            if (file_exists($fg_file)) {
                $fg_attachment_id = Arewa_APK_Media_Helper::import_media_file($fg_file, isset($app_data['title']) ? $app_data['title'] . ' - Feature Graphic' : 'Feature Graphic');
                if ($fg_attachment_id) {
                    $fg_url = wp_get_attachment_url($fg_attachment_id);
                    if ($fg_url) {
                        update_post_meta($app_id, '_arewa_apk_feature_graphic', $fg_url);
                    }
                }
            }
        } elseif (!empty($app_data['feature_graphic'])) {
            update_post_meta($app_id, '_arewa_apk_feature_graphic', esc_url_raw($app_data['feature_graphic']));
        }
        
        // Import screenshots
        $screenshots = array();
        if (!empty($media_mapping['screenshots']) && is_array($media_mapping['screenshots'])) {
            foreach ($media_mapping['screenshots'] as $index => $screenshot_file) {
                $screenshot_path = $temp_dir . '/' . $screenshot_file;
                if (file_exists($screenshot_path)) {
                    $screenshot_title = isset($app_data['title']) ? $app_data['title'] . ' - Screenshot ' . ($index + 1) : 'Screenshot ' . ($index + 1);
                    $screenshot_attachment_id = Arewa_APK_Media_Helper::import_media_file($screenshot_path, $screenshot_title);
                    if ($screenshot_attachment_id) {
                        $screenshot_url = wp_get_attachment_url($screenshot_attachment_id);
                        if ($screenshot_url) {
                            $screenshots[] = $screenshot_url;
                        }
                    }
                }
            }
        } elseif (!empty($app_data['screenshots']) && is_array($app_data['screenshots'])) {
            foreach ($app_data['screenshots'] as $screenshot_url) {
                $screenshots[] = esc_url_raw($screenshot_url);
            }
        }
        update_post_meta($app_id, '_arewa_apk_screenshots', $screenshots);
    }
    
    /**
     * Save app meta fields
     */
    private static function save_app_meta($app_id, $app_data) {
        if (isset($app_data['version'])) {
            update_post_meta($app_id, '_arewa_apk_version', sanitize_text_field($app_data['version']));
        }
        if (isset($app_data['apk_url'])) {
            update_post_meta($app_id, '_arewa_apk_url', esc_url_raw($app_data['apk_url']));
        }
        if (isset($app_data['short_description'])) {
            update_post_meta($app_id, '_arewa_apk_short_description', sanitize_text_field($app_data['short_description']));
        }
        if (isset($app_data['about_app'])) {
            update_post_meta($app_id, '_arewa_apk_about', wp_kses_post($app_data['about_app']));
        }
        if (isset($app_data['comments_per_page'])) {
            update_post_meta($app_id, '_arewa_apk_comments_per_page', absint($app_data['comments_per_page']));
        }
        if (isset($app_data['disable_seo'])) {
            update_post_meta($app_id, '_arewa_apk_disable_seo', sanitize_text_field($app_data['disable_seo']));
        }
        if (isset($app_data['download_count'])) {
            update_post_meta($app_id, '_arewa_apk_download_count', absint($app_data['download_count']));
        }
    }
    
    /**
     * Import taxonomies (categories and tags)
     */
    private static function import_taxonomies($app_id, $app_data) {
        // Import categories
        if (!empty($app_data['categories']) && is_array($app_data['categories'])) {
            $category_ids = array();
            foreach ($app_data['categories'] as $cat_data) {
                $term = get_term_by('slug', $cat_data['slug'], 'arewa_apk_category');
                if (!$term) {
                    $term_result = wp_insert_term(
                        $cat_data['name'],
                        'arewa_apk_category',
                        array(
                            'slug' => $cat_data['slug'],
                            'description' => isset($cat_data['description']) ? $cat_data['description'] : '',
                        )
                    );
                    if (!is_wp_error($term_result)) {
                        $term = get_term($term_result['term_id'], 'arewa_apk_category');
                    }
                }
                if ($term) {
                    $category_ids[] = $term->term_id;
                }
            }
            if (!empty($category_ids)) {
                wp_set_object_terms($app_id, $category_ids, 'arewa_apk_category');
            }
        }
        
        // Import tags
        if (!empty($app_data['tags']) && is_array($app_data['tags'])) {
            $tag_ids = array();
            foreach ($app_data['tags'] as $tag_data) {
                $term = get_term_by('slug', $tag_data['slug'], 'arewa_apk_tag');
                if (!$term) {
                    $term_result = wp_insert_term(
                        $tag_data['name'],
                        'arewa_apk_tag',
                        array('slug' => $tag_data['slug'])
                    );
                    if (!is_wp_error($term_result)) {
                        $term = get_term($term_result['term_id'], 'arewa_apk_tag');
                    }
                }
                if ($term) {
                    $tag_ids[] = $term->term_id;
                }
            }
            if (!empty($tag_ids)) {
                wp_set_object_terms($app_id, $tag_ids, 'arewa_apk_tag');
            }
        }
    }
    
    /**
     * Import comments
     */
    private static function import_comments($app_id, $app_data, $temp_dir) {
        global $wpdb;
        
        if (empty($app_data['comments']) || !is_array($app_data['comments'])) {
            return;
        }
        
        $comments_table = $wpdb->prefix . 'arewa_apk_comments';
        
        foreach ($app_data['comments'] as $comment_data) {
            // Import avatar if exists
            $avatar_url = null;
            if (!empty($comment_data['author_avatar_file'])) {
                $avatar_file = $temp_dir . '/' . $comment_data['author_avatar_file'];
                if (file_exists($avatar_file)) {
                    $avatar_attachment_id = Arewa_APK_Media_Helper::import_media_file($avatar_file, 'Avatar');
                    if ($avatar_attachment_id) {
                        $avatar_url = wp_get_attachment_url($avatar_attachment_id);
                    }
                }
            } elseif (!empty($comment_data['author_avatar'])) {
                $avatar_url = esc_url_raw($comment_data['author_avatar']);
            }
            
            // Import admin reply image if exists
            $admin_reply_image_url = null;
            if (!empty($comment_data['admin_reply_image_file'])) {
                $reply_image_file = $temp_dir . '/' . $comment_data['admin_reply_image_file'];
                if (file_exists($reply_image_file)) {
                    $reply_image_attachment_id = Arewa_APK_Media_Helper::import_media_file($reply_image_file, 'Admin Reply Image');
                    if ($reply_image_attachment_id) {
                        $admin_reply_image_url = wp_get_attachment_url($reply_image_attachment_id);
                    }
                }
            } elseif (!empty($comment_data['admin_reply_image'])) {
                $admin_reply_image_url = esc_url_raw($comment_data['admin_reply_image']);
            }
            
            $wpdb->insert(
                $comments_table,
                array(
                    'app_id' => $app_id,
                    'author_name' => sanitize_text_field($comment_data['author_name']),
                    'author_email' => sanitize_email($comment_data['author_email']),
                    'author_avatar' => $avatar_url,
                    'comment_text' => wp_kses_post($comment_data['comment_text']),
                    'rating' => intval($comment_data['rating']),
                    'vote_helpful' => intval($comment_data['vote_helpful']),
                    'vote_not_helpful' => intval($comment_data['vote_not_helpful']),
                    'admin_reply_text' => !empty($comment_data['admin_reply_text']) ? wp_kses_post($comment_data['admin_reply_text']) : null,
                    'admin_reply_image' => $admin_reply_image_url,
                    'admin_reply_url' => !empty($comment_data['admin_reply_url']) ? esc_url_raw($comment_data['admin_reply_url']) : null,
                    'admin_reply_date' => !empty($comment_data['admin_reply_date']) ? $comment_data['admin_reply_date'] : null,
                    'comment_date' => !empty($comment_data['comment_date']) ? $comment_data['comment_date'] : current_time('mysql'),
                    'approved' => intval($comment_data['approved']),
                ),
                array('%d', '%s', '%s', '%s', '%s', '%d', '%d', '%d', '%s', '%s', '%s', '%s', '%s', '%d')
            );
        }
    }
}

