<?php
// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Determine if we're using Custom Post Type or legacy settings
$using_cpt = false;
$app_post = null;

// Initialize $app_id if not set
if (!isset($app_id)) {
    $app_id = 0;
}

if (!empty($app_id) && $app_id > 0) {
    // Using Custom Post Type
    $app_post = get_post($app_id);
    if ($app_post && $app_post->post_type === 'arewa_apk_app') {
        $using_cpt = true;
        
        // Get app data from post meta
        $app_name = $app_post->post_title;
        $app_version = get_post_meta($app_id, '_arewa_apk_version', true) ?: '1.0.0';
        $app_icon = get_post_meta($app_id, '_arewa_apk_icon', true);
        $app_screenshots = get_post_meta($app_id, '_arewa_apk_screenshots', true);
        if (!is_array($app_screenshots)) {
            $app_screenshots = array();
        }
        $apk_url = get_post_meta($app_id, '_arewa_apk_url', true);
        $about_app = get_post_meta($app_id, '_arewa_apk_about', true);
        // Get comments_per_page from app meta, fallback to global plugin settings
        $plugin_settings = Arewa_APK_App::get_plugin_settings();
        $app_comments_per_page = get_post_meta($app_id, '_arewa_apk_comments_per_page', true);
        $comments_per_page = !empty($app_comments_per_page) ? absint($app_comments_per_page) : absint($plugin_settings['comments_per_page'] ?? 10);
        $download_count = absint(get_post_meta($app_id, '_arewa_apk_download_count', true) ?: 0);
        $disable_seo = get_post_meta($app_id, '_arewa_apk_disable_seo', true) ?: 'no';
    } else {
        // Invalid app ID, fall back to legacy
        $app_id = 0;
    }
}

if (!$using_cpt) {
    // Using legacy settings (backward compatibility)
    $settings = Arewa_APK_App::get_settings();
    $app_name = $settings['app_name'] ?? 'My App';
    $app_version = $settings['app_version'] ?? '1.0.0';
    $app_icon = $settings['app_icon'] ?? '';
    // Handle both old single screenshot and new multiple screenshots
    $app_screenshots = $settings['app_screenshots'] ?? array();
    if (empty($app_screenshots) && !empty($settings['app_screenshot'])) {
        // Migrate old single screenshot to array
        $app_screenshots = array($settings['app_screenshot']);
    }
    if (!is_array($app_screenshots)) {
        $app_screenshots = array();
    }
    $apk_url = $settings['apk_url'] ?? '';
    $about_app = $settings['about_app'] ?? '';
    // Get comments_per_page from legacy settings, fallback to global plugin settings
    $plugin_settings = Arewa_APK_App::get_plugin_settings();
    $comments_per_page = !empty($settings['comments_per_page']) ? absint($settings['comments_per_page']) : absint($plugin_settings['comments_per_page'] ?? 10);
    $download_count = Arewa_APK_App::get_download_count();
    $disable_seo = $settings['disable_seo'] ?? 'no';
}

// Default to 'yes' if not specified
// $atts may not be defined if called directly, so check first
if (!isset($atts)) {
    $atts = array();
}
$show_icon = !isset($atts['show_icon']) || $atts['show_icon'] !== 'no';
$show_screenshot = !isset($atts['show_screenshot']) || $atts['show_screenshot'] !== 'no';
$show_version = !isset($atts['show_version']) || $atts['show_version'] !== 'no';
$show_comments = !isset($atts['show_comments']) || $atts['show_comments'] !== 'no';
?>

<div class="arewa-apk-app-container">
    <div class="arewa-apk-app-header">
        <?php if ($show_icon && $app_icon): ?>
            <div class="arewa-apk-icon">
                <img src="<?php echo esc_url($app_icon); ?>" alt="<?php echo esc_attr($app_name); ?>" />
            </div>
        <?php endif; ?>
        
        <div class="arewa-apk-info">
            <h2 class="arewa-apk-title"><?php echo esc_html($app_name); ?></h2>
            <?php if ($show_version): ?>
                <p class="arewa-apk-version"><?php
                // translators: %s: App version
                printf(esc_html__('Version: %s', 'arewa-apk-apps-manager'), esc_html($app_version)); ?></p>
            <?php endif; ?>
            
            <?php if ($apk_url): ?>
                <a href="<?php echo esc_url($apk_url); ?>" class="arewa-apk-download-btn" download data-apk-url="<?php echo esc_url($apk_url); ?>" data-app-id="<?php echo esc_attr($app_id); ?>">
                    <?php if ($app_icon): ?>
                        <img src="<?php echo esc_url($app_icon); ?>" alt="<?php echo esc_attr($app_name); ?>" class="arewa-download-icon" />
                    <?php else: ?>
                        <span class="dashicons dashicons-download"></span>
                    <?php endif; ?>
                    <span class="arewa-download-text"><?php esc_html_e('Download APK', 'arewa-apk-apps-manager'); ?></span>
                    <?php if ($download_count > 0): 
                        // Format download count in abbreviated form (K, M)
                        $formatted_count = $download_count;
                        if ($download_count >= 1000000) {
                            $formatted_count = number_format_i18n($download_count / 1000000, 1) . 'M';
                        } elseif ($download_count >= 1000) {
                            $formatted_count = number_format_i18n($download_count / 1000, 1) . 'K';
                        } else {
                            $formatted_count = number_format_i18n($download_count);
                        }
                        // Remove .0 from whole numbers
                        $formatted_count = str_replace('.0', '', $formatted_count);
                    ?>
                        <span class="arewa-download-count"><?php echo esc_html($formatted_count); ?> <?php esc_html_e('downloads', 'arewa-apk-apps-manager'); ?></span>
                    <?php endif; ?>
                </a>
            <?php endif; ?>
        </div>
    </div>
    
    <?php if ($show_screenshot && !empty($app_screenshots)): ?>
        <div class="arewa-apk-screenshots">
            <div class="arewa-screenshots-carousel" id="arewa-screenshots-carousel" data-total-slides="<?php echo esc_attr(count($app_screenshots)); ?>">
                <div class="arewa-screenshots-carousel-wrapper">
                    <?php foreach ($app_screenshots as $index => $screenshot): ?>
                        <div class="arewa-screenshot-slide" data-slide-index="<?php echo esc_attr($index); ?>">
                            <img src="<?php echo esc_url($screenshot); ?>" alt="<?php echo esc_attr($app_name) . ' ' . esc_attr__('Screenshot', 'arewa-apk-apps-manager') . ' ' . esc_attr($index + 1); ?>" data-full-image="<?php echo esc_url($screenshot); ?>" data-image-index="<?php echo esc_attr($index); ?>" />
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
            
            <!-- Lightbox Modal -->
            <div class="arewa-lightbox" id="arewa-lightbox">
                <span class="arewa-lightbox-close">&times;</span>
                <button class="arewa-lightbox-nav arewa-lightbox-prev" aria-label="<?php esc_attr_e('Previous', 'arewa-apk-apps-manager'); ?>">
                    <span class="dashicons dashicons-arrow-left-alt2"></span>
                </button>
                <div class="arewa-lightbox-content">
                    <img src="" alt="" id="arewa-lightbox-image" />
                </div>
                <button class="arewa-lightbox-nav arewa-lightbox-next" aria-label="<?php esc_attr_e('Next', 'arewa-apk-apps-manager'); ?>">
                    <span class="dashicons dashicons-arrow-right-alt2"></span>
                </button>
            </div>
            <?php if (count($app_screenshots) > 1): ?>
                <!-- Carousel is now draggable/swipeable - no arrows needed -->
            <?php endif; ?>
        </div>
    <?php endif; ?>
    
    <?php 
    // Check if we have about_app or categories to show
    $has_about = !empty($about_app);
    $has_categories = false;
    $categories = array();
    if ($using_cpt && $app_id > 0) {
        $categories = get_the_terms($app_id, 'arewa_apk_category');
        $has_categories = $categories && !is_wp_error($categories) && !empty($categories);
    }
    
    if ($has_about || $has_categories): ?>
        <div class="arewa-apk-about-categories-wrapper">
            <?php if ($has_about): ?>
                <div class="arewa-apk-about-section">
                    <h3 class="arewa-about-title"><?php esc_html_e('About this app', 'arewa-apk-apps-manager'); ?></h3>
                    <div class="arewa-about-content">
                        <?php 
                        // Process content to preserve line breaks and paragraphs
                        // wpautop converts double line breaks to <p> tags and single line breaks to <br>
                        echo wp_kses_post(wpautop($about_app)); 
                        ?>
                    </div>
                </div>
            <?php endif; ?>
            
            <?php if ($has_categories): ?>
                <div class="arewa-apk-categories-section">
                    <h3 class="arewa-categories-title"><?php esc_html_e('Categories', 'arewa-apk-apps-manager'); ?></h3>
                    <div class="arewa-apk-categories-list">
                        <?php 
                        $category_count = 0;
                        foreach ($categories as $category) : 
                            if ($category_count >= 5) break; // Limit to 5 categories
                        ?>
                            <a href="<?php echo esc_url(get_term_link($category)); ?>" class="arewa-apk-category-tag">
                                <?php echo esc_html($category->name); ?>
                            </a>
                        <?php 
                            $category_count++;
                        endforeach; 
                        ?>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    <?php endif; ?>
    
    <?php 
    // Check global plugin settings
    $plugin_settings = Arewa_APK_App::get_plugin_settings();
    $comments_enabled = $plugin_settings['enable_comments'] ?? 'yes';
    $allow_avatar_upload = $plugin_settings['allow_avatar_upload'] ?? 'yes';
    
    // Only show comments if enabled globally AND show_comments is true
    if ($show_comments && $comments_enabled === 'yes'): 
    ?>
        <div class="arewa-apk-comments-section">
            <div class="arewa-comments-header">
                <h3><?php esc_html_e('Comments & Reviews', 'arewa-apk-apps-manager'); ?></h3>
                <?php
                $total_comments = Arewa_APK_App::get_comments_count($app_id);
                $average_rating = Arewa_APK_App::get_average_rating($app_id);
                $rating_count = Arewa_APK_App::get_rating_count($app_id);
                ?>
                <div class="arewa-reviews-stats">
                    <?php if ($total_comments > 0): ?>
                        <span class="arewa-total-reviews">
                            <strong><?php echo esc_html(number_format_i18n($total_comments)); ?></strong>
                            <?php echo esc_html(_n('Review', 'Reviews', $total_comments, 'arewa-apk-apps-manager')); ?>
                        </span>
                    <?php endif; ?>
                    <?php if ($average_rating > 0 && $rating_count > 0): ?>
                        <span class="arewa-average-rating">
                            <strong><?php echo esc_html(number_format_i18n($average_rating, 1)); ?></strong>
                            <span class="arewa-rating-stars-inline">
                                <?php 
                                $full_stars = floor($average_rating);
                                $has_half = ($average_rating - $full_stars) >= 0.5;
                                for ($i = 1; $i <= 5; $i++): 
                                    if ($i <= $full_stars): ?>
                                        <span class="arewa-star arewa-star-filled" style="color: #00ff45;">★</span>
                                    <?php elseif ($i == $full_stars + 1 && $has_half): ?>
                                        <span class="arewa-star arewa-star-half" style="color: #00ff45;">★</span>
                                    <?php else: ?>
                                        <span class="arewa-star arewa-star-empty" style="color: #ddd;">☆</span>
                                    <?php endif;
                                endfor; ?>
                            </span>
                            <span class="arewa-rating-count">(<?php echo esc_html(number_format_i18n($rating_count)); ?>)</span>
                        </span>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="arewa-apk-comment-form">
                <h4><?php esc_html_e('Write a Review', 'arewa-apk-apps-manager'); ?></h4>
                <form id="arewa-apk-comment-form" enctype="multipart/form-data" data-app-id="<?php echo esc_attr($app_id); ?>">
                    <div class="arewa-form-group">
                        <label for="arewa-comment-author"><?php esc_html_e('Your Name', 'arewa-apk-apps-manager'); ?> <span class="required">*</span></label>
                        <input type="text" id="arewa-comment-author" name="author_name" required />
                    </div>
                    
                    <div class="arewa-form-group">
                        <label for="arewa-comment-email"><?php esc_html_e('Gmail Address', 'arewa-apk-apps-manager'); ?> <span class="required">*</span></label>
                        <input type="email" id="arewa-comment-email" name="author_email" required />
                        <small class="arewa-form-hint"><?php esc_html_e('Your email will not be displayed publicly.', 'arewa-apk-apps-manager'); ?></small>
                    </div>
                    
                    <?php if ($allow_avatar_upload === 'yes'): ?>
                    <div class="arewa-form-group">
                        <label for="arewa-comment-avatar"><?php esc_html_e('Profile Photo (Optional)', 'arewa-apk-apps-manager'); ?></label>
                        <div class="arewa-avatar-upload-container">
                            <input type="file" id="arewa-comment-avatar" name="author_avatar" accept="image/jpeg,image/jpg,image/png,image/gif,image/webp" />
                            <button type="button" class="arewa-avatar-upload-btn" id="arewa-avatar-upload-btn">
                                <span class="dashicons dashicons-camera"></span>
                                <span class="arewa-avatar-text"><?php esc_html_e('Choose Photo', 'arewa-apk-apps-manager'); ?></span>
                            </button>
                            <div class="arewa-avatar-preview" id="arewa-avatar-preview">
                                <img src="" alt="<?php esc_attr_e('Preview', 'arewa-apk-apps-manager'); ?>" id="arewa-avatar-preview-img" />
                                <button type="button" class="arewa-avatar-remove" id="arewa-avatar-remove" aria-label="<?php esc_attr_e('Remove photo', 'arewa-apk-apps-manager'); ?>">×</button>
                            </div>
                        </div>
                        <small class="arewa-form-hint"><?php esc_html_e('If no photo is uploaded, the first letter of your name will be displayed.', 'arewa-apk-apps-manager'); ?></small>
                    </div>
                    <?php endif; ?>
                    
                    <div class="arewa-form-group">
                        <label><?php esc_html_e('Rating', 'arewa-apk-apps-manager'); ?></label>
                        <div class="arewa-rating-input">
                            <input type="radio" id="rating-5" name="rating" value="5" checked="checked" />
                            <label for="rating-5" class="star-label" data-rating="5" aria-label="<?php esc_attr_e('5 stars', 'arewa-apk-apps-manager'); ?>"><span class="arewa-star-input">★</span></label>
                            
                            <input type="radio" id="rating-4" name="rating" value="4" />
                            <label for="rating-4" class="star-label" data-rating="4" aria-label="<?php esc_attr_e('4 stars', 'arewa-apk-apps-manager'); ?>"><span class="arewa-star-input">★</span></label>
                            
                            <input type="radio" id="rating-3" name="rating" value="3" />
                            <label for="rating-3" class="star-label" data-rating="3" aria-label="<?php esc_attr_e('3 stars', 'arewa-apk-apps-manager'); ?>"><span class="arewa-star-input">★</span></label>
                            
                            <input type="radio" id="rating-2" name="rating" value="2" />
                            <label for="rating-2" class="star-label" data-rating="2" aria-label="<?php esc_attr_e('2 stars', 'arewa-apk-apps-manager'); ?>"><span class="arewa-star-input">★</span></label>
                            
                            <input type="radio" id="rating-1" name="rating" value="1" />
                            <label for="rating-1" class="star-label" data-rating="1" aria-label="<?php esc_attr_e('1 star', 'arewa-apk-apps-manager'); ?>"><span class="arewa-star-input">★</span></label>
                            
                            <input type="radio" id="rating-0" name="rating" value="0" style="display: none;" />
                            <button type="button" class="arewa-clear-rating" style="margin-left: 10px; padding: 5px 10px; background: #f0f0f0; border: 1px solid #ddd; border-radius: 3px; cursor: pointer; font-size: 12px; color: #666;"><?php esc_html_e('Clear Rating', 'arewa-apk-apps-manager'); ?></button>
                        </div>
                        <small style="color: #999; display: block; margin-top: 5px;"><?php esc_html_e('Default: 5 stars. Click to change or clear.', 'arewa-apk-apps-manager'); ?></small>
                    </div>
                    
                    <div class="arewa-form-group">
                        <label for="arewa-comment-text"><?php esc_html_e('Your Comment', 'arewa-apk-apps-manager'); ?> <span class="required">*</span></label>
                        <textarea id="arewa-comment-text" name="comment_text" rows="5" required></textarea>
                    </div>
                    
                    <div class="arewa-form-group">
                        <button type="submit" class="arewa-submit-btn"><?php esc_html_e('Submit Comment', 'arewa-apk-apps-manager'); ?></button>
                    </div>
                    
                    <div class="arewa-form-message" id="arewa-comment-message"></div>
                </form>
            </div>
            
            <div class="arewa-apk-comments-list-wrapper">
                <div class="arewa-comments-filters">
                    <div class="arewa-filter-group">
                        <label for="arewa-sort-by"><?php esc_html_e('Sort by:', 'arewa-apk-apps-manager'); ?></label>
                        <?php
                        // Get current filter values from URL
                        $current_page = 1;
                        if (isset($_GET['comment_page'])) {
                            $page_value = absint($_GET['comment_page']);
                            if ($page_value > 0) {
                                $current_page = $page_value;
                            }
                        }
                        $sort_by = isset($_GET['sort_by']) ? sanitize_text_field($_GET['sort_by']) : 'date';
                        $order = isset($_GET['order']) ? sanitize_text_field($_GET['order']) : 'desc';
                        
                        // Validate sort_by and order
                        $allowed_sort_by = array('date', 'rating', 'votes', 'helpful', 'not_helpful');
                        $allowed_order = array('asc', 'desc');
                        if (!in_array($sort_by, $allowed_sort_by)) {
                            $sort_by = 'date';
                        }
                        if (!in_array($order, $allowed_order)) {
                            $order = 'desc';
                        }
                        ?>
                        <select id="arewa-sort-by" name="sort_by" class="arewa-filter-select">
                            <option value="date" data-order="desc" <?php selected($sort_by, 'date'); selected($order, 'desc'); ?>><?php esc_html_e('Newest First', 'arewa-apk-apps-manager'); ?></option>
                            <option value="date" data-order="asc" <?php selected($sort_by, 'date'); selected($order, 'asc'); ?>><?php esc_html_e('Oldest First', 'arewa-apk-apps-manager'); ?></option>
                            <option value="rating" data-order="desc" <?php selected($sort_by, 'rating'); selected($order, 'desc'); ?>><?php esc_html_e('Highest Rating', 'arewa-apk-apps-manager'); ?></option>
                            <option value="rating" data-order="asc" <?php selected($sort_by, 'rating'); selected($order, 'asc'); ?>><?php esc_html_e('Lowest Rating', 'arewa-apk-apps-manager'); ?></option>
                            <option value="votes" data-order="desc" <?php selected($sort_by, 'votes'); selected($order, 'desc'); ?>><?php esc_html_e('Most Votes', 'arewa-apk-apps-manager'); ?></option>
                            <option value="votes" data-order="asc" <?php selected($sort_by, 'votes'); selected($order, 'asc'); ?>><?php esc_html_e('Least Votes', 'arewa-apk-apps-manager'); ?></option>
                            <option value="helpful" data-order="desc" <?php selected($sort_by, 'helpful'); selected($order, 'desc'); ?>><?php esc_html_e('Most Helpful', 'arewa-apk-apps-manager'); ?></option>
                            <option value="not_helpful" data-order="desc" <?php selected($sort_by, 'not_helpful'); selected($order, 'desc'); ?>><?php esc_html_e('Most Not Helpful', 'arewa-apk-apps-manager'); ?></option>
                        </select>
                    </div>
                    <div class="arewa-filter-actions">
                        <button type="button" class="arewa-filter-btn" data-sort-by="date" data-order="desc" id="arewa-filter-reset">
                            <?php esc_html_e('Reset', 'arewa-apk-apps-manager'); ?>
                        </button>
                    </div>
                </div>
                
                <div class="arewa-comments-loading" id="arewa-comments-loading" style="display: none;">
                    <div class="arewa-loading-spinner"></div>
                    <p><?php esc_html_e('Loading comments...', 'arewa-apk-apps-manager'); ?></p>
                </div>
                
                <div class="arewa-apk-comments-list" id="arewa-comments-list">
                    <?php
                    
                    $comments = Arewa_APK_App::get_comments($current_page, $comments_per_page, $sort_by, $order, $app_id);
                    $total_comments = Arewa_APK_App::get_comments_count($app_id);
                    $total_pages = $comments_per_page > 0 ? ceil($total_comments / $comments_per_page) : 1;
                    
                    if (empty($comments)): ?>
                        <p class="arewa-no-comments"><?php esc_html_e('No comments yet. Be the first to review!', 'arewa-apk-apps-manager'); ?></p>
                    <?php else: ?>
                        <?php foreach ($comments as $comment): ?>
                        <div class="arewa-comment-item">
                            <div class="arewa-comment-header">
                                <div class="arewa-comment-author-info">
                                    <?php 
                                    // Get first letter of name for avatar fallback
                                    $first_letter = mb_substr(trim($comment->author_name), 0, 1);
                                    $first_letter = mb_strtoupper($first_letter);
                                    $author_avatar = !empty($comment->author_avatar) ? esc_url($comment->author_avatar) : '';
                                    
                                    // Generate color based on first letter for consistent colors
                                    $colors = array(
                                        'A' => '#667eea', 'B' => '#764ba2', 'C' => '#f093fb', 'D' => '#4facfe',
                                        'E' => '#43e97b', 'F' => '#fa709a', 'G' => '#fee140', 'H' => '#30cfd0',
                                        'I' => '#a8edea', 'J' => '#fed6e3', 'K' => '#ff9a9e', 'L' => '#fecfef',
                                        'M' => '#fecfef', 'N' => '#ffecd2', 'O' => '#fcb69f', 'P' => '#ff8a80',
                                        'Q' => '#ffab91', 'R' => '#ffccbc', 'S' => '#ffe0b2', 'T' => '#fff9c4',
                                        'U' => '#c5e1a5', 'V' => '#a5d6a7', 'W' => '#80cbc4', 'X' => '#b2dfdb',
                                        'Y' => '#b3e5fc', 'Z' => '#81d4fa'
                                    );
                                    $avatar_color = isset($colors[$first_letter]) ? $colors[$first_letter] : '#667eea';
                                    ?>
                                    <div class="arewa-comment-avatar" <?php if (!$author_avatar): ?>style="background: <?php echo esc_attr($avatar_color); ?>;"<?php endif; ?>>
                                        <?php if ($author_avatar): ?>
                                            <img src="<?php echo esc_url($author_avatar); ?>" alt="<?php echo esc_attr($comment->author_name); ?>" />
                                        <?php else: ?>
                                            <div class="arewa-comment-avatar-initial"><?php echo esc_html($first_letter); ?></div>
                                        <?php endif; ?>
                                    </div>
                                    <div class="arewa-comment-author">
                                        <strong><?php echo esc_html($comment->author_name); ?></strong>
                                        <?php 
                                        // Hide email for security - show only first letter and domain
                                        $email_parts = explode('@', $comment->author_email);
                                        if (count($email_parts) === 2) {
                                            $email_local = $email_parts[0];
                                            $email_domain = $email_parts[1];
                                            $hidden_email = mb_substr($email_local, 0, 1) . '***@' . $email_domain;
                                        } else {
                                            $hidden_email = '***@' . (isset($email_parts[1]) ? $email_parts[1] : 'gmail.com');
                                        }
                                        ?>
                                        <span class="arewa-comment-email"><?php echo esc_html($hidden_email); ?></span>
                                    </div>
                                </div>
                                <?php if ($comment->rating > 0): ?>
                                    <div class="arewa-comment-rating">
                                        <?php for ($i = 1; $i <= 5; $i++): ?>
                                            <?php if ($i <= $comment->rating): ?>
                                                <span class="arewa-star arewa-star-filled" aria-label="<?php
                                                // translators: %d: Star rating number (1-5)
                                                echo esc_attr(sprintf(__('%d out of 5 stars', 'arewa-apk-apps-manager'), $i));
                                                ?>">★</span>
                                            <?php else: ?>
                                                <span class="arewa-star arewa-star-empty" aria-label="<?php
                                                // translators: %d: Star rating number (1-5)
                                                echo esc_attr(sprintf(__('%d out of 5 stars', 'arewa-apk-apps-manager'), $i));
                                                ?>">☆</span>
                                            <?php endif; ?>
                                        <?php endfor; ?>
                                        <span class="arewa-rating-text"><?php echo esc_html($comment->rating); ?>/5</span>
                                    </div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="arewa-comment-date">
                                <?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($comment->comment_date))); ?>
                            </div>
                            
                            <div class="arewa-comment-text">
                                <?php echo nl2br(esc_html($comment->comment_text)); ?>
                            </div>
                            
                            <?php if (!empty($comment->admin_reply_text) || !empty($comment->admin_reply_image) || !empty($comment->admin_reply_url)): ?>
                                <div class="arewa-admin-reply">
                                    <div class="arewa-admin-reply-header">
                                        <span class="arewa-admin-reply-badge"><?php esc_html_e('Admin Reply', 'arewa-apk-apps-manager'); ?></span>
                                        <?php if (!empty($comment->admin_reply_date)): ?>
                                            <span class="arewa-admin-reply-date"><?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($comment->admin_reply_date))); ?></span>
                                        <?php endif; ?>
                                    </div>
                                    <div class="arewa-admin-reply-content">
                                        <?php if (!empty($comment->admin_reply_text)): ?>
                                            <div class="arewa-admin-reply-text">
                                                <?php echo nl2br(esc_html($comment->admin_reply_text)); ?>
                                            </div>
                                        <?php endif; ?>
                                        
                                        <?php if (!empty($comment->admin_reply_image)): ?>
                                            <div class="arewa-admin-reply-image">
                                                <img src="<?php echo esc_url($comment->admin_reply_image); ?>" alt="<?php esc_attr_e('Admin reply image', 'arewa-apk-apps-manager'); ?>" />
                                            </div>
                                        <?php endif; ?>
                                        
                                        <?php if (!empty($comment->admin_reply_url)): ?>
                                            <div class="arewa-admin-reply-url">
                                                <a href="<?php echo esc_url($comment->admin_reply_url); ?>" target="_blank" rel="noopener noreferrer" class="arewa-admin-reply-link">
                                                    <span class="dashicons dashicons-external"></span>
                                                    <?php echo esc_html($comment->admin_reply_url); ?>
                                                </a>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php endif; ?>
                            
                            <div class="arewa-comment-votes">
                                <button type="button" class="arewa-vote-btn arewa-vote-helpful" data-comment-id="<?php echo esc_attr($comment->id); ?>" data-vote-type="helpful">
                                    <span class="arewa-vote-icon">👍</span>
                                    <span class="vote-count"><?php echo absint($comment->vote_helpful); ?></span>
                                    <span class="vote-text"><?php esc_html_e('Helpful', 'arewa-apk-apps-manager'); ?></span>
                                </button>
                                
                                <button type="button" class="arewa-vote-btn arewa-vote-not-helpful" data-comment-id="<?php echo esc_attr($comment->id); ?>" data-vote-type="not_helpful">
                                    <span class="arewa-vote-icon">👎</span>
                                    <span class="vote-count"><?php echo absint($comment->vote_not_helpful); ?></span>
                                    <span class="vote-text"><?php esc_html_e('Not Helpful', 'arewa-apk-apps-manager'); ?></span>
                                </button>
                            </div>
                        </div>
                    <?php endforeach; ?>
                    
                    <?php if ($total_pages > 1): ?>
                        <div class="arewa-comments-pagination" id="arewa-comments-pagination">
                            <?php
                            // Build base URL - only include comment_page, not sort_by/order in URL
                            $current_url = remove_query_arg(array('comment_page', 'sort_by', 'order'));
                            $page_args = array(
                                'base' => add_query_arg('comment_page', '%#%', $current_url),
                                'format' => '',
                                'prev_text' => '&laquo; ' . __('Previous', 'arewa-apk-apps-manager'),
                                'next_text' => __('Next', 'arewa-apk-apps-manager') . ' &raquo;',
                                'total' => $total_pages,
                                'current' => $current_page,
                                'type' => 'list'
                            );
                            $page_links = paginate_links($page_args);
                            if ($page_links) {
                                echo wp_kses_post($page_links);
                            }
                            ?>
                        </div>
                    <?php endif; ?>
                <?php endif; ?>
                </div>
            </div>
        </div>
    <?php endif; ?>
</div>

