/**
 * Admin JavaScript
 * Mobile Bottom Navigation Settings
 *
 * @package Arewa_Mobile_Bottom_Navigation
 */

(function($) {
	'use strict';

	$( document ).ready(
		function() {
			initTabs();
			initSortable();
			initSubmenuManagement();
			initVisibilityToggle();
			initButtonCountChange();
			initSubmenuHeader();
			initMediaUploader();
			initMultiSelect();
			initUserAgentToggle();

			// Initialize color pickers immediately and with retry mechanism.
			initColorPickers();
			setTimeout( initColorPickers, 100 );
			setTimeout( initColorPickers, 300 );
			setTimeout( initColorPickers, 600 );
			setTimeout( initColorPickers, 1000 );
		}
	);

	/**
	 * Initialize tabs
	 */
	function initTabs() {
		$( '.arewa-mbn-tabs .nav-tab' ).on(
			'click',
			function(e) {
				e.preventDefault();

				const target = $( this ).attr( 'href' );

				// Update active tab.
				$( '.arewa-mbn-tabs .nav-tab' ).removeClass( 'nav-tab-active' );
				$( this ).addClass( 'nav-tab-active' );

				// Update active content.
				$( '.arewa-mbn-tabs .tab-content' ).removeClass( 'active' );
				$( target ).addClass( 'active' );

				// Reinitialize color pickers in the new tab.
				setTimeout(
					function() {
						if ( typeof $.fn.wpColorPicker !== 'undefined' ) {
							$( target ).find( '.color-picker' ).each(
								function() {
									// Only initialize if not already initialized.
									const $input = $( this );
									if ( ! $input.hasClass( 'wp-color-picker' ) && ! $input.closest( '.wp-picker-container' ).length ) {
										try {
											initSingleColorPicker( $input );
										} catch (e) {
											console.error( 'Error initializing color picker in tab:', e );
										}
									}
								}
							);
						}
					},
					100
				);
			}
		);
	}

	/**
	 * Get defaultColor for wpColorPicker based on input name and current value.
	 * WordPress Iris requires a valid hex; "transparent" or empty would break the swatch.
	 *
	 * @param {jQuery} $input Color input element.
	 * @return {string} Hex color for the picker.
	 */
	function getColorPickerDefault( $input ) {
		const val = $input.val() ? $input.val().trim() : '';
		// If we have a valid hex color, use it as the default.
		if ( val && val !== 'transparent' && /^#[0-9A-Fa-f]{3,6}$/.test( val ) ) {
			// Normalize 3-digit hex to 6-digit.
			if ( val.length === 4 ) {
				return '#' + val[1] + val[1] + val[2] + val[2] + val[3] + val[3];
			}
			return val;
		}
		// Otherwise, return default based on field name.
		const name = ( $input.attr( 'name' ) || '' );
		if ( name.indexOf( 'bg_color' ) !== -1 ) {
			return '#ffffff';
		}
		if ( name.indexOf( 'active_color' ) !== -1 ) {
			return '#0073aa';
		}
		return '#333333';
	}

	/**
	 * Set color picker swatch (wp-color-result) to a given hex.
	 *
	 * @param {jQuery} $input The color input (inside .wp-picker-container).
	 * @param {string} hex    Hex color e.g. #333333.
	 */
	function setColorPickerSwatch( $input, hex ) {
		const $container = $input.closest( '.wp-picker-container' );
		if ( $container.length ) {
			$container.find( '.wp-color-result' ).css( 'background-color', hex || '#ffffff' );
		}
	}

	/**
	 * Refresh all color picker swatches to match their input values.
	 * Fixes Iris bug where changing one picker can corrupt other pickers' display.
	 */
	function refreshAllColorPickerSwatches() {
		$( '#arewa-mbn-buttons-form .color-picker' ).each(
			function() {
				const $input     = $( this );
				const $container = $input.closest( '.wp-picker-container' );
				if ( ! $container.length ) {
					return;
				}
				let hex = ( $input.val() || '' ).trim();
				if ( $input.attr( 'data-arewa-bg-transparent' ) === '1' ) {
					hex = '#ffffff';
				}
				if ( hex ) {
					$container.find( '.wp-color-result' ).css( 'background-color', hex );
				}
			}
		);
	}

	/**
	 * Initialize a single color picker input with proper defaultColor so swatch displays.
	 *
	 * @param {jQuery} $input Color input element.
	 */
	function initSingleColorPicker( $input ) {
		const defaultColor = getColorPickerDefault( $input );
		const name         = ( $input.attr( 'name' ) || '' );
		const isBgColor    = name.indexOf( 'bg_color' ) !== -1;
		// Unique ID to prevent multiple wpColorPicker instances from sharing state (known wpColorPicker bug).
		if ( ! $input.attr( 'id' ) ) {
			$input.attr( 'id', 'arewa-mbn-color-' + ( name.replace( /[^a-zA-Z0-9]/g, '_' ) || 'picker' ) + '-' + Math.random().toString( 36 ).slice( 2, 9 ) );
		}
		// So the swatch displays when stored value is transparent/empty, use a display hex and mark for submit.
		if ( isBgColor ) {
			const val = $input.val() ? $input.val().trim() : '';
			if ( val === '' || val === 'transparent' ) {
				$input.val( defaultColor );
				$input.attr( 'data-arewa-bg-transparent', '1' );
			}
		}
		$input.wpColorPicker(
			{
				defaultColor: defaultColor,
				change: function( event, ui ) {
					if ( ! ui || ! ui.color ) {
						return;
					}
					const hex = ui.color.toString();
					// Update only this instance's input (closure). Do not use Iris state or event.target; multiple pickers share state.
					$input.val( hex );
					$input.removeAttr( 'data-arewa-bg-transparent' );
					setColorPickerSwatch( $input, hex );
					refreshAllColorPickerSwatches();
				},
				clear: function() {
					// Restore our plugin defaults: icon/text #333333, bg transparent, active #0073aa.
					if ( isBgColor ) {
						$input.val( '' );
						$input.attr( 'data-arewa-bg-transparent', '1' );
						setColorPickerSwatch( $input, '#ffffff' );
					} else {
						$input.val( defaultColor );
						setColorPickerSwatch( $input, defaultColor );
					}
				}
			}
		);
		// When user clicks "Default" (wp-picker-default) or "Clear" (wp-picker-clear), ensure value is our default.
		const $container = $input.closest( '.wp-picker-container' );
		$container.find( '.wp-picker-default, .wp-picker-clear' ).on(
			'click',
			function() {
				setTimeout(
					function() {
						if ( isBgColor ) {
							$input.val( '' );
							$input.attr( 'data-arewa-bg-transparent', '1' );
							setColorPickerSwatch( $input, '#ffffff' );
						} else {
							$input.val( defaultColor );
							setColorPickerSwatch( $input, defaultColor );
						}
						refreshAllColorPickerSwatches();
					},
					0
				);
			}
		);
	}

	// When user clicks outside to close any picker, refresh all swatches (fixes cross-picker display bug).
	$( document ).on(
		'click.arewaMbnColorPicker',
		'body',
		function() {
			setTimeout( refreshAllColorPickerSwatches, 80 );
		}
	);

	/**
	 * Initialize color pickers
	 */
	function initColorPickers() {
		// Check if jQuery and wpColorPicker are available.
		if ( typeof $ === 'undefined' || typeof $.fn.wpColorPicker === 'undefined' ) {
			console.log( 'Arewa MBN: wpColorPicker not ready yet, will retry...' );
			return;
		}

		console.log( 'Arewa MBN: Initializing color pickers...' );

		// Find all color picker inputs.
		$( '.color-picker' ).each(
			function() {
				const $input = $( this );

				// Check if already initialized.
				if ( $input.hasClass( 'wp-color-picker' ) || $input.closest( '.wp-picker-container' ).length > 0 ) {
					return;
				}

				try {
					initSingleColorPicker( $input );
					console.log( 'Arewa MBN: Successfully initialized color picker for', $input.attr( 'name' ) );
				} catch (e) {
					console.error( 'Arewa MBN: Error initializing color picker for', $input.attr( 'name' ), e );
				}
			}
		);
		if ( $( '#arewa-mbn-buttons-form' ).length ) {
			refreshAllColorPickerSwatches();
		}
	}

	/**
	 * Initialize sortable buttons
	 */
	function initSortable() {
		$( '#arewa-mbn-buttons-container' ).sortable(
			{
				handle: '.button-handle',
				placeholder: 'ui-sortable-placeholder',
				axis: 'y',
				tolerance: 'pointer',
				update: function(event, ui) {
					// Update order values.
					$( '#arewa-mbn-buttons-container .arewa-mbn-button-item' ).each(
						function(index) {
							$( this ).find( '.button-order' ).val( index + 1 );
						}
					);
				}
			}
		);
	}

	/**
	 * Initialize submenu management
	 */
	function initSubmenuManagement() {
		// Add submenu item.
		$( document ).on(
			'click',
			'.add-submenu-item',
			function() {
				const buttonId  = $( this ).data( 'button-id' );
				const container = $( this ).closest( '.submenu-items-container' );
				const itemCount = container.find( '.submenu-item' ).length;

				const newItem                      = `
				< div class                        = "submenu-item" >
					< table class                  = "form-table" >
						< tr >
							< th > < label > Label < / label > < / th >
							< td >
								< input type       = "text" name = "arewa_mbn_submenus[${buttonId}][${itemCount}][label]" value = "" class = "regular-text" >
							< / td >
						< / tr >
						< tr >
							< th > < label > SVG Icon Code < / label > < / th >
							< td >
								< textarea name    = "arewa_mbn_submenus[${buttonId}][${itemCount}][icon]" rows = "5" class = "large-text code" placeholder = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M10 20v-6h4v6h5v-8h3L12 3 2 12h3v8z"/></svg>' > < / textarea >
								< p class          = "description" > Paste your SVG code here.< / p >
							< / td >
						< / tr >
						< tr >
							< th > < label > Icon Color < / label > < / th >
							< td >
								< input type       = "text" name = "arewa_mbn_submenus[${buttonId}][${itemCount}][icon_color]" value = "#333333" class = "color-picker" >
							< / td >
						< / tr >
						< tr >
							< th > < label > URL < / label > < / th >
							< td >
								< input type       = "url" name = "arewa_mbn_submenus[${buttonId}][${itemCount}][url]" value = "" class = "regular-text" >
							< / td >
						< / tr >
						< tr >
							< th > < label > Link Target < / label > < / th >
							< td >
								< select name      = "arewa_mbn_submenus[${buttonId}][${itemCount}][target]" >
									< option value = "_self" > Same Tab < / option >
									< option value = "_blank" > New Tab() < / option >
								< / select >
							< / td >
						< / tr >
					< / table >
					< button type                  = "button" class = "button remove-submenu-item" > Remove Item < / button >
				< / div >
				`;

				$( this ).before( newItem );

				// Initialize color picker for new item.
				setTimeout(
					function() {
						if ( typeof $.fn.wpColorPicker !== 'undefined' ) {
								$( container ).find( '.submenu-item:last .color-picker' ).each(
									function() {
										const $input = $( this );
										if ( ! $input.hasClass( 'wp-color-picker' ) && ! $input.closest( '.wp-picker-container' ).length ) {
											try {
												initSingleColorPicker( $input );
											} catch (e) {
												console.error( 'Error initializing color picker for submenu item:', e );
											}
										}
									}
								);
						}
					},
					100
				);
			}
		);

		// Remove submenu item.
		$( document ).on(
			'click',
			'.remove-submenu-item',
			function() {
				if (confirm( 'Are you sure you want to remove this submenu item?' )) {
					$( this ).closest( '.submenu-item' ).remove();
				}
			}
		);

		// Update submenu sections and URL fields when has_submenu select changes.
		$( document ).on(
			'change',
			'.has-submenu-select',
			function() {
				const buttonItem = $( this ).closest( '.arewa-mbn-button-item' );
				const buttonId   = buttonItem.data( 'button-id' );
				const isEnabled  = $( this ).val() === 'enable';

				// Show/hide URL and Target fields.
				const urlRow    = buttonItem.find( '.url-row' );
				const targetRow = buttonItem.find( '.target-row' );

				if (isEnabled) {
					urlRow.hide();
					targetRow.hide();
				} else {
					urlRow.show();
					targetRow.show();
				}

				// Show/hide submenu section.
				const submenuSection = $( `.arewa - mbn - submenu - section[data - button - id = "${buttonId}"]` );
				if (isEnabled) {
					if (submenuSection.length === 0) {
						// Create submenu section if it doesn't exist.
						createSubmenuSection( buttonId );
					} else {
						submenuSection.show();
					}
				} else {
					submenuSection.hide();
				}
			}
		);

		// Initialize visibility on page load.
		$( '.has-submenu-select' ).each(
			function() {
				if ($( this ).val() === 'enable') {
					const buttonItem = $( this ).closest( '.arewa-mbn-button-item' );
					buttonItem.find( '.url-row' ).hide();
					buttonItem.find( '.target-row' ).hide();
				}
			}
		);
	}

	/**
	 * Create submenu section
	 */
	function createSubmenuSection(buttonId) {
		const buttonItem  = $( `.arewa - mbn - button - item[data - button - id = "${buttonId}"]` );
		const buttonLabel = buttonItem.find( 'input[name*="[label]"]' ).val() || 'Button';

		const submenuSection = `
			< div class           = "arewa-mbn-submenu-section" data - button - id = "${buttonId}" >
				< h3 > ${buttonLabel} < / h3 >
				< div class       = "submenu-items-container" >
					< button type = "button" class = "button add-submenu-item" data - button - id = "${buttonId}" > Add Submenu Item < / button >
				< / div >
			< / div >
		`;

		$( '#submenus .tab-content' ).append( submenuSection );
	}

	/**
	 * Initialize visibility toggle
	 */
	function initVisibilityToggle() {
		// Handle multiple display mode checkboxes.
		$( document ).on(
			'change',
			'input[name="arewa_mbn_settings[visibility_mode][]"]',
			function() {
				const clickedValue = $( this ).val();
				const isChecked    = $( this ).is( ':checked' );

				// If "all" is checked, uncheck all others.
				if (clickedValue === 'all' && isChecked) {
					$( 'input[name="arewa_mbn_settings[visibility_mode][]"]' ).not( '#arewa_mbn_visibility_mode_all' ).prop( 'checked', false );
				}

				// If any other option is checked, uncheck "all".
				if (clickedValue !== 'all' && isChecked) {
					$( '#arewa_mbn_visibility_mode_all' ).prop( 'checked', false );
				}

				updateVisibilityRows();
			}
		);

		// Initial update on page load.
		updateVisibilityRows();
	}

	/**
	 * Update visibility rows based on selected modes
	 */
	function updateVisibilityRows() {
		// Get all checked modes.
		const checkedModes = [];
		$( 'input[name="arewa_mbn_settings[visibility_mode][]"]:checked' ).each(
			function() {
				checkedModes.push( $( this ).val() );
			}
		);

		// Handle "all" mode - if checked, hide all specific rows.
		if (checkedModes.includes( 'all' )) {
			$( '.visibility-pages-row, .visibility-posts-row, .visibility-categories-row, .visibility-post-types-row' ).hide();
			return;
		}

		// Show/hide rows based on selected modes.
		if (checkedModes.includes( 'pages' )) {
			$( '.visibility-pages-row' ).show();
		} else {
			$( '.visibility-pages-row' ).hide();
		}

		if (checkedModes.includes( 'posts' )) {
			$( '.visibility-posts-row' ).show();
		} else {
			$( '.visibility-posts-row' ).hide();
		}

		if (checkedModes.includes( 'categories' )) {
			$( '.visibility-categories-row' ).show();
		} else {
			$( '.visibility-categories-row' ).hide();
		}

		if (checkedModes.includes( 'post_types' )) {
			$( '.visibility-post-types-row' ).show();
		} else {
			$( '.visibility-post-types-row' ).hide();
		}
	}

	/**
	 * Initialize multi-select functionality
	 */
	function initMultiSelect() {
		// Search functionality.
		$( document ).on(
			'input',
			'.arewa-mbn-search-input',
			function() {
				const searchTerm = $( this ).val().toLowerCase();
				const target     = $( this ).data( 'target' );
				const container  = $( this ).closest( '.arewa-mbn-multi-select-container' ).find( '.arewa-mbn-checkbox-container[data-type="' + target + '"]' );

				container.find( '.arewa-mbn-checkbox-item' ).each(
					function() {
						const text = $( this ).find( 'span' ).text().toLowerCase();
						if (text.indexOf( searchTerm ) !== -1) {
							$( this ).removeClass( 'arewa-mbn-hidden' );
						} else {
							$( this ).addClass( 'arewa-mbn-hidden' );
						}
					}
				);
			}
		);

		// Select All functionality.
		$( document ).on(
			'click',
			'.arewa-mbn-select-all',
			function(e) {
				e.preventDefault();
				const target    = $( this ).data( 'target' );
				const container = $( this ).closest( '.arewa-mbn-multi-select-container' ).find( '.arewa-mbn-checkbox-container[data-type="' + target + '"]' );

				container.find( '.arewa-mbn-checkbox-item:not(.arewa-mbn-hidden) input[type="checkbox"]' ).prop( 'checked', true );
			}
		);

		// Deselect All functionality.
		$( document ).on(
			'click',
			'.arewa-mbn-deselect-all',
			function(e) {
				e.preventDefault();
				const target    = $( this ).data( 'target' );
				const container = $( this ).closest( '.arewa-mbn-multi-select-container' ).find( '.arewa-mbn-checkbox-container[data-type="' + target + '"]' );

				container.find( '.arewa-mbn-checkbox-item:not(.arewa-mbn-hidden) input[type="checkbox"]' ).prop( 'checked', false );
			}
		);
	}

	/**
	 * Handle button count change
	 */
	function initButtonCountChange() {
		$( '#arewa_mbn_button_count' ).on(
			'change',
			function() {
				const newCount       = parseInt( $( this ).val() );
				const currentButtons = $( '#arewa-mbn-buttons-container .arewa-mbn-button-item' ).length;

				if (newCount > currentButtons) {
					// Add new buttons.
					for (let i = currentButtons; i < newCount; i++) {
						addNewButton( i + 1 );
					}
				} else if (newCount < currentButtons) {
					// Remove excess buttons.
					$( '#arewa-mbn-buttons-container .arewa-mbn-button-item' ).slice( newCount ).remove();
				}
			}
		);
	}

	/**
	 * Add new button (template is provided by PHP via arewaMBNAdmin.newButtonTemplate).
	 */
	function addNewButton(index) {
		const buttonId = 'btn_' + index;
		const idx      = index - 1;
		const tpl      = typeof arewaMBNAdmin !== 'undefined' && arewaMBNAdmin.newButtonTemplate
			? arewaMBNAdmin.newButtonTemplate
			: '';
		if ( ! tpl ) {
			return;
		}
		const newButton = tpl
			.replace( /\{\{buttonId\}\}/g, buttonId )
			.replace( /\{\{index\}\}/g, String( index ) )
			.replace( /\{\{idx\}\}/g, String( idx ) );

		$( '#arewa-mbn-buttons-container' ).append( newButton );

		// Button form uses native type="color" inputs (no Iris) so no init needed.

		// Reinitialize sortable.
		$( '#arewa-mbn-buttons-container' ).sortable( 'refresh' );
	}

	// Handle form submission for buttons form.
	// Do NOT use preventDefault - we only sync values before submit. Native submit must run so the
	// submit button is included in POST (programmatic form.submit() omits it).
	$( '#arewa-mbn-buttons-form' ).on(
		'submit',
		function() {
			// Update all button orders before submit.
			$( '#arewa-mbn-buttons-container .arewa-mbn-button-item' ).each(
				function(index) {
					$( this ).find( '.button-order' ).val( index + 1 );
				}
			);
			// Button form uses native type="color" and .arewa-mbn-bg-color-value; no Iris sync. bg_color empty = transparent.
		}
	);

	// Native color inputs in button form: sync background color picker to text input; Transparent button clears it.
	$( document ).on(
		'input change',
		'.arewa-mbn-bg-color-picker',
		function() {
			const target = $( this ).data( 'target' );
			if ( target ) {
				$( target ).val( $( this ).val() );
			}
		}
	);
	$( document ).on(
		'click',
		'.arewa-mbn-bg-transparent',
		function() {
			const target = $( this ).data( 'target' );
			if ( target ) {
				$( target ).val( '' );
				$( this ).siblings( '.arewa-mbn-bg-color-picker' ).val( '#ffffff' );
			}
		}
	);

	/**
	 * Initialize submenu header toggle
	 */
	function initSubmenuHeader() {
		// Use event delegation to ensure it works even if elements are added dynamically.
		$( document ).on(
			'change',
			'#arewa_mbn_submenu_header_enable',
			function() {
				const isEnabled = $( this ).val() === 'enable';
				$( '.submenu-header-options' ).toggle( isEnabled );
			}
		);

		// Initialize on page load.
		if ($( '#arewa_mbn_submenu_header_enable' ).length) {
			const isEnabled = $( '#arewa_mbn_submenu_header_enable' ).val() === 'enable';
			$( '.submenu-header-options' ).toggle( isEnabled );
		}

		// Character counter for description.
		$( '#arewa_mbn_submenu_header_description' ).on(
			'input',
			function() {
				const length    = $( this ).val().length;
				const maxLength = 150;
				$( '#arewa_mbn_description_counter' ).text( length + '/' + maxLength );

				if (length > maxLength) {
					$( this ).val( $( this ).val().substring( 0, maxLength ) );
					$( '#arewa_mbn_description_counter' ).text( maxLength + '/' + maxLength );
				}
			}
		);
	}

	/**
	 * Initialize WordPress media uploader
	 */
	function initMediaUploader() {
		// Upload logo button.
		$( document ).on(
			'click',
			'.arewa-mbn-upload-logo',
			function(e) {
				e.preventDefault();

				// Check if wp.media is available.
				if (typeof wp === 'undefined' || typeof wp.media === 'undefined') {
					alert( 'WordPress media library is not available. Please refresh the page.' );
					return;
				}

				const targetInput      = $( this ).data( 'target' );
				const targetIdInput    = $( this ).data( 'id-target' );
				const previewContainer = $( this ).closest( 'td' ).find( '.submenu-logo-preview' );
				const button           = $( this );

				// Create the media uploader (create new instance each time to avoid conflicts).
				const mediaUploader = wp.media(
					{
						title: 'Choose Logo',
						button: {
							text: 'Use this logo'
						},
						multiple: false,
						library: {
							type: 'image'
						}
					}
				);

				// When an image is selected, run a callback.
				mediaUploader.on(
					'select',
					function() {
						const attachment = mediaUploader.state().get( 'selection' ).first().toJSON();

						// Set the URL and ID.
						$( '#' + targetInput ).val( attachment.url );
						if (targetIdInput) {
							$( '#' + targetIdInput ).val( attachment.id );
						}

						// Show preview.
						if (previewContainer.length) {
							previewContainer.html( '<img src="' + attachment.url + '" alt="Logo Preview" style="max-width: 150px; max-height: 60px; display: block; margin-bottom: 10px;">' );
						}

						// Show remove button.
						button.closest( 'td' ).find( '.arewa-mbn-remove-logo' ).show();
					}
				);

				// Open the uploader.
				mediaUploader.open();
			}
		);

		// Remove logo button.
		$( document ).on(
			'click',
			'.arewa-mbn-remove-logo',
			function(e) {
				e.preventDefault();

				const targetInput      = $( this ).data( 'target' );
				const targetIdInput    = $( this ).data( 'id-target' );
				const previewContainer = $( this ).closest( 'td' ).find( '.submenu-logo-preview' );

				// Clear inputs.
				$( '#' + targetInput ).val( '' );
				$( '#' + targetIdInput ).val( '' );

				// Clear preview.
				previewContainer.html( '' );

				// Hide remove button.
				$( this ).hide();
			}
		);
	}

	/**
	 * Initialize User-Agent settings toggle
	 */
	function initUserAgentToggle() {
		// Handle User-Agent enabled/disabled toggle.
		$( document ).on(
			'change',
			'#arewa_mbn_user_agent_enabled',
			function() {
				const isEnabled = $( this ).val() === 'enable';
				$( '.user-agent-settings-row' ).toggle( isEnabled );
			}
		);

		// Initialize on page load.
		if ( $( '#arewa_mbn_user_agent_enabled' ).length ) {
			const isEnabled = $( '#arewa_mbn_user_agent_enabled' ).val() === 'enable';
			$( '.user-agent-settings-row' ).toggle( isEnabled );
		}
	}

})( jQuery );
