/**
 * Frontend JavaScript
 * Mobile Bottom Navigation
 *
 * @package Arewa_Mobile_Bottom_Navigation
 */

(function() {
	'use strict';

	// Wait for DOM to be ready.
	if (document.readyState === 'loading') {
		document.addEventListener( 'DOMContentLoaded', init );
	} else {
		init();
	}

	function init() {
		const navigation = document.getElementById( 'arewa-mbn-navigation' );
		if ( ! navigation) {
			return;
		}

		initSubmenus();
		initActiveStates();
		initButtonClicks();
		initTouchSupport();
	}

	/**
	 * Initialize submenu functionality
	 */
	function initSubmenus() {
		const triggers = document.querySelectorAll( '.arewa-mbn-submenu-trigger' );
		const overlay  = createOverlay();

		triggers.forEach(
			function(trigger) {
				trigger.addEventListener(
					'click',
					function(e) {
						e.preventDefault();
						e.stopPropagation();

						const buttonId = this.getAttribute( 'data-button-id' );
						const submenu  = document.querySelector( '.arewa-mbn-submenu[data-button-id="' + buttonId + '"]' );

						if ( ! submenu) {
							return;
						}

						// Check if current submenu is already open (not closing).
						const isOpen = submenu.classList.contains( 'arewa-mbn-open' ) && ! submenu.classList.contains( 'arewa-mbn-closing' );

						if (isOpen) {
							// If open, just close it.
							closeSubmenu( submenu, overlay );
						} else {
							// If closed or closing, close all other submenus first, then open this one.
							closeAllSubmenus();
							// Small delay to ensure closing animation completes.
							setTimeout(
								function() {
									openSubmenu( submenu, overlay );
								},
								50
							);
						}
					}
				);
			}
		);

		// Close submenu when clicking overlay.
		overlay.addEventListener(
			'click',
			function() {
				closeAllSubmenus();
			}
		);

		// Close submenu when clicking outside (but not on the trigger button itself).
		document.addEventListener(
			'click',
			function(e) {
				// Don't close if clicking on the trigger button (it handles its own toggle).
				if (e.target.closest( '.arewa-mbn-submenu-trigger' )) {
					return;
				}
				// Close if clicking outside the submenu area.
				if ( ! e.target.closest( '.arewa-mbn-button-wrapper.arewa-mbn-has-submenu' ) && ! e.target.closest( '.arewa-mbn-submenu' )) {
					closeAllSubmenus();
				}
			}
		);

		// Close submenu on escape key.
		document.addEventListener(
			'keydown',
			function(e) {
				if (e.key === 'Escape') {
					closeAllSubmenus();
				}
			}
		);

		// Initialize submenu item animations.
		initSubmenuItemAnimations();
	}

	/**
	 * Initialize submenu item animations
	 */
	function initSubmenuItemAnimations() {
		// Use event delegation for dynamically loaded items.
		document.addEventListener(
			'touchstart',
			function(e) {
				const item = e.target.closest( '.arewa-mbn-submenu-item' );
				if (item) {
					item.classList.add( 'touch-active' );
				}
			},
			{ passive: true }
		);

		document.addEventListener(
			'touchend',
			function(e) {
				const item = e.target.closest( '.arewa-mbn-submenu-item' );
				if (item) {
					setTimeout(
						function() {
							item.classList.remove( 'touch-active' );
						},
						150
					);
				}
			},
			{ passive: true }
		);

		document.addEventListener(
			'touchcancel',
			function(e) {
				const item = e.target.closest( '.arewa-mbn-submenu-item' );
				if (item) {
					item.classList.remove( 'touch-active' );
				}
			},
			{ passive: true }
		);
	}

	/**
	 * Create overlay element
	 */
	function createOverlay() {
		let overlay = document.querySelector( '.arewa-mbn-overlay' );
		if ( ! overlay) {
			overlay           = document.createElement( 'div' );
			overlay.className = 'arewa-mbn-overlay';
			document.body.appendChild( overlay );
		}
		return overlay;
	}

	/**
	 * Open submenu
	 */
	function openSubmenu(submenu, overlay) {
		// Remove any inline styles that might interfere.
		submenu.style.visibility = '';
		submenu.classList.remove( 'arewa-mbn-closing' );
		submenu.classList.add( 'arewa-mbn-open' );

		// Add class to button wrapper for CSS targeting.
		const buttonId = submenu.getAttribute( 'data-button-id' );
		if (buttonId) {
			const trigger = document.querySelector( '.arewa-mbn-submenu-trigger[data-button-id="' + buttonId + '"]' );
			if (trigger) {
				const buttonWrapper = trigger.closest( '.arewa-mbn-button-wrapper' );
				if (buttonWrapper) {
					buttonWrapper.classList.add( 'arewa-mbn-submenu-open' );
				}
			}
		}
		overlay.classList.add( 'arewa-mbn-active' );

		// Prevent body scroll when submenu is open.
		document.body.style.overflow = 'hidden';
	}

	/**
	 * Close submenu with impressive animation
	 */
	function closeSubmenu(submenu, overlay) {
		// Add closing class for animation.
		submenu.classList.add( 'arewa-mbn-closing' );
		submenu.classList.remove( 'arewa-mbn-open' );

		// Remove class from button wrapper.
		const buttonId = submenu.getAttribute( 'data-button-id' );
		if (buttonId) {
			const trigger = document.querySelector( '.arewa-mbn-submenu-trigger[data-button-id="' + buttonId + '"]' );
			if (trigger) {
				const buttonWrapper = trigger.closest( '.arewa-mbn-button-wrapper' );
				if (buttonWrapper) {
					buttonWrapper.classList.remove( 'arewa-mbn-submenu-open' );
				}
			}
		}

		// Fade out overlay.
		overlay.classList.remove( 'arewa-mbn-active' );

		// Remove closing class and hide after animation.
		setTimeout(
			function() {
				submenu.classList.remove( 'arewa-mbn-closing' );
				submenu.style.visibility = 'hidden';
			},
			500
		); // Match animation duration.

		// Restore body scroll.
		document.body.style.overflow = '';
	}

	/**
	 * Close all submenus
	 */
	function closeAllSubmenus() {
		const openSubmenus = document.querySelectorAll( '.arewa-mbn-submenu.arewa-mbn-open' );
		const overlay      = document.querySelector( '.arewa-mbn-overlay' );

		openSubmenus.forEach(
			function(submenu) {
				// Use closeSubmenu for proper animation.
				closeSubmenu( submenu, overlay );
			}
		);

		if (overlay) {
			overlay.classList.remove( 'arewa-mbn-active' );
		}

		// Restore body scroll.
		document.body.style.overflow = '';
	}

	/**
	 * Initialize active states
	 */
	function initActiveStates() {
		const buttons      = document.querySelectorAll( '.arewa-mbn-button' );
		const currentUrl   = window.location.href.replace( /\/$/, '' );
		const currentPath  = window.location.pathname.replace( /\/$/, '' ) || '/';
		const homeUrl      = window.location.origin;
		const homeUrlClean = homeUrl.replace( /\/$/, '' );

		// Remove all active states first.
		buttons.forEach(
			function(button) {
				button.classList.remove( 'arewa-mbn-active' );
				const wrapper = button.closest( '.arewa-mbn-button-wrapper' );
				if (wrapper) {
					wrapper.classList.remove( 'arewa-mbn-active' );
				}
			}
		);

		// Find and activate only the matching button.
		let activeButton    = null;
		let bestMatch       = null;
		let bestMatchLength = 0;

		buttons.forEach(
			function(button) {
				if (button.tagName === 'A') {
					const buttonUrl = button.getAttribute( 'href' );
					if (buttonUrl && buttonUrl !== '#') {
						let cleanButtonUrl = buttonUrl.replace( /\/$/, '' );

						// Handle relative URLs.
						if (cleanButtonUrl.startsWith( '/' )) {
							cleanButtonUrl = homeUrlClean + cleanButtonUrl;
						} else if ( ! cleanButtonUrl.startsWith( 'http' )) {
							cleanButtonUrl = homeUrlClean + '/' + cleanButtonUrl;
						}

						cleanButtonUrl = cleanButtonUrl.replace( /\/$/, '' );

						// Check for homepage.
						if (cleanButtonUrl === homeUrlClean || cleanButtonUrl === homeUrl || buttonUrl === '/' || buttonUrl === homeUrl) {
							// Only activate if we're on the homepage.
							if (currentPath === '/' || currentUrl === homeUrlClean || currentUrl === homeUrl) {
								if ( ! activeButton) {
									activeButton = button;
								}
							}
						} else if (currentUrl === cleanButtonUrl) { // Exact match.
							if ( ! activeButton || bestMatchLength < cleanButtonUrl.length) {
								activeButton    = button;
								bestMatch       = cleanButtonUrl;
								bestMatchLength = cleanButtonUrl.length;
							}
						} else { // Path match (for parent pages).
							const buttonPath = new URL( cleanButtonUrl ).pathname.replace( /\/$/, '' ) || '/';
							if (buttonPath !== '/' && currentPath.startsWith( buttonPath )) {
								if ( ! activeButton || bestMatchLength < buttonPath.length) {
									activeButton    = button;
									bestMatch       = buttonPath;
									bestMatchLength = buttonPath.length;
								}
							}
						}
					}
				}
			}
		);

		// Activate only the best match.
		if (activeButton) {
			activeButton.classList.add( 'arewa-mbn-active' );
			const activeWrapper = activeButton.closest( '.arewa-mbn-button-wrapper' );
			if (activeWrapper) {
				activeWrapper.classList.add( 'arewa-mbn-active' );
			}
		}
	}

	/**
	 * Initialize button click handlers for active state
	 */
	let buttonClicksInitialized = false;
	function initButtonClicks() {
		if (buttonClicksInitialized) {
			return; // Already initialized.
		}

		// Only handle regular link buttons, not submenu triggers.
		const buttons = document.querySelectorAll( '.arewa-mbn-button:not(.arewa-mbn-submenu-trigger)' );

		buttons.forEach(
			function(button) {
				if (button.tagName === 'A' && ! button.hasAttribute( 'data-click-handler' )) {
					button.setAttribute( 'data-click-handler', 'true' );

					// Add click listener.
					button.addEventListener(
						'click',
						function(e) {
							// Remove active from all buttons and wrappers.
							const allButtons  = document.querySelectorAll( '.arewa-mbn-button' );
							const allWrappers = document.querySelectorAll( '.arewa-mbn-button-wrapper' );

							allButtons.forEach(
								function(btn) {
									btn.classList.remove( 'arewa-mbn-active' );
								}
							);

							allWrappers.forEach(
								function(wrapper) {
									wrapper.classList.remove( 'arewa-mbn-active' );
								}
							);

							// Add active to clicked button and its wrapper.
							this.classList.add( 'arewa-mbn-active' );
							const wrapper = this.closest( '.arewa-mbn-button-wrapper' );
							if (wrapper) {
								wrapper.classList.add( 'arewa-mbn-active' );
							}
						}
					);
				}
			}
		);

		buttonClicksInitialized = true;
	}

	/**
	 * Initialize touch support
	 */
	function initTouchSupport() {
		const navigation = document.getElementById( 'arewa-mbn-navigation' );
		if ( ! navigation) {
			return;
		}

		// Add touch class for CSS targeting.
		if ('ontouchstart' in window || navigator.maxTouchPoints > 0) {
			navigation.classList.add( 'arewa-mbn-touch' );
		}

		// Prevent double-tap zoom on buttons.
		let lastTouchEnd = 0;
		navigation.addEventListener(
			'touchend',
			function(e) {
				const now = Date.now();
				if (now - lastTouchEnd <= 300) {
					e.preventDefault();
				}
				lastTouchEnd = now;
			},
			false
		);
	}

	// Handle window resize.
	let resizeTimer;
	window.addEventListener(
		'resize',
		function() {
			clearTimeout( resizeTimer );
			resizeTimer = setTimeout(
				function() {
					closeAllSubmenus();
				},
				250
			);
		}
	);

	// Handle scroll (optional: hide/show navigation on scroll).
	let lastScrollTop = 0;
	const navigation  = document.getElementById( 'arewa-mbn-navigation' );

	if (navigation) {
		window.addEventListener(
			'scroll',
			function() {
				const scrollTop = window.pageYOffset || document.documentElement.scrollTop;

				// Optional: Hide navigation when scrolling down, show when scrolling up.
				// Uncomment if desired.
				/*
				if (scrollTop > lastScrollTop && scrollTop > 100) {
				navigation.style.transform = 'translateY(100%)';
				} else {
				navigation.style.transform = 'translateY(0)';
				}
				*/

				lastScrollTop = scrollTop <= 0 ? 0 : scrollTop;
			},
			false
		);
	}
})();
