<?php
/**
 * Admin Settings
 *
 * @package Arewa_Mobile_Bottom_Navigation
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin Settings Class
 */
class Arewa_MBN_Admin_Settings {

	/**
	 * Constructor
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		// Use load-{hook} to check permissions before WordPress blocks access.
		// The hook name for a toplevel menu page is: load-{hook} where hook is toplevel_page_{menu_slug}.
		add_action( 'load-toplevel_page_arewa-mobile-bottom-navigation-bar', array( $this, 'check_page_permissions' ) );
		// Process form submissions early, before WordPress auto-save.
		add_action( 'admin_init', array( $this, 'process_form_submissions' ), 1 );
		// Prevent WordPress auto-save after our methods run.
		add_filter( 'pre_update_option_arewa_mbn_settings', array( $this, 'prevent_auto_save_after_manual' ), 20, 2 );
	}

	/**
	 * Check page permissions early to prevent WordPress from blocking access
	 */
	public function check_page_permissions() {
		// Check user capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Sorry, you are not allowed to access this page.', 'arewa-mobile-bottom-navigation-bar' ) );
		}
	}

	/**
	 * Process form submissions before WordPress auto-save
	 */
	public function process_form_submissions() {
		// Only process on our settings page.
		if ( ! isset( $_GET['page'] ) || 'arewa-mobile-bottom-navigation-bar' !== sanitize_text_field( wp_unslash( $_GET['page'] ) ) ) {
			return;
		}

		// Check user capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Sorry, you are not allowed to access this page.', 'arewa-mobile-bottom-navigation-bar' ) );
		}

		// Check if form was submitted.
		if ( ! isset( $_POST['option_page'] ) || 'arewa_mbn_settings_group' !== sanitize_text_field( wp_unslash( $_POST['option_page'] ) ) ) {
			return;
		}

		// Verify nonce.
		if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'arewa_mbn_settings_group-options' ) ) {
			return;
		}

		// Get current tab (sanitized and validated against whitelist).
		$request_tab  = isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : '';
		$allowed_tabs = array( 'general', 'buttons', 'submenus', 'styling', 'visibility' );
		$current_tab  = in_array( $request_tab, $allowed_tabs, true ) ? $request_tab : 'general';

		// Process form submission with our manual methods.
		if ( 'general' === $current_tab && isset( $_POST['arewa_mbn_save_general'] ) ) {
			$this->save_general_settings();
			$this->form_processed = true;
			// Redirect immediately to prevent WordPress from processing.
			wp_safe_redirect(
				add_query_arg(
					array(
						'tab'              => $current_tab,
						'settings-updated' => 'true',
					),
					admin_url( 'admin.php?page=arewa-mobile-bottom-navigation-bar' )
				)
			);
			exit;
		} elseif ( 'buttons' === $current_tab && isset( $_POST['arewa_mbn_save_buttons'] ) ) {
			$this->save_buttons_settings();
			$this->form_processed = true;
			wp_safe_redirect(
				add_query_arg(
					array(
						'tab'              => $current_tab,
						'settings-updated' => 'true',
					),
					admin_url( 'admin.php?page=arewa-mobile-bottom-navigation-bar' )
				)
			);
			exit;
		} elseif ( 'submenus' === $current_tab && isset( $_POST['arewa_mbn_save_submenus'] ) ) {
			$this->save_submenus_settings();
			$this->form_processed = true;
			wp_safe_redirect(
				add_query_arg(
					array(
						'tab'              => $current_tab,
						'settings-updated' => 'true',
					),
					admin_url( 'admin.php?page=arewa-mobile-bottom-navigation-bar' )
				)
			);
			exit;
		} elseif ( 'styling' === $current_tab && isset( $_POST['arewa_mbn_save_styling'] ) ) {
			$this->save_styling_settings();
			$this->form_processed = true;
			wp_safe_redirect(
				add_query_arg(
					array(
						'tab'              => $current_tab,
						'settings-updated' => 'true',
					),
					admin_url( 'admin.php?page=arewa-mobile-bottom-navigation-bar' )
				)
			);
			exit;
		} elseif ( 'visibility' === $current_tab && isset( $_POST['arewa_mbn_save_visibility'] ) ) {
			$this->save_visibility_settings();
			$this->form_processed = true;
			wp_safe_redirect(
				add_query_arg(
					array(
						'tab'              => $current_tab,
						'settings-updated' => 'true',
					),
					admin_url( 'admin.php?page=arewa-mobile-bottom-navigation-bar' )
				)
			);
			exit;
		}
	}

	/**
	 * Prevent WordPress auto-save after our manual methods have run
	 *
	 * @param mixed $value     The new option value.
	 * @param mixed $old_value The old option value.
	 * @return mixed The option value to save.
	 */
	public function prevent_auto_save_after_manual( $value, $old_value ) {
		// If we already processed the form manually, prevent WordPress from overwriting.
		if ( isset( $this->form_processed ) && true === $this->form_processed ) {
			// Return old value to prevent WordPress from saving (we already saved manually).
			return $old_value;
		}
		return $value;
	}

	/**
	 * Add admin menu
	 */
	public function add_admin_menu() {
		// Main menu only - all settings in one page with horizontal tabs.
		add_menu_page(
			__( 'Mobile Bottom Navigation', 'arewa-mobile-bottom-navigation-bar' ),
			__( 'Mobile Bottom', 'arewa-mobile-bottom-navigation-bar' ),
			'manage_options',
			'arewa-mobile-bottom-navigation-bar',
			array( $this, 'render_settings_page' ),
			'dashicons-smartphone',
			30
		);
	}

	/**
	 * Register settings
	 */
	public function register_settings() {
		register_setting( 'arewa_mbn_settings_group', 'arewa_mbn_settings', array( $this, 'sanitize_settings' ) );
		register_setting( 'arewa_mbn_settings_group', 'arewa_mbn_buttons', array( $this, 'sanitize_buttons' ) );
		register_setting( 'arewa_mbn_settings_group', 'arewa_mbn_submenus', array( $this, 'sanitize_submenus' ) );
	}


	/**
	 * Sanitize settings
	 *
	 * @param array $input The input array to sanitize.
	 * @return array The sanitized settings array.
	 */
	public function sanitize_settings( $input ) {
		// Get current settings to preserve values not in the input.
		$current_settings = get_option( 'arewa_mbn_settings', array() );
		$sanitized        = $current_settings; // Start with current settings to preserve all existing values.

		// Only update fields that are present in the input.
		if ( ! is_array( $input ) ) {
			return $sanitized;
		}

		// Convert enable/disable to 1/0 for enabled.
		if ( isset( $input['enabled'] ) ) {
			$enabled_value        = sanitize_text_field( $input['enabled'] );
			$sanitized['enabled'] = ( 'enable' === $enabled_value ) ? '1' : '0';
		}
		if ( isset( $input['button_count'] ) ) {
			$sanitized['button_count'] = absint( $input['button_count'] );
			$sanitized['button_count'] = max( 2, min( 6, $sanitized['button_count'] ) );
		}
		if ( isset( $input['layout_style'] ) ) {
			$layout_style              = sanitize_text_field( $input['layout_style'] );
			$allowed_layouts           = array( 'icon_only', 'icon_text', 'icon_text_active' );
			$default_layout            = isset( $current_settings['layout_style'] ) ? $current_settings['layout_style'] : 'icon_text';
			$sanitized['layout_style'] = in_array( $layout_style, $allowed_layouts, true ) ? $layout_style : $default_layout;
		}
		if ( isset( $input['background_color'] ) ) {
			$bg_color                      = sanitize_hex_color( $input['background_color'] );
			$default_bg                    = isset( $current_settings['background_color'] ) ? $current_settings['background_color'] : '#ffffff';
			$sanitized['background_color'] = ( false !== $bg_color ) ? $bg_color : $default_bg;
		}
		if ( isset( $input['border_radius'] ) ) {
			$sanitized['border_radius'] = absint( $input['border_radius'] );
		}
		// Convert enable/disable to 1/0 for box_shadow.
		if ( isset( $input['box_shadow'] ) ) {
			$box_shadow_value        = sanitize_text_field( $input['box_shadow'] );
			$sanitized['box_shadow'] = ( 'enable' === $box_shadow_value ) ? '1' : '0';
		} else {
			// Default to enable if not set.
			$sanitized['box_shadow'] = isset( $current_settings['box_shadow'] ) ? $current_settings['box_shadow'] : '1';
		}
		if ( isset( $input['active_color'] ) ) {
			$active_color              = sanitize_hex_color( $input['active_color'] );
			$default_active            = isset( $current_settings['active_color'] ) ? $current_settings['active_color'] : '#0073aa';
			$sanitized['active_color'] = ( false !== $active_color ) ? $active_color : $default_active;
		}
		// Convert enable/disable to 1/0 for hover_effect.
		if ( isset( $input['hover_effect'] ) ) {
			$hover_effect_value        = sanitize_text_field( $input['hover_effect'] );
			$sanitized['hover_effect'] = ( 'enable' === $hover_effect_value ) ? '1' : '0';
		} else {
			// Default to enable if not set.
			$sanitized['hover_effect'] = isset( $current_settings['hover_effect'] ) ? $current_settings['hover_effect'] : '1';
		}
		// Convert enable/disable to 1/0 for pulse_animation.
		if ( isset( $input['pulse_animation'] ) ) {
			$pulse_animation_value        = sanitize_text_field( $input['pulse_animation'] );
			$sanitized['pulse_animation'] = ( 'enable' === $pulse_animation_value ) ? '1' : '0';
		} else {
			// Default to enable if not set.
			$sanitized['pulse_animation'] = isset( $current_settings['pulse_animation'] ) ? $current_settings['pulse_animation'] : '1';
		}
		if ( isset( $input['submenu_bg_color'] ) ) {
			$submenu_bg                    = sanitize_hex_color( $input['submenu_bg_color'] );
			$default_submenu_bg            = isset( $current_settings['submenu_bg_color'] ) ? $current_settings['submenu_bg_color'] : '#1e1e28';
			$sanitized['submenu_bg_color'] = ( false !== $submenu_bg ) ? $submenu_bg : $default_submenu_bg;
		}
		if ( isset( $input['submenu_bg_opacity'] ) ) {
			$sanitized['submenu_bg_opacity'] = absint( $input['submenu_bg_opacity'] );
			$sanitized['submenu_bg_opacity'] = max( 0, min( 100, $sanitized['submenu_bg_opacity'] ) );
		}
		// Convert enable/disable to 1/0 for submenu_shadow.
		if ( isset( $input['submenu_shadow'] ) ) {
			$submenu_shadow_value        = sanitize_text_field( $input['submenu_shadow'] );
			$sanitized['submenu_shadow'] = ( 'enable' === $submenu_shadow_value ) ? '1' : '0';
		} else {
			// Default to enable if not set.
			$sanitized['submenu_shadow'] = isset( $current_settings['submenu_shadow'] ) ? $current_settings['submenu_shadow'] : '1';
		}
		if ( isset( $input['submenu_width'] ) ) {
			$sanitized['submenu_width'] = absint( $input['submenu_width'] );
			$sanitized['submenu_width'] = max( 200, min( 600, $sanitized['submenu_width'] ) );
		}
		if ( isset( $input['submenu_text_color'] ) ) {
			$submenu_text                    = sanitize_hex_color( $input['submenu_text_color'] );
			$default_submenu_text            = isset( $current_settings['submenu_text_color'] ) ? $current_settings['submenu_text_color'] : '#ffffff';
			$sanitized['submenu_text_color'] = ( false !== $submenu_text ) ? $submenu_text : $default_submenu_text;
		}
		if ( isset( $input['submenu_icon_spacing'] ) ) {
			$sanitized['submenu_icon_spacing'] = absint( $input['submenu_icon_spacing'] );
			$sanitized['submenu_icon_spacing'] = max( 0, min( 30, $sanitized['submenu_icon_spacing'] ) );
		}
		if ( isset( $input['submenu_shadow_color'] ) ) {
			$submenu_shadow                    = sanitize_hex_color( $input['submenu_shadow_color'] );
			$default_submenu_shadow            = isset( $current_settings['submenu_shadow_color'] ) ? $current_settings['submenu_shadow_color'] : '#000000';
			$sanitized['submenu_shadow_color'] = ( false !== $submenu_shadow ) ? $submenu_shadow : $default_submenu_shadow;
		}
		if ( isset( $input['button_text_size'] ) ) {
			$sanitized['button_text_size'] = absint( $input['button_text_size'] );
			$sanitized['button_text_size'] = max( 8, min( 20, $sanitized['button_text_size'] ) );
		}
		if ( isset( $input['button_icon_size'] ) ) {
			$sanitized['button_icon_size'] = absint( $input['button_icon_size'] );
			$sanitized['button_icon_size'] = max( 12, min( 32, $sanitized['button_icon_size'] ) );
		}
		if ( isset( $input['submenu_text_size'] ) ) {
			$sanitized['submenu_text_size'] = absint( $input['submenu_text_size'] );
			$sanitized['submenu_text_size'] = max( 10, min( 20, $sanitized['submenu_text_size'] ) );
		}
		if ( isset( $input['submenu_icon_size'] ) ) {
			$sanitized['submenu_icon_size'] = absint( $input['submenu_icon_size'] );
			$sanitized['submenu_icon_size'] = max( 16, min( 32, $sanitized['submenu_icon_size'] ) );
		}
		// Convert enable/disable to 1/0 for submenu_header_enable.
		if ( isset( $input['submenu_header_enable'] ) ) {
			$submenu_header_enable_value        = sanitize_text_field( $input['submenu_header_enable'] );
			$sanitized['submenu_header_enable'] = ( 'enable' === $submenu_header_enable_value ) ? '1' : '0';
		} else {
			// Default to enable if not set.
			$sanitized['submenu_header_enable'] = isset( $current_settings['submenu_header_enable'] ) ? $current_settings['submenu_header_enable'] : '1';
		}
		if ( isset( $input['submenu_header_logo'] ) ) {
			$sanitized['submenu_header_logo'] = esc_url_raw( $input['submenu_header_logo'] );
		}
		if ( isset( $input['submenu_header_logo_id'] ) ) {
			$sanitized['submenu_header_logo_id'] = absint( $input['submenu_header_logo_id'] );
		}
		if ( isset( $input['submenu_header_logo_size'] ) ) {
			$sanitized['submenu_header_logo_size'] = absint( $input['submenu_header_logo_size'] );
			$sanitized['submenu_header_logo_size'] = max( 60, min( 200, $sanitized['submenu_header_logo_size'] ) );
		}
		if ( isset( $input['submenu_header_title'] ) ) {
			$sanitized['submenu_header_title'] = sanitize_text_field( $input['submenu_header_title'] );
		}
		if ( isset( $input['submenu_header_description'] ) ) {
			$submenu_header_description              = sanitize_text_field( $input['submenu_header_description'] );
			$sanitized['submenu_header_description'] = mb_substr( $submenu_header_description, 0, 150 );
		}
		if ( isset( $input['submenu_header_bg_color'] ) ) {
			$sanitized['submenu_header_bg_color'] = sanitize_text_field( $input['submenu_header_bg_color'] );
		}
		if ( isset( $input['submenu_header_title_color'] ) ) {
			$header_title_color                      = sanitize_hex_color( $input['submenu_header_title_color'] );
			$default_header_title                    = isset( $current_settings['submenu_header_title_color'] ) ? $current_settings['submenu_header_title_color'] : '#ffffff';
			$sanitized['submenu_header_title_color'] = ( false !== $header_title_color ) ? $header_title_color : $default_header_title;
		}
		if ( isset( $input['submenu_header_title_size'] ) ) {
			$sanitized['submenu_header_title_size'] = absint( $input['submenu_header_title_size'] );
			$sanitized['submenu_header_title_size'] = max( 12, min( 32, $sanitized['submenu_header_title_size'] ) );
		}
		if ( isset( $input['submenu_header_description_color'] ) ) {
			$sanitized['submenu_header_description_color'] = sanitize_text_field( $input['submenu_header_description_color'] );
		}
		if ( isset( $input['submenu_header_description_size'] ) ) {
			$sanitized['submenu_header_description_size'] = absint( $input['submenu_header_description_size'] );
			$sanitized['submenu_header_description_size'] = max( 10, min( 20, $sanitized['submenu_header_description_size'] ) );
		}
		if ( isset( $input['visibility_mode'] ) ) {
			// Handle both array (new) and string (old) formats.
			if ( is_array( $input['visibility_mode'] ) ) {
				$allowed_modes                = array( 'all', 'homepage', 'pages', 'posts', 'categories', 'post_types' );
				$sanitized['visibility_mode'] = array_intersect( $input['visibility_mode'], $allowed_modes );
				// If "all" is selected, only keep "all".
				if ( in_array( 'all', $sanitized['visibility_mode'], true ) ) {
					$sanitized['visibility_mode'] = array( 'all' );
				}
				// If nothing selected, default to "all".
				if ( empty( $sanitized['visibility_mode'] ) ) {
					$sanitized['visibility_mode'] = array( 'all' );
				}
			} else {
				// Legacy support: convert single value to array.
				$sanitized['visibility_mode'] = array( sanitize_text_field( $input['visibility_mode'] ) );
			}
		} else {
			// Default to "all" if not set.
			$sanitized['visibility_mode'] = array( 'all' );
		}
		if ( isset( $input['device_visibility'] ) ) {
			$sanitized['device_visibility'] = sanitize_text_field( $input['device_visibility'] );
		}
		// Convert enable/disable to 1/0 for hide_on_login.
		if ( isset( $input['hide_on_login'] ) ) {
			$hide_on_login_value        = sanitize_text_field( $input['hide_on_login'] );
			$sanitized['hide_on_login'] = ( 'enable' === $hide_on_login_value ) ? '1' : '0';
		} else {
			// Default to enable if not set.
			$sanitized['hide_on_login'] = isset( $current_settings['hide_on_login'] ) ? $current_settings['hide_on_login'] : '1';
		}
		if ( isset( $input['hide_in_admin'] ) ) {
			$hide_in_admin_value        = sanitize_text_field( $input['hide_in_admin'] );
			$sanitized['hide_in_admin'] = ( 'enable' === $hide_in_admin_value ) ? '1' : '0';
		} else {
			// Default to enable if not set.
			$sanitized['hide_in_admin'] = isset( $current_settings['hide_in_admin'] ) ? $current_settings['hide_in_admin'] : '1';
		}

		// Sanitize visibility arrays.
		if ( isset( $input['visibility_pages'] ) ) {
			$sanitized['visibility_pages'] = array_map( 'absint', $input['visibility_pages'] );
		}
		if ( isset( $input['visibility_posts'] ) ) {
			$sanitized['visibility_posts'] = array_map( 'absint', $input['visibility_posts'] );
		}
		if ( isset( $input['visibility_categories'] ) ) {
			$sanitized['visibility_categories'] = array_map( 'absint', $input['visibility_categories'] );
		}
		if ( isset( $input['visibility_post_types'] ) ) {
			$sanitized['visibility_post_types'] = array_map( 'sanitize_text_field', $input['visibility_post_types'] );
		}

		// Sanitize User-Agent settings.
		if ( isset( $input['user_agent_enabled'] ) ) {
			$user_agent_enabled_value        = sanitize_text_field( $input['user_agent_enabled'] );
			$sanitized['user_agent_enabled'] = ( 'enable' === $user_agent_enabled_value ) ? '1' : '0';
		} else {
			$sanitized['user_agent_enabled'] = isset( $current_settings['user_agent_enabled'] ) ? $current_settings['user_agent_enabled'] : '0';
		}

		if ( isset( $input['user_agent_mode'] ) ) {
			$allowed_modes                = array( 'show_all', 'show_only_in_app', 'hide_only_in_app' );
			$user_agent_mode              = sanitize_text_field( $input['user_agent_mode'] );
			$sanitized['user_agent_mode'] = in_array( $user_agent_mode, $allowed_modes, true ) ? $user_agent_mode : 'show_all';
		} else {
			$sanitized['user_agent_mode'] = isset( $current_settings['user_agent_mode'] ) ? $current_settings['user_agent_mode'] : 'show_all';
		}

		if ( isset( $input['user_agent_list'] ) ) {
			if ( is_array( $input['user_agent_list'] ) ) {
				$sanitized['user_agent_list'] = array_map( 'sanitize_text_field', array_filter( $input['user_agent_list'] ) );
			} else {
				// Handle textarea input - split by lines.
				$user_agent_text              = sanitize_textarea_field( $input['user_agent_list'] );
				$user_agent_lines             = preg_split( '/\r\n|\r|\n/', $user_agent_text );
				$sanitized['user_agent_list'] = array_map( 'trim', array_filter( $user_agent_lines ) );
			}
		} else {
			$sanitized['user_agent_list'] = isset( $current_settings['user_agent_list'] ) ? $current_settings['user_agent_list'] : array();
		}

		return $sanitized;
	}

	/**
	 * Restore SVG icon fields from original POST data (before sanitize_text_field stripped HTML).
	 *
	 * @param array  $original_data Original unslashed data with SVG intact.
	 * @param array  $sanitized_data Data after map_deep with sanitize_text_field (SVG stripped).
	 * @param string $type Type of data: 'buttons' or 'submenus'.
	 * @return array Data with SVG fields restored.
	 */
	private function restore_svg_from_post( $original_data, $sanitized_data, $type ) {
		if ( ! is_array( $original_data ) || ! is_array( $sanitized_data ) ) {
			return $sanitized_data;
		}

		if ( 'buttons' === $type ) {
			foreach ( $sanitized_data as $key => $button ) {
				if ( isset( $original_data[ $key ]['icon'] ) && is_string( $original_data[ $key ]['icon'] ) ) {
					$sanitized_data[ $key ]['icon'] = $original_data[ $key ]['icon'];
				}
			}
		} elseif ( 'submenus' === $type ) {
			foreach ( $sanitized_data as $button_id => $items ) {
				if ( ! is_array( $items ) || ! isset( $original_data[ $button_id ] ) ) {
					continue;
				}
				foreach ( $items as $key => $item ) {
					if ( isset( $original_data[ $button_id ][ $key ]['icon'] ) && is_string( $original_data[ $button_id ][ $key ]['icon'] ) ) {
						$sanitized_data[ $button_id ][ $key ]['icon'] = $original_data[ $button_id ][ $key ]['icon'];
					}
				}
			}
		}

		return $sanitized_data;
	}

	/**
	 * Sanitize a color value for button fields (hex or rgb/rgba from Iris).
	 *
	 * @param string $color Raw color value from POST.
	 * @return string|false Hex color or false if invalid.
	 */
	private function sanitize_color_for_buttons( $color ) {
		if ( ! is_string( $color ) ) {
			return false;
		}
		$color = trim( $color );
		if ( '' === $color ) {
			return false;
		}
		$hex = sanitize_hex_color( $color );
		if ( false !== $hex ) {
			return $hex;
		}
		// Iris may send rgb(r,g,b) or rgba(r,g,b,a) - convert to hex.
		if ( preg_match( '/^rgba?\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)/', $color, $m ) ) {
			$r = min( 255, max( 0, (int) $m[1] ) );
			$g = min( 255, max( 0, (int) $m[2] ) );
			$b = min( 255, max( 0, (int) $m[3] ) );
			return sprintf( '#%02x%02x%02x', $r, $g, $b );
		}
		return false;
	}

	/**
	 * Sanitize buttons
	 *
	 * @param array $input The input array to sanitize.
	 * @return array The sanitized buttons array.
	 */
	public function sanitize_buttons( $input ) {
		if ( ! is_array( $input ) ) {
			return array();
		}

		$sanitized = array();
		foreach ( $input as $button ) {
			$sanitized_button          = array();
			$sanitized_button['id']    = isset( $button['id'] ) ? sanitize_text_field( $button['id'] ) : '';
			$sanitized_button['label'] = isset( $button['label'] ) ? sanitize_text_field( $button['label'] ) : '';
			// SVG code - allow HTML/SVG code with custom sanitization.
			$sanitized_button['icon'] = isset( $button['icon'] ) ? $this->sanitize_svg( $button['icon'] ) : '';
			// Use default when empty or invalid hex (sanitize_hex_color returns false for empty/invalid).
			// Also accept rgb/rgba from Iris and convert to hex.
			$_icon_color                    = isset( $button['icon_color'] ) ? $this->sanitize_color_for_buttons( $button['icon_color'] ) : false;
			$sanitized_button['icon_color'] = ( false !== $_icon_color ) ? $_icon_color : '#333333';
			$_text_color                    = isset( $button['text_color'] ) ? $this->sanitize_color_for_buttons( $button['text_color'] ) : false;
			$sanitized_button['text_color'] = ( false !== $_text_color ) ? $_text_color : '#333333';
			// Empty or missing bg_color means transparent (color picker sends empty when cleared).
			$_bg = isset( $button['bg_color'] ) ? trim( (string) $button['bg_color'] ) : '';
			if ( '' === $_bg ) {
				$sanitized_button['bg_color'] = 'transparent';
			} else {
				$_bg_hex                      = $this->sanitize_color_for_buttons( $_bg );
				$sanitized_button['bg_color'] = ( false !== $_bg_hex ) ? $_bg_hex : 'transparent';
			}
			$_active_color                    = isset( $button['active_color'] ) ? $this->sanitize_color_for_buttons( $button['active_color'] ) : false;
			$sanitized_button['active_color'] = ( false !== $_active_color ) ? $_active_color : '#0073aa';
			$sanitized_button['url']          = isset( $button['url'] ) ? esc_url_raw( $button['url'] ) : '';
			$sanitized_button['target']       = isset( $button['target'] ) ? sanitize_text_field( $button['target'] ) : '_self';
			// has_submenu: convert enable/disable to 1/0.
			$has_submenu_value               = isset( $button['has_submenu'] ) ? sanitize_text_field( $button['has_submenu'] ) : 'disable';
			$sanitized_button['has_submenu'] = ( 'enable' === $has_submenu_value ) ? '1' : '0';
			$sanitized_button['order']       = isset( $button['order'] ) ? absint( $button['order'] ) : 0;

			$sanitized[] = $sanitized_button;
		}

		return $sanitized;
	}

	/**
	 * Sanitize submenus
	 *
	 * @param array $input The input array to sanitize.
	 * @return array The sanitized submenus array.
	 */
	public function sanitize_submenus( $input ) {
		if ( ! is_array( $input ) ) {
			return array();
		}

		$sanitized = array();
		foreach ( $input as $button_id => $submenu_items ) {
			if ( ! is_array( $submenu_items ) ) {
				continue;
			}

			$sanitized_items = array();
			foreach ( $submenu_items as $item ) {
				$sanitized_item          = array();
				$sanitized_item['label'] = isset( $item['label'] ) ? sanitize_text_field( $item['label'] ) : '';
				// SVG code - allow HTML/SVG code with custom sanitization.
				$sanitized_item['icon']       = isset( $item['icon'] ) ? $this->sanitize_svg( $item['icon'] ) : '';
				$sanitized_item['icon_color'] = isset( $item['icon_color'] ) ? sanitize_hex_color( $item['icon_color'] ) : '#333333';
				$sanitized_item['url']        = isset( $item['url'] ) ? esc_url_raw( $item['url'] ) : '';
				$sanitized_item['target']     = isset( $item['target'] ) ? sanitize_text_field( $item['target'] ) : '_self';

				$sanitized_items[] = $sanitized_item;
			}

			$sanitized[ sanitize_text_field( $button_id ) ] = $sanitized_items;
		}

		return $sanitized;
	}

	/**
	 * Render main settings page with horizontal tabs
	 */
	public function render_settings_page() {
		// Check user capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Sorry, you are not allowed to access this page.', 'arewa-mobile-bottom-navigation-bar' ) );
		}

		// Get current tab from URL (sanitized, validated against whitelist, nonce verified when present).
		$allowed_tabs = array( 'general', 'buttons', 'submenus', 'styling', 'visibility' );
		$request_tab  = isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : '';
		if ( isset( $_GET['_wpnonce'] ) && ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'arewa_mbn_tab' ) ) {
			$request_tab = '';
		}
		$current_tab = in_array( $request_tab, $allowed_tabs, true ) ? $request_tab : 'general';

		// Form submissions are now handled in process_form_submissions() which runs earlier.

		// Get all data needed.
		$settings = get_option( 'arewa_mbn_settings', array() );
		$buttons  = get_option( 'arewa_mbn_buttons', array() );
		$submenus = get_option( 'arewa_mbn_submenus', array() );

		// Define tabs.
		$tabs = array(
			'general'    => __( 'General Settings', 'arewa-mobile-bottom-navigation-bar' ),
			'buttons'    => __( 'Buttons Configuration', 'arewa-mobile-bottom-navigation-bar' ),
			'submenus'   => __( 'Submenu Configuration', 'arewa-mobile-bottom-navigation-bar' ),
			'styling'    => __( 'Styling & Appearance', 'arewa-mobile-bottom-navigation-bar' ),
			'visibility' => __( 'Visibility Conditions', 'arewa-mobile-bottom-navigation-bar' ),
		);

		// Include the main settings template.
		include AREWA_MBN_PLUGIN_DIR . 'templates/admin-settings.php';
	}

	/**
	 * Render general settings page (kept for backward compatibility)
	 */
	public function render_general_settings() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Handle form submission.
		if ( isset( $_POST['arewa_mbn_save_general'] ) && check_admin_referer( 'arewa_mbn_general_nonce' ) ) {
			$this->save_general_settings();
		}

		$settings = get_option( 'arewa_mbn_settings', array() );
		include AREWA_MBN_PLUGIN_DIR . 'templates/admin-general.php';
	}

	/**
	 * Render buttons settings page
	 */
	public function render_buttons_settings() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Handle form submission.
		if ( isset( $_POST['arewa_mbn_save_buttons'] ) && check_admin_referer( 'arewa_mbn_buttons_nonce' ) ) {
			$this->save_buttons_settings();
		}

		$settings = get_option( 'arewa_mbn_settings', array() );
		$buttons  = get_option( 'arewa_mbn_buttons', array() );
		include AREWA_MBN_PLUGIN_DIR . 'templates/admin-buttons.php';
	}

	/**
	 * Render submenus settings page
	 */
	public function render_submenus_settings() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Handle form submission.
		if ( isset( $_POST['arewa_mbn_save_submenus'] ) && check_admin_referer( 'arewa_mbn_submenus_nonce' ) ) {
			$this->save_submenus_settings();
		}

		$settings = get_option( 'arewa_mbn_settings', array() );
		$buttons  = get_option( 'arewa_mbn_buttons', array() );
		$submenus = get_option( 'arewa_mbn_submenus', array() );
		include AREWA_MBN_PLUGIN_DIR . 'templates/admin-submenus.php';
	}

	/**
	 * Render styling settings page
	 */
	public function render_styling_settings() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Handle form submission.
		if ( isset( $_POST['arewa_mbn_save_styling'] ) && check_admin_referer( 'arewa_mbn_styling_nonce' ) ) {
			$this->save_styling_settings();
		}

		$settings = get_option( 'arewa_mbn_settings', array() );
		include AREWA_MBN_PLUGIN_DIR . 'templates/admin-styling.php';
	}

	/**
	 * Render visibility settings page
	 */
	public function render_visibility_settings() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Handle form submission.
		if ( isset( $_POST['arewa_mbn_save_visibility'] ) && check_admin_referer( 'arewa_mbn_visibility_nonce' ) ) {
			$this->save_visibility_settings();
		}

		$settings = get_option( 'arewa_mbn_settings', array() );
		include AREWA_MBN_PLUGIN_DIR . 'templates/admin-visibility.php';
	}

	/**
	 * Save general settings
	 */
	private function save_general_settings() {
		if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'arewa_mbn_settings_group-options' ) ) {
			return;
		}
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$current_settings = get_option( 'arewa_mbn_settings', array() );
		$general_settings = array();
		$allowed_layouts  = array( 'icon_only', 'icon_text', 'icon_text_active' );

		// Get settings array once and sanitize.
		$raw_settings = array();
		if ( ! empty( $_POST['arewa_mbn_settings'] ) && is_array( $_POST['arewa_mbn_settings'] ) ) {
			$raw_settings = map_deep( wp_unslash( $_POST['arewa_mbn_settings'] ), 'sanitize_text_field' );
		}

		// Enabled - convert enable/disable to 1/0.
		if ( isset( $raw_settings['enabled'] ) ) {
			$general_settings['enabled'] = ( 'enable' === trim( $raw_settings['enabled'] ) ) ? '1' : '0';
		} else {
			$general_settings['enabled'] = isset( $current_settings['enabled'] ) ? $current_settings['enabled'] : '1';
		}

		// Button count - sanitize and clamp.
		if ( isset( $raw_settings['button_count'] ) ) {
			$general_settings['button_count'] = max( 2, min( 6, absint( $raw_settings['button_count'] ) ) );
		} else {
			$general_settings['button_count'] = isset( $current_settings['button_count'] ) ? $current_settings['button_count'] : 4;
		}

		// Layout style - whitelist.
		if ( isset( $raw_settings['layout_style'] ) ) {
			$general_settings['layout_style'] = in_array( $raw_settings['layout_style'], $allowed_layouts, true ) ? $raw_settings['layout_style'] : ( $current_settings['layout_style'] ?? 'icon_text' );
		} else {
			$general_settings['layout_style'] = isset( $current_settings['layout_style'] ) ? $current_settings['layout_style'] : 'icon_text';
		}

		$settings = array_merge( $current_settings, $general_settings );
		update_option( 'arewa_mbn_settings', $settings, false );

		add_settings_error(
			'arewa_mbn_settings',
			'arewa_mbn_settings_updated',
			__( 'General settings saved successfully.', 'arewa-mobile-bottom-navigation-bar' ),
			'updated'
		);
	}

	/**
	 * Save buttons settings
	 */
	private function save_buttons_settings() {
		if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'arewa_mbn_settings_group-options' ) ) {
			return;
		}
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$buttons = array();
		if ( ! empty( $_POST['arewa_mbn_buttons'] ) && is_array( $_POST['arewa_mbn_buttons'] ) ) {
			// Unslash first, then sanitize_buttons handles field-specific sanitization including SVG via wp_kses.
			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce verified, sanitized in sanitize_buttons() method.
			$unslashed_buttons = wp_unslash( $_POST['arewa_mbn_buttons'] );
			// Debug: log first button colors when WP_DEBUG_LOG is on (remove after fixing).
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG && ! empty( $unslashed_buttons[0] ) ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
				error_log( 'Arewa MBN POST colors btn0: icon=' . ( $unslashed_buttons[0]['icon_color'] ?? '?' ) . ' text=' . ( $unslashed_buttons[0]['text_color'] ?? '?' ) . ' bg=' . ( $unslashed_buttons[0]['bg_color'] ?? '?' ) );
			}
			// Don't use map_deep with sanitize_text_field as it strips color values (#).
			// Instead, pass directly to sanitize_buttons which handles each field appropriately.
			$buttons = $this->sanitize_buttons( $unslashed_buttons );
		}
		if ( ! empty( $buttons ) ) {
			update_option( 'arewa_mbn_buttons', $buttons );
		}

		add_settings_error(
			'arewa_mbn_settings',
			'arewa_mbn_settings_updated',
			__( 'Buttons configuration saved successfully.', 'arewa-mobile-bottom-navigation-bar' ),
			'updated'
		);
	}

	/**
	 * Save submenus settings
	 */
	private function save_submenus_settings() {
		if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'arewa_mbn_settings_group-options' ) ) {
			return;
		}
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$submenus = array();
		if ( ! empty( $_POST['arewa_mbn_submenus'] ) && is_array( $_POST['arewa_mbn_submenus'] ) ) {
			// Unslash first, then sanitize_submenus handles field-specific sanitization including SVG via wp_kses.
			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce verified, sanitized in sanitize_submenus() method.
			$unslashed_submenus = wp_unslash( $_POST['arewa_mbn_submenus'] );
			// Don't use map_deep with sanitize_text_field as it strips color values (#).
			// Instead, pass directly to sanitize_submenus which handles each field appropriately.
			$submenus = $this->sanitize_submenus( $unslashed_submenus );
		}
		if ( ! empty( $submenus ) ) {
			update_option( 'arewa_mbn_submenus', $submenus );
		}

		add_settings_error(
			'arewa_mbn_settings',
			'arewa_mbn_settings_updated',
			__( 'Submenu configuration saved successfully.', 'arewa-mobile-bottom-navigation-bar' ),
			'updated'
		);
	}

	/**
	 * Save styling settings
	 */
	private function save_styling_settings() {
		if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'arewa_mbn_settings_group-options' ) ) {
			return;
		}
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$current_settings = get_option( 'arewa_mbn_settings', array() );
		$styling_settings = array(
			'background_color'                 => $current_settings['background_color'] ?? '#ffffff',
			'border_radius'                    => $current_settings['border_radius'] ?? 0,
			'box_shadow'                       => $current_settings['box_shadow'] ?? '1',
			'active_color'                     => $current_settings['active_color'] ?? '#0073aa',
			'hover_effect'                     => $current_settings['hover_effect'] ?? '1',
			'pulse_animation'                  => $current_settings['pulse_animation'] ?? '1',
			'submenu_bg_color'                 => $current_settings['submenu_bg_color'] ?? '#1e1e28',
			'submenu_bg_opacity'               => $current_settings['submenu_bg_opacity'] ?? 95,
			'submenu_shadow'                   => $current_settings['submenu_shadow'] ?? '1',
			'submenu_width'                    => $current_settings['submenu_width'] ?? 320,
			'submenu_text_color'               => $current_settings['submenu_text_color'] ?? '#ffffff',
			'submenu_icon_spacing'             => $current_settings['submenu_icon_spacing'] ?? 8,
			'submenu_shadow_color'             => $current_settings['submenu_shadow_color'] ?? '#000000',
			'button_text_size'                 => $current_settings['button_text_size'] ?? 12,
			'button_icon_size'                 => $current_settings['button_icon_size'] ?? 20,
			'submenu_text_size'                => $current_settings['submenu_text_size'] ?? 14,
			'submenu_icon_size'                => $current_settings['submenu_icon_size'] ?? 22,
			'submenu_header_enable'            => $current_settings['submenu_header_enable'] ?? '1',
			'submenu_header_logo'              => $current_settings['submenu_header_logo'] ?? '',
			'submenu_header_logo_id'           => $current_settings['submenu_header_logo_id'] ?? 0,
			'submenu_header_logo_size'         => $current_settings['submenu_header_logo_size'] ?? 120,
			'submenu_header_title'             => $current_settings['submenu_header_title'] ?? get_bloginfo( 'name' ),
			'submenu_header_description'       => $current_settings['submenu_header_description'] ?? '',
			'submenu_header_bg_color'          => $current_settings['submenu_header_bg_color'] ?? 'transparent',
			'submenu_header_title_color'       => $current_settings['submenu_header_title_color'] ?? '#ffffff',
			'submenu_header_title_size'        => $current_settings['submenu_header_title_size'] ?? 18,
			'submenu_header_description_color' => $current_settings['submenu_header_description_color'] ?? 'rgba(255, 255, 255, 0.7)',
			'submenu_header_description_size'  => $current_settings['submenu_header_description_size'] ?? 13,
		);

		// Get settings array once and sanitize.
		$raw_settings = array();
		if ( ! empty( $_POST['arewa_mbn_settings'] ) && is_array( $_POST['arewa_mbn_settings'] ) ) {
			$raw_settings = map_deep( wp_unslash( $_POST['arewa_mbn_settings'] ), 'sanitize_text_field' );

			$styling_settings['background_color']                 = isset( $raw_settings['background_color'] ) ? sanitize_hex_color( $raw_settings['background_color'] ) : $styling_settings['background_color'];
			$styling_settings['border_radius']                    = isset( $raw_settings['border_radius'] ) ? absint( $raw_settings['border_radius'] ) : $styling_settings['border_radius'];
			$styling_settings['box_shadow']                       = isset( $raw_settings['box_shadow'] ) ? ( 'enable' === trim( sanitize_text_field( $raw_settings['box_shadow'] ) ) ? '1' : '0' ) : $styling_settings['box_shadow'];
			$styling_settings['active_color']                     = isset( $raw_settings['active_color'] ) ? sanitize_hex_color( $raw_settings['active_color'] ) : $styling_settings['active_color'];
			$styling_settings['hover_effect']                     = isset( $raw_settings['hover_effect'] ) ? ( 'enable' === trim( sanitize_text_field( $raw_settings['hover_effect'] ) ) ? '1' : '0' ) : $styling_settings['hover_effect'];
			$styling_settings['pulse_animation']                  = isset( $raw_settings['pulse_animation'] ) ? ( 'enable' === trim( sanitize_text_field( $raw_settings['pulse_animation'] ) ) ? '1' : '0' ) : $styling_settings['pulse_animation'];
			$styling_settings['submenu_bg_color']                 = isset( $raw_settings['submenu_bg_color'] ) ? sanitize_hex_color( $raw_settings['submenu_bg_color'] ) : $styling_settings['submenu_bg_color'];
			$styling_settings['submenu_bg_opacity']               = isset( $raw_settings['submenu_bg_opacity'] ) ? absint( $raw_settings['submenu_bg_opacity'] ) : $styling_settings['submenu_bg_opacity'];
			$styling_settings['submenu_shadow']                   = isset( $raw_settings['submenu_shadow'] ) ? ( 'enable' === trim( sanitize_text_field( $raw_settings['submenu_shadow'] ) ) ? '1' : '0' ) : $styling_settings['submenu_shadow'];
			$styling_settings['submenu_width']                    = isset( $raw_settings['submenu_width'] ) ? absint( $raw_settings['submenu_width'] ) : $styling_settings['submenu_width'];
			$styling_settings['submenu_text_color']               = isset( $raw_settings['submenu_text_color'] ) ? sanitize_hex_color( $raw_settings['submenu_text_color'] ) : $styling_settings['submenu_text_color'];
			$styling_settings['submenu_icon_spacing']             = isset( $raw_settings['submenu_icon_spacing'] ) ? absint( $raw_settings['submenu_icon_spacing'] ) : $styling_settings['submenu_icon_spacing'];
			$styling_settings['submenu_shadow_color']             = isset( $raw_settings['submenu_shadow_color'] ) ? sanitize_hex_color( $raw_settings['submenu_shadow_color'] ) : $styling_settings['submenu_shadow_color'];
			$styling_settings['button_text_size']                 = isset( $raw_settings['button_text_size'] ) ? absint( $raw_settings['button_text_size'] ) : $styling_settings['button_text_size'];
			$styling_settings['button_icon_size']                 = isset( $raw_settings['button_icon_size'] ) ? absint( $raw_settings['button_icon_size'] ) : $styling_settings['button_icon_size'];
			$styling_settings['submenu_text_size']                = isset( $raw_settings['submenu_text_size'] ) ? absint( $raw_settings['submenu_text_size'] ) : $styling_settings['submenu_text_size'];
			$styling_settings['submenu_icon_size']                = isset( $raw_settings['submenu_icon_size'] ) ? absint( $raw_settings['submenu_icon_size'] ) : $styling_settings['submenu_icon_size'];
			$styling_settings['submenu_header_enable']            = isset( $raw_settings['submenu_header_enable'] ) ? ( 'enable' === trim( sanitize_text_field( $raw_settings['submenu_header_enable'] ) ) ? '1' : '0' ) : $styling_settings['submenu_header_enable'];
			$styling_settings['submenu_header_logo']              = isset( $raw_settings['submenu_header_logo'] ) ? esc_url_raw( $raw_settings['submenu_header_logo'] ) : $styling_settings['submenu_header_logo'];
			$styling_settings['submenu_header_logo_id']           = isset( $raw_settings['submenu_header_logo_id'] ) ? absint( $raw_settings['submenu_header_logo_id'] ) : $styling_settings['submenu_header_logo_id'];
			$styling_settings['submenu_header_logo_size']         = isset( $raw_settings['submenu_header_logo_size'] ) ? absint( $raw_settings['submenu_header_logo_size'] ) : $styling_settings['submenu_header_logo_size'];
			$styling_settings['submenu_header_title']             = isset( $raw_settings['submenu_header_title'] ) ? sanitize_text_field( $raw_settings['submenu_header_title'] ) : $styling_settings['submenu_header_title'];
			$styling_settings['submenu_header_description']       = isset( $raw_settings['submenu_header_description'] ) ? mb_substr( sanitize_text_field( $raw_settings['submenu_header_description'] ), 0, 150 ) : $styling_settings['submenu_header_description'];
			$styling_settings['submenu_header_bg_color']          = isset( $raw_settings['submenu_header_bg_color'] ) ? sanitize_text_field( $raw_settings['submenu_header_bg_color'] ) : $styling_settings['submenu_header_bg_color'];
			$styling_settings['submenu_header_title_color']       = isset( $raw_settings['submenu_header_title_color'] ) ? sanitize_hex_color( $raw_settings['submenu_header_title_color'] ) : $styling_settings['submenu_header_title_color'];
			$styling_settings['submenu_header_title_size']        = isset( $raw_settings['submenu_header_title_size'] ) ? absint( $raw_settings['submenu_header_title_size'] ) : $styling_settings['submenu_header_title_size'];
			$styling_settings['submenu_header_description_color'] = isset( $raw_settings['submenu_header_description_color'] ) ? sanitize_text_field( $raw_settings['submenu_header_description_color'] ) : $styling_settings['submenu_header_description_color'];
			$styling_settings['submenu_header_description_size']  = isset( $raw_settings['submenu_header_description_size'] ) ? absint( $raw_settings['submenu_header_description_size'] ) : $styling_settings['submenu_header_description_size'];
		}
		$settings = array_merge( $current_settings, $styling_settings );
		update_option( 'arewa_mbn_settings', $settings );

		add_settings_error(
			'arewa_mbn_settings',
			'arewa_mbn_settings_updated',
			__( 'Styling settings saved successfully.', 'arewa-mobile-bottom-navigation-bar' ),
			'updated'
		);
	}

	/**
	 * Sanitize and normalize SVG code
	 *
	 * @param string $svg_code The SVG code to sanitize.
	 * @return string The sanitized and normalized SVG code.
	 */
	private function sanitize_svg( $svg_code ) {
		if ( empty( $svg_code ) ) {
			return '';
		}

		// Remove HTML comments.
		$svg_code = preg_replace( '/<!--.*?-->/s', '', $svg_code );

		// Remove XML processing instructions if present.
		$svg_code = preg_replace( '/<\?xml[^>]*\?>/i', '', $svg_code );

		// Extract SVG content - handle cases where SVG might be wrapped or have extra whitespace.
		$svg_code = trim( $svg_code );

		// If the code doesn't start with <svg, try to extract it.
		if ( false === strpos( $svg_code, '<svg' ) ) {
			// Try to find SVG tag.
			preg_match( '/<svg[^>]*>.*?<\/svg>/is', $svg_code, $matches );
			if ( ! empty( $matches[0] ) ) {
				$svg_code = $matches[0];
			} else {
				return '';
			}
		}

		// Normalize SVG attributes - ensure xmlns is present.
		if ( false === strpos( $svg_code, 'xmlns=' ) && false === strpos( $svg_code, 'xmlns =' ) ) {
			$svg_code = preg_replace( '/<svg\s+/i', '<svg xmlns="http://www.w3.org/2000/svg" ', $svg_code, 1 );
		}

		// Ensure viewBox is present - try to extract from width/height if viewBox is missing.
		if ( false === strpos( $svg_code, 'viewBox=' ) && false === strpos( $svg_code, 'viewbox=' ) ) {
			// Try to extract width and height.
			preg_match( '/width=["\']?(\d+)/i', $svg_code, $width_match );
			preg_match( '/height=["\']?(\d+)/i', $svg_code, $height_match );
			if ( ! empty( $width_match[1] ) && ! empty( $height_match[1] ) ) {
				$viewbox  = '0 0 ' . $width_match[1] . ' ' . $height_match[1];
				$svg_code = preg_replace( '/<svg\s+([^>]*)>/i', '<svg $1 viewBox="' . $viewbox . '">', $svg_code, 1 );
			} else {
				// Default viewBox if we can't determine size.
				$svg_code = preg_replace( '/<svg\s+([^>]*)>/i', '<svg $1 viewBox="0 0 24 24">', $svg_code, 1 );
			}
		}

		// Replace fill and stroke attributes with currentColor for better color control.
		// But preserve if they're already currentColor or none.
		$svg_code = preg_replace_callback(
			'/(<(?:path|circle|rect|line|polyline|polygon|g|ellipse|text|use)[^>]*)\s+(fill|stroke)=["\']([^"\']*)["\']/i',
			function( $matches ) {
				$tag   = $matches[1];
				$attr  = $matches[2];
				$value = strtolower( trim( $matches[3] ) );
				// Don't replace if it's already currentColor, none, or a URL (like gradients).
				if ( 'currentcolor' === $value || 'none' === $value || 0 === strpos( $value, 'url(' ) ) {
					return $matches[0];
				}
				// Replace with currentColor.
				return $tag . ' ' . $attr . '="currentColor"';
			},
			$svg_code
		);

		// Remove width and height attributes from SVG tag (we control size via CSS).
		$svg_code = preg_replace( '/<svg\s+([^>]*)\s+(width|height)=["\'][^"\']*["\']/i', '<svg $1', $svg_code );
		$svg_code = preg_replace( '/<svg\s+([^>]*)\s+(width|height)=["\'][^"\']*["\']/i', '<svg $1', $svg_code );

		// Allow SVG and common SVG elements.
		$allowed_html = array(
			'svg'      => array(
				'xmlns'               => true,
				'xmlns:xlink'         => true,
				'viewbox'             => true,
				'viewBox'             => true,
				'width'               => true,
				'height'              => true,
				'fill'                => true,
				'stroke'              => true,
				'class'               => true,
				'id'                  => true,
				'style'               => true,
				'version'             => true,
				'xml:space'           => true,
				'preserveaspectratio' => true,
				'preserveAspectRatio' => true,
			),
			'path'     => array(
				'd'                 => true,
				'fill'              => true,
				'stroke'            => true,
				'stroke-width'      => true,
				'stroke-linecap'    => true,
				'stroke-linejoin'   => true,
				'stroke-miterlimit' => true,
				'class'             => true,
				'id'                => true,
				'style'             => true,
			),
			'circle'   => array(
				'cx'                => true,
				'cy'                => true,
				'r'                 => true,
				'fill'              => true,
				'stroke'            => true,
				'stroke-width'      => true,
				'stroke-linejoin'   => true,
				'stroke-miterlimit' => true,
				'class'             => true,
				'id'                => true,
				'style'             => true,
			),
			'rect'     => array(
				'x'      => true,
				'y'      => true,
				'width'  => true,
				'height' => true,
				'rx'     => true,
				'ry'     => true,
				'fill'   => true,
				'stroke' => true,
				'class'  => true,
				'id'     => true,
				'style'  => true,
			),
			'line'     => array(
				'x1'           => true,
				'y1'           => true,
				'x2'           => true,
				'y2'           => true,
				'stroke'       => true,
				'stroke-width' => true,
				'class'        => true,
				'id'           => true,
				'style'        => true,
			),
			'polyline' => array(
				'points'       => true,
				'fill'         => true,
				'stroke'       => true,
				'stroke-width' => true,
				'class'        => true,
				'id'           => true,
				'style'        => true,
			),
			'polygon'  => array(
				'points'       => true,
				'fill'         => true,
				'stroke'       => true,
				'stroke-width' => true,
				'class'        => true,
				'id'           => true,
				'style'        => true,
			),
			'g'        => array(
				'fill'      => true,
				'stroke'    => true,
				'class'     => true,
				'id'        => true,
				'style'     => true,
				'transform' => true,
			),
			'defs'     => array(),
			'use'      => array(
				'href'       => true,
				'xlink:href' => true,
				'x'          => true,
				'y'          => true,
				'class'      => true,
				'id'         => true,
			),
			'desc'     => array(),
			'title'    => array(),
			'symbol'   => array(
				'id'      => true,
				'viewBox' => true,
				'class'   => true,
			),
		);

		return wp_kses( $svg_code, $allowed_html );
	}

	/**
	 * Sanitize and validate visibility_mode array (whitelist).
	 *
	 * @param array $raw_input Raw visibility_mode array (e.g. from wp_unslash( $_POST['arewa_mbn_settings']['visibility_mode'] )).
	 * @return array Sanitized visibility mode array.
	 */
	private function get_sanitized_visibility_mode( $raw_input ) {
		$allowed   = array( 'all', 'homepage', 'pages', 'posts', 'categories', 'post_types' );
		$sanitized = array_values( array_intersect( array_map( 'sanitize_text_field', (array) $raw_input ), $allowed ) );
		if ( in_array( 'all', $sanitized, true ) ) {
			return array( 'all' );
		}
		return empty( $sanitized ) ? array( 'all' ) : $sanitized;
	}

	/**
	 * Sanitize raw POST settings array into visibility settings (for PHPCS: custom sanitizer).
	 *
	 * @param array $raw_settings Unslashed raw settings (e.g. wp_unslash( $_POST['arewa_mbn_settings'] )). May be empty.
	 * @param array $current_settings Current option values for defaults.
	 * @return array Sanitized visibility_settings array (keys: visibility_mode, device_visibility, ...).
	 */
	public function sanitize_visibility_settings( $raw_settings, $current_settings = array() ) {
		$raw_settings          = is_array( $raw_settings ) ? $raw_settings : array();
		$visibility_mode       = array( 'all' );
		$device_visibility     = $current_settings['device_visibility'] ?? 'all';
		$hide_on_login         = $current_settings['hide_on_login'] ?? '1';
		$hide_in_admin         = $current_settings['hide_in_admin'] ?? '1';
		$visibility_pages      = $current_settings['visibility_pages'] ?? array();
		$visibility_posts      = $current_settings['visibility_posts'] ?? array();
		$visibility_categories = $current_settings['visibility_categories'] ?? array();
		$visibility_post_types = $current_settings['visibility_post_types'] ?? array();
		$user_agent_enabled    = $current_settings['user_agent_enabled'] ?? '0';
		$user_agent_mode       = $current_settings['user_agent_mode'] ?? 'show_all';
		$user_agent_list       = $current_settings['user_agent_list'] ?? array();

		if ( isset( $raw_settings['visibility_mode'] ) ) {
			$allowed         = array( 'all', 'homepage', 'pages', 'posts', 'categories', 'post_types' );
			$visibility_mode = array_values( array_intersect( array_map( 'sanitize_text_field', (array) $raw_settings['visibility_mode'] ), $allowed ) );
			if ( in_array( 'all', $visibility_mode, true ) ) {
				$visibility_mode = array( 'all' );
			}
			if ( empty( $visibility_mode ) ) {
				$visibility_mode = array( 'all' );
			}
		}
		if ( isset( $raw_settings['device_visibility'] ) ) {
			$dv                = sanitize_text_field( $raw_settings['device_visibility'] );
			$device_visibility = in_array( $dv, array( 'all', 'mobile_only', 'mobile_tablet', 'tablet_only', 'desktop_only' ), true ) ? $dv : 'all';
		}
		if ( isset( $raw_settings['hide_on_login'] ) ) {
			$hide_on_login = ( 'enable' === trim( sanitize_text_field( $raw_settings['hide_on_login'] ) ) ) ? '1' : '0';
		}
		if ( isset( $raw_settings['hide_in_admin'] ) ) {
			$hide_in_admin = ( 'enable' === trim( sanitize_text_field( $raw_settings['hide_in_admin'] ) ) ) ? '1' : '0';
		}
		if ( isset( $raw_settings['visibility_pages'] ) && is_array( $raw_settings['visibility_pages'] ) ) {
			$visibility_pages = array_map( 'absint', $raw_settings['visibility_pages'] );
		}
		if ( isset( $raw_settings['visibility_posts'] ) && is_array( $raw_settings['visibility_posts'] ) ) {
			$visibility_posts = array_map( 'absint', $raw_settings['visibility_posts'] );
		}
		if ( isset( $raw_settings['visibility_categories'] ) && is_array( $raw_settings['visibility_categories'] ) ) {
			$visibility_categories = array_map( 'absint', $raw_settings['visibility_categories'] );
		}
		if ( isset( $raw_settings['visibility_post_types'] ) && is_array( $raw_settings['visibility_post_types'] ) ) {
			$visibility_post_types = array_map( 'sanitize_text_field', $raw_settings['visibility_post_types'] );
		}
		if ( isset( $raw_settings['user_agent_enabled'] ) ) {
			$user_agent_enabled = ( 'enable' === trim( sanitize_text_field( $raw_settings['user_agent_enabled'] ) ) ) ? '1' : '0';
		}
		if ( isset( $raw_settings['user_agent_mode'] ) ) {
			$ua_mode_raw     = sanitize_text_field( $raw_settings['user_agent_mode'] );
			$user_agent_mode = in_array( $ua_mode_raw, array( 'show_all', 'show_only_in_app', 'hide_only_in_app' ), true ) ? $ua_mode_raw : 'show_all';
		}
		if ( isset( $raw_settings['user_agent_list'] ) ) {
			if ( is_array( $raw_settings['user_agent_list'] ) ) {
				$user_agent_list = array_map( 'sanitize_text_field', array_filter( $raw_settings['user_agent_list'] ) );
			} else {
				$user_agent_list = array_map( 'trim', array_filter( preg_split( '/\r\n|\r|\n/', sanitize_textarea_field( $raw_settings['user_agent_list'] ) ) ) );
			}
		}

		return array(
			'visibility_mode'       => $visibility_mode,
			'device_visibility'     => $device_visibility,
			'hide_on_login'         => $hide_on_login,
			'hide_in_admin'         => $hide_in_admin,
			'visibility_pages'      => $visibility_pages,
			'visibility_posts'      => $visibility_posts,
			'visibility_categories' => $visibility_categories,
			'visibility_post_types' => $visibility_post_types,
			'user_agent_enabled'    => $user_agent_enabled,
			'user_agent_mode'       => $user_agent_mode,
			'user_agent_list'       => $user_agent_list,
		);
	}

	/**
	 * Save visibility settings
	 */
	private function save_visibility_settings() {
		if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'arewa_mbn_settings_group-options' ) ) {
			return;
		}
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$current_settings = get_option( 'arewa_mbn_settings', array() );
		$raw_settings     = array();
		if ( ! empty( $_POST['arewa_mbn_settings'] ) && is_array( $_POST['arewa_mbn_settings'] ) ) {
			$raw_settings = map_deep( wp_unslash( $_POST['arewa_mbn_settings'] ), 'sanitize_text_field' );
		}

		$visibility_settings = $this->sanitize_visibility_settings( $raw_settings, $current_settings );

		$settings = array_merge( $current_settings, $visibility_settings );
		update_option( 'arewa_mbn_settings', $settings );

		add_settings_error(
			'arewa_mbn_settings',
			'arewa_mbn_settings_updated',
			__( 'Visibility settings saved successfully.', 'arewa-mobile-bottom-navigation-bar' ),
			'updated'
		);
	}
}
