<?php
/**
 * Visibility Conditions
 *
 * @package Arewa_Mobile_Bottom_Navigation
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Visibility Conditions Class
 */
class Arewa_MBN_Visibility_Conditions {

	/**
	 * Check if navigation should be displayed
	 *
	 * @param array $settings The plugin settings array.
	 * @return bool True if navigation should be displayed, false otherwise.
	 */
	public function should_display( $settings ) {
		// Generate cache key based on current context to prevent cache issues.
		$cache_key = $this->generate_cache_key( $settings );

		// Check if we have a cached result for this specific context.
		// Only use cache if WordPress is fully loaded.
		if ( did_action( 'wp_loaded' ) ) {
			$cached_result = wp_cache_get( $cache_key, 'arewa_mbn_visibility' );
			if ( false !== $cached_result ) {
				return $cached_result;
			}
		}

		// Hide on login page.
		if ( ! empty( $settings['hide_on_login'] ) && '1' === $settings['hide_on_login'] ) {
			if ( is_admin() || ( isset( $GLOBALS['pagenow'] ) && in_array( $GLOBALS['pagenow'], array( 'wp-login.php', 'wp-register.php' ), true ) ) ) {
				if ( did_action( 'wp_loaded' ) ) {
					wp_cache_set( $cache_key, false, 'arewa_mbn_visibility', 300 );
				}
				return false;
			}
		}

		// Hide in admin dashboard.
		if ( ! empty( $settings['hide_in_admin'] ) && '1' === $settings['hide_in_admin'] ) {
			if ( is_admin() ) {
				if ( did_action( 'wp_loaded' ) ) {
					wp_cache_set( $cache_key, false, 'arewa_mbn_visibility', 300 );
				}
				return false;
			}
		}

		// Check User-Agent conditions (WebView/App detection).
		if ( ! $this->check_user_agent_visibility( $settings ) ) {
			if ( did_action( 'wp_loaded' ) ) {
				wp_cache_set( $cache_key, false, 'arewa_mbn_visibility', 300 );
			}
			return false;
		}

		// Check device visibility.
		if ( ! $this->check_device_visibility( $settings ) ) {
			if ( did_action( 'wp_loaded' ) ) {
				wp_cache_set( $cache_key, false, 'arewa_mbn_visibility', 300 );
			}
			return false;
		}

		// Check page/post visibility.
		if ( ! $this->check_page_visibility( $settings ) ) {
			if ( did_action( 'wp_loaded' ) ) {
				wp_cache_set( $cache_key, false, 'arewa_mbn_visibility', 300 );
			}
			return false;
		}

		// Cache positive result.
		if ( did_action( 'wp_loaded' ) ) {
			wp_cache_set( $cache_key, true, 'arewa_mbn_visibility', 300 );
		}
		return true;
	}

	/**
	 * Generate cache key based on current context.
	 *
	 * @param array $settings The plugin settings array.
	 * @return string Cache key for current context.
	 */
	private function generate_cache_key( $settings ) {
		$ua_raw = '';
		if ( ! empty( $_SERVER['HTTP_USER_AGENT'] ) ) {
			$ua_raw = sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) );
		}
		if ( '' === $ua_raw && ! empty( $_SERVER['HTTP_X_USER_AGENT'] ) ) {
			$ua_raw = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_USER_AGENT'] ) );
		}
		if ( '' === $ua_raw && ! empty( $_SERVER['HTTP_X_REQUESTED_USER_AGENT'] ) ) {
			$ua_raw = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_REQUESTED_USER_AGENT'] ) );
		}
		// Include cache version so old entries are orphaned when settings are updated.
		$cache_version = (int) get_option( 'arewa_mbn_visibility_cache_version', 1 );
		// Use simpler context data to avoid WordPress function dependency issues.
		$context_data = array(
			'url'           => isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '',
			'user_agent'    => '' !== $ua_raw ? md5( $ua_raw ) : '',
			'is_admin'      => is_admin(),
			'settings_hash' => md5( wp_json_encode( $settings ) ),
			'cache_version' => $cache_version,
		);

		// Add WordPress query context only if available.
		if ( function_exists( 'is_front_page' ) && did_action( 'wp' ) ) {
			$context_data['wp_context'] = array(
				'is_front_page' => is_front_page(),
				'is_page'       => is_page(),
				'is_single'     => is_single(),
				'is_category'   => is_category(),
				'post_id'       => get_the_ID(),
				'post_type'     => get_post_type(),
			);
		}

		return 'arewa_mbn_visibility_' . md5( wp_json_encode( $context_data ) );
	}

	/**
	 * Check User-Agent visibility conditions for WebView/App detection
	 *
	 * @param array $settings The plugin settings array.
	 * @return bool True if User-Agent conditions are met, false otherwise.
	 */
	private function check_user_agent_visibility( $settings ) {
		// Check if User-Agent conditions are enabled.
		if ( empty( $settings['user_agent_enabled'] ) || '1' !== $settings['user_agent_enabled'] ) {
			return true; // No User-Agent restrictions.
		}

		// Use standard User-Agent; fallback to custom headers (some apps/proxies send UA only there).
		$user_agent = '';
		if ( ! empty( $_SERVER['HTTP_USER_AGENT'] ) ) {
			$user_agent = sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) );
		}
		if ( '' === $user_agent && ! empty( $_SERVER['HTTP_X_USER_AGENT'] ) ) {
			$user_agent = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_USER_AGENT'] ) );
		}
		if ( '' === $user_agent && ! empty( $_SERVER['HTTP_X_REQUESTED_USER_AGENT'] ) ) {
			$user_agent = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_REQUESTED_USER_AGENT'] ) );
		}

		if ( '' === $user_agent ) {
			// If no User-Agent, apply default behavior based on settings.
			return empty( $settings['user_agent_mode'] ) || 'show_all' === $settings['user_agent_mode'];
		}

		$user_agent_mode = isset( $settings['user_agent_mode'] ) ? $settings['user_agent_mode'] : 'show_all';
		// Normalize list: may be stored as array (one per line) or as string.
		$user_agent_list = isset( $settings['user_agent_list'] ) ? $settings['user_agent_list'] : array();
		if ( is_string( $user_agent_list ) ) {
			$user_agent_list = array_map( 'trim', array_filter( preg_split( '/\r\n|\r|\n/', $user_agent_list ) ) );
		}
		if ( ! is_array( $user_agent_list ) ) {
			$user_agent_list = array();
		}

		// If no User-Agents configured, show everywhere.
		if ( empty( $user_agent_list ) ) {
			return true;
		}

		// Check if current User-Agent matches any in the list (partial match, case-insensitive).
		$matches_list     = false;
		$user_agent_lower = strtolower( $user_agent );
		foreach ( $user_agent_list as $ua_pattern ) {
			$ua_pattern = trim( (string) $ua_pattern );
			if ( '' === $ua_pattern ) {
				continue;
			}
			if ( false !== strpos( $user_agent_lower, strtolower( $ua_pattern ) ) ) {
				$matches_list = true;
				break;
			}
		}

		// Apply visibility logic based on mode.
		switch ( $user_agent_mode ) {
			case 'show_only_in_app':
				// Show only if User-Agent matches the list (app/webview).
				return $matches_list;

			case 'hide_only_in_app':
				// Hide only if User-Agent matches the list (hide in app/webview).
				return ! $matches_list;

			case 'show_all':
			default:
				// Show everywhere (no restrictions).
				return true;
		}
	}

	/**
	 * Check device visibility
	 *
	 * @param array $settings The plugin settings array.
	 * @return bool True if device visibility matches, false otherwise.
	 */
	private function check_device_visibility( $settings ) {
		$device_visibility = isset( $settings['device_visibility'] ) ? $settings['device_visibility'] : 'all';

		if ( 'all' === $device_visibility ) {
			return true;
		}

		// Detect device type.
		$is_mobile  = wp_is_mobile();
		$user_agent = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';

		// Simple device detection.
		$is_tablet      = preg_match( '/tablet|ipad|playbook|silk/i', $user_agent );
		$is_mobile_only = $is_mobile && ! $is_tablet;

		switch ( $device_visibility ) {
			case 'mobile_tablet':
				return $is_mobile;
			case 'mobile_only':
				return $is_mobile_only;
			case 'tablet_only':
				return $is_tablet;
			case 'desktop_only':
				return ! $is_mobile;
			default:
				return true;
		}
	}

	/**
	 * Check page visibility
	 *
	 * @param array $settings The plugin settings array.
	 * @return bool True if page visibility matches, false otherwise.
	 */
	private function check_page_visibility( $settings ) {
		// Get visibility modes - support both array (new) and string (old) formats.
		$visibility_mode = isset( $settings['visibility_mode'] ) ? $settings['visibility_mode'] : 'all';

		// Convert to array for consistent handling.
		if ( ! is_array( $visibility_mode ) ) {
			$visibility_mode = ( 'all' === $visibility_mode ) ? array( 'all' ) : array( $visibility_mode );
		}

		// If visibility mode is 'all', show everywhere.
		if ( in_array( 'all', $visibility_mode, true ) ) {
			return true;
		}

		// Check each mode - if ANY mode matches, show navigation (OR logic).
		$matches = false;

		// Check homepage.
		if ( in_array( 'homepage', $visibility_mode, true ) ) {
			if ( is_front_page() ) {
				$matches = true;
			}
		}

		// Check specific pages.
		if ( in_array( 'pages', $visibility_mode, true ) ) {
			$visibility_pages = isset( $settings['visibility_pages'] ) ? $settings['visibility_pages'] : array();
			if ( is_page() && ! empty( $visibility_pages ) ) {
				if ( in_array( get_the_ID(), $visibility_pages, true ) ) {
					$matches = true;
				}
			}
		}

		// Check specific posts.
		if ( in_array( 'posts', $visibility_mode, true ) ) {
			$visibility_posts = isset( $settings['visibility_posts'] ) ? $settings['visibility_posts'] : array();
			if ( is_single() && ! empty( $visibility_posts ) ) {
				if ( in_array( get_the_ID(), $visibility_posts, true ) ) {
					$matches = true;
				}
			}
		}

		// Check categories.
		if ( in_array( 'categories', $visibility_mode, true ) ) {
			$visibility_categories = isset( $settings['visibility_categories'] ) ? $settings['visibility_categories'] : array();
			if ( is_category() && ! empty( $visibility_categories ) ) {
				if ( in_array( get_queried_object_id(), $visibility_categories, true ) ) {
					$matches = true;
				}
			}
			// Also check if current post/page belongs to selected categories.
			if ( is_singular() && ! empty( $visibility_categories ) ) {
				$post_categories = wp_get_post_categories( get_the_ID() );
				if ( ! empty( array_intersect( $post_categories, $visibility_categories ) ) ) {
					$matches = true;
				}
			}
		}

		// Check post types.
		if ( in_array( 'post_types', $visibility_mode, true ) ) {
			$visibility_post_types = isset( $settings['visibility_post_types'] ) ? $settings['visibility_post_types'] : array();
			if ( is_singular() && ! empty( $visibility_post_types ) ) {
				if ( in_array( get_post_type(), $visibility_post_types, true ) ) {
					$matches = true;
				}
			}
		}

		return $matches;
	}
}
