<?php
/**
 * Main plugin class
 *
 * @package Arewa_Mobile_Bottom_Navigation
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main plugin class
 *
 * @package Arewa_Mobile_Bottom_Navigation
 */
class Arewa_Mobile_Bottom_Navigation {

	/**
	 * Instance of this class
	 *
	 * @var Arewa_Mobile_Bottom_Navigation
	 */
	private static $instance = null;

	/**
	 * Get instance of this class
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->init_hooks();
	}

	/**
	 * Initialize hooks
	 */
	private function init_hooks() {
		// Include required files.
		$this->includes();

		// Enqueue scripts and styles.
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_frontend_assets' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );

		// Render frontend navigation.
		add_action( 'wp_footer', array( $this, 'render_navigation' ) );

		// Send Vary: User-Agent early when User-Agent detection is enabled (before any output).
		add_action( 'send_headers', array( $this, 'maybe_send_user_agent_vary_header' ), 1 );

		// Clear visibility cache when settings are updated.
		add_action( 'update_option_arewa_mbn_settings', array( $this, 'clear_visibility_cache' ) );
		add_action( 'update_option_arewa_mbn_buttons', array( $this, 'clear_visibility_cache' ) );
		add_action( 'update_option_arewa_mbn_submenus', array( $this, 'clear_visibility_cache' ) );

		// Register activation and deactivation hooks.
		register_activation_hook( AREWA_MBN_PLUGIN_BASENAME, array( $this, 'activate' ) );
		register_deactivation_hook( AREWA_MBN_PLUGIN_BASENAME, array( $this, 'deactivate' ) );

		// Note: Uninstall is handled by uninstall.php file.
		// The uninstall.php file preserves settings for reactivation.
		// register_uninstall_hook is not needed when uninstall.php exists.
	}

	/**
	 * Normalize SVG for display
	 * Processes SVG code to ensure it works with different formats (Font Awesome, etc.)
	 *
	 * @param string $svg_code The SVG code to normalize.
	 * @return string The normalized SVG code.
	 */
	public static function normalize_svg_for_display( $svg_code ) {
		if ( empty( $svg_code ) ) {
			return '';
		}

		// Remove HTML comments.
		$svg_code = preg_replace( '/<!--.*?-->/s', '', $svg_code );

		// Remove XML processing instructions.
		$svg_code = preg_replace( '/<\?xml[^>]*\?>/i', '', $svg_code );

		// Trim whitespace.
		$svg_code = trim( $svg_code );

		// Extract SVG content if wrapped.
		if ( false === strpos( $svg_code, '<svg' ) ) {
			preg_match( '/<svg[^>]*>.*?<\/svg>/is', $svg_code, $matches );
			if ( ! empty( $matches[0] ) ) {
				$svg_code = $matches[0];
			} else {
				return '';
			}
		}

		// Ensure xmlns is present.
		if ( false === strpos( $svg_code, 'xmlns=' ) ) {
			$svg_code = preg_replace( '/<svg\s+/i', '<svg xmlns="http://www.w3.org/2000/svg" ', $svg_code, 1 );
		}

		// Ensure viewBox is present.
		if ( false === strpos( $svg_code, 'viewBox=' ) && false === strpos( $svg_code, 'viewbox=' ) ) {
			// Try to extract width and height.
			preg_match( '/width=["\']?(\d+)/i', $svg_code, $width_match );
			preg_match( '/height=["\']?(\d+)/i', $svg_code, $height_match );
			if ( ! empty( $width_match[1] ) && ! empty( $height_match[1] ) ) {
				$viewbox  = '0 0 ' . $width_match[1] . ' ' . $height_match[1];
				$svg_code = preg_replace( '/<svg\s+([^>]*)>/i', '<svg $1 viewBox="' . $viewbox . '">', $svg_code, 1 );
			} else {
				// Default viewBox.
				$svg_code = preg_replace( '/<svg\s+([^>]*)>/i', '<svg $1 viewBox="0 0 24 24">', $svg_code, 1 );
			}
		}

		// Remove width and height attributes (controlled via CSS).
		$svg_code = preg_replace( '/\s+(width|height)=["\'][^"\']*["\']/i', '', $svg_code );

		// Replace fill/stroke with currentColor if they have color values (but preserve currentColor, none, and URLs).
		$svg_code = preg_replace_callback(
			'/(<(?:path|circle|rect|line|polyline|polygon|g|ellipse|text|use)[^>]*)\s+(fill|stroke)=["\']([^"\']*)["\']/i',
			function( $matches ) {
				$tag   = $matches[1];
				$attr  = $matches[2];
				$value = strtolower( trim( $matches[3] ) );
				// Don't replace if it's already currentColor, none, or a URL (like gradients).
				if ( 'currentcolor' === $value || 'none' === $value || 0 === strpos( $value, 'url(' ) ) {
					return $matches[0];
				}
				// Replace with currentColor.
				return $tag . ' ' . $attr . '="currentColor"';
			},
			$svg_code
		);

		return $svg_code;
	}

	/**
	 * Sanitize SVG for output (allows SVG elements)
	 *
	 * @param string $svg_code The SVG code to sanitize.
	 * @return string The sanitized SVG code.
	 */
	public static function sanitize_svg_output( $svg_code ) {
		if ( empty( $svg_code ) ) {
			return '';
		}
		return wp_kses( $svg_code, self::get_allowed_svg_kses() );
	}

	/**
	 * Allowed SVG tags and attributes for wp_kses (sanitize and escape output).
	 *
	 * @return array Allowed HTML for SVG.
	 */
	public static function get_allowed_svg_kses() {
		return array(
			'svg'      => array(
				'xmlns'               => true,
				'xmlns:xlink'         => true,
				'viewbox'             => true,
				'viewBox'             => true,
				'width'               => true,
				'height'              => true,
				'fill'                => true,
				'stroke'              => true,
				'class'               => true,
				'id'                  => true,
				'style'               => true,
				'version'             => true,
				'xml:space'           => true,
				'preserveaspectratio' => true,
				'preserveAspectRatio' => true,
			),
			'path'     => array(
				'd'                 => true,
				'fill'              => true,
				'stroke'            => true,
				'stroke-width'      => true,
				'stroke-linecap'    => true,
				'stroke-linejoin'   => true,
				'stroke-miterlimit' => true,
				'class'             => true,
				'id'                => true,
				'style'             => true,
			),
			'circle'   => array(
				'cx'                => true,
				'cy'                => true,
				'r'                 => true,
				'fill'              => true,
				'stroke'            => true,
				'stroke-width'      => true,
				'stroke-linejoin'   => true,
				'stroke-miterlimit' => true,
				'class'             => true,
				'id'                => true,
				'style'             => true,
			),
			'rect'     => array(
				'x'      => true,
				'y'      => true,
				'width'  => true,
				'height' => true,
				'rx'     => true,
				'ry'     => true,
				'fill'   => true,
				'stroke' => true,
				'class'  => true,
				'id'     => true,
				'style'  => true,
			),
			'line'     => array(
				'x1'           => true,
				'y1'           => true,
				'x2'           => true,
				'y2'           => true,
				'stroke'       => true,
				'stroke-width' => true,
				'class'        => true,
				'id'           => true,
				'style'        => true,
			),
			'polyline' => array(
				'points'       => true,
				'fill'         => true,
				'stroke'       => true,
				'stroke-width' => true,
				'class'        => true,
				'id'           => true,
				'style'        => true,
			),
			'polygon'  => array(
				'points'       => true,
				'fill'         => true,
				'stroke'       => true,
				'stroke-width' => true,
				'class'        => true,
				'id'           => true,
				'style'        => true,
			),
			'g'        => array(
				'fill'      => true,
				'stroke'    => true,
				'class'     => true,
				'id'        => true,
				'style'     => true,
				'transform' => true,
			),
			'defs'     => array(),
			'use'      => array(
				'href'       => true,
				'xlink:href' => true,
				'x'          => true,
				'y'          => true,
				'class'      => true,
				'id'         => true,
			),
			'ellipse'  => array(
				'cx'     => true,
				'cy'     => true,
				'rx'     => true,
				'ry'     => true,
				'fill'   => true,
				'stroke' => true,
				'class'  => true,
				'id'     => true,
				'style'  => true,
			),
			'text'     => array(
				'x'      => true,
				'y'      => true,
				'fill'   => true,
				'stroke' => true,
				'class'  => true,
				'id'     => true,
				'style'  => true,
			),
		);
	}

	/**
	 * Include required files
	 */
	private function includes() {
		require_once AREWA_MBN_PLUGIN_DIR . 'includes/class-arewa-mbn-admin-settings.php';
		require_once AREWA_MBN_PLUGIN_DIR . 'includes/class-arewa-mbn-button-manager.php';
		require_once AREWA_MBN_PLUGIN_DIR . 'includes/class-arewa-mbn-submenu-manager.php';
		require_once AREWA_MBN_PLUGIN_DIR . 'includes/class-arewa-mbn-visibility-conditions.php';

		// Initialize admin settings.
		if ( is_admin() ) {
			new Arewa_MBN_Admin_Settings();
		}
	}


	/**
	 * Enqueue frontend assets
	 */
	public function enqueue_frontend_assets() {
		// Check if navigation should be displayed.
		if ( ! $this->should_display_navigation() ) {
			return;
		}

		// Get file modification time to prevent caching.
		$css_file    = AREWA_MBN_PLUGIN_DIR . 'assets/css/frontend.css';
		$js_file     = AREWA_MBN_PLUGIN_DIR . 'assets/js/frontend.js';
		$css_version = file_exists( $css_file ) ? filemtime( $css_file ) : AREWA_MBN_VERSION;
		$js_version  = file_exists( $js_file ) ? filemtime( $js_file ) : AREWA_MBN_VERSION;

		wp_enqueue_style(
			'arewa-mbn-frontend',
			AREWA_MBN_PLUGIN_URL . 'assets/css/frontend.css',
			array(),
			$css_version
		);

		wp_enqueue_script(
			'arewa-mbn-frontend',
			AREWA_MBN_PLUGIN_URL . 'assets/js/frontend.js',
			array(),
			$js_version,
			true
		);

		// Localize script.
		wp_localize_script(
			'arewa-mbn-frontend',
			'arewaMBN',
			array(
				'ajaxurl' => admin_url( 'admin-ajax.php' ),
				'nonce'   => wp_create_nonce( 'arewa_mbn_nonce' ),
			)
		);
	}

	/**
	 * Enqueue admin assets
	 *
	 * @param string $hook The current admin page hook.
	 */
	public function enqueue_admin_assets( $hook ) {
		// Only load on our settings pages.
		// The hook for our menu page is: toplevel_page_arewa-mobile-bottom-navigation-bar.
		if ( 'toplevel_page_arewa-mobile-bottom-navigation-bar' !== $hook && false === strpos( $hook, 'arewa-mobile-bottom' ) && false === strpos( $hook, 'arewa-mbn' ) ) {
			return;
		}

		// Get file modification time to prevent caching.
		$admin_css_file    = AREWA_MBN_PLUGIN_DIR . 'assets/css/admin.css';
		$admin_js_file     = AREWA_MBN_PLUGIN_DIR . 'assets/js/admin.js';
		$admin_css_version = file_exists( $admin_css_file ) ? filemtime( $admin_css_file ) : AREWA_MBN_VERSION;
		$admin_js_version  = file_exists( $admin_js_file ) ? filemtime( $admin_js_file ) : AREWA_MBN_VERSION;

		// Enqueue WordPress color picker and its dependencies (required for color inputs).
		wp_enqueue_style( 'wp-color-picker' );
		wp_enqueue_script( 'iris' );
		wp_enqueue_script( 'wp-color-picker' );

		wp_enqueue_style(
			'arewa-mbn-admin',
			AREWA_MBN_PLUGIN_URL . 'assets/css/admin.css',
			array( 'wp-color-picker' ),
			$admin_css_version
		);

		wp_enqueue_script(
			'arewa-mbn-admin',
			AREWA_MBN_PLUGIN_URL . 'assets/js/admin.js',
			array( 'jquery', 'jquery-ui-sortable', 'wp-color-picker', 'iris' ),
			$admin_js_version,
			false
		);

		// Enqueue media uploader.
		wp_enqueue_media();

		// New button HTML template (passed from PHP to avoid HTML-in-JS lint/phpcs spacing errors).
		$new_button_template = $this->get_new_button_html_template();

		// Localize script.
		wp_localize_script(
			'arewa-mbn-admin',
			'arewaMBNAdmin',
			array(
				'ajaxurl'           => admin_url( 'admin-ajax.php' ),
				'nonce'             => wp_create_nonce( 'arewa_mbn_admin_nonce' ),
				'newButtonTemplate' => $new_button_template,
			)
		);
	}

	/**
	 * Returns the HTML template for a new button (used by admin.js addNewButton).
	 * Placeholders: {{buttonId}}, {{index}}, {{idx}}.
	 *
	 * @return string HTML template string.
	 */
	private function get_new_button_html_template() {
		$svg_placeholder = esc_attr( '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M10 20v-6h4v6h5v-8h3L12 3 2 12h3v8z"/></svg>' );
		return '<div class="arewa-mbn-button-item" data-button-id="{{buttonId}}">'
			. '<div class="button-handle"><span class="dashicons dashicons-menu"></span>Button {{index}}</div>'
			. '<div class="button-content">'
			. '<input type="hidden" name="arewa_mbn_buttons[{{idx}}][id]" value="{{buttonId}}">'
			. '<input type="hidden" name="arewa_mbn_buttons[{{idx}}][order]" value="{{index}}" class="button-order">'
			. '<table class="form-table">'
			. '<tr><th><label>Label</label></th><td><input type="text" name="arewa_mbn_buttons[{{idx}}][label]" value="" class="regular-text"></td></tr>'
			. '<tr><th><label>SVG Icon Code</label></th><td><textarea name="arewa_mbn_buttons[{{idx}}][icon]" rows="5" class="large-text code" placeholder="' . $svg_placeholder . '"></textarea><p class="description">Paste your SVG code here.</p></td></tr>'
			. '<tr><th><label>Icon Color</label></th><td><input type="color" name="arewa_mbn_buttons[{{idx}}][icon_color]" value="#333333" class="arewa-mbn-native-color" aria-label="Icon Color"></td></tr>'
			. '<tr><th><label>Text Color</label></th><td><input type="color" name="arewa_mbn_buttons[{{idx}}][text_color]" value="#333333" class="arewa-mbn-native-color" aria-label="Text Color"></td></tr>'
			. '<tr><th><label>Background Color</label></th><td><input type="text" name="arewa_mbn_buttons[{{idx}}][bg_color]" value="" class="arewa-mbn-bg-color-value" id="arewa-mbn-bg-{{idx}}"><input type="color" class="arewa-mbn-bg-color-picker" value="#ffffff" data-target="#arewa-mbn-bg-{{idx}}" aria-label="Background Color"><button type="button" class="button arewa-mbn-bg-transparent" data-target="#arewa-mbn-bg-{{idx}}">Transparent</button></td></tr>'
			. '<tr><th><label>Active State Color</label></th><td><input type="color" name="arewa_mbn_buttons[{{idx}}][active_color]" value="#0073aa" class="arewa-mbn-native-color" aria-label="Active State Color"></td></tr>'
			. '<tr class="url-row"><th><label>URL</label></th><td><input type="url" name="arewa_mbn_buttons[{{idx}}][url]" value="" class="regular-text"><p class="description">URL is not needed when submenu is enabled</p></td></tr>'
			. '<tr class="target-row"><th><label>Link Target</label></th><td><select name="arewa_mbn_buttons[{{idx}}][target]"><option value="_self">Same Tab</option><option value="_blank">New Tab</option></select></td></tr>'
			. '<tr><th><label>Has Submenu</label></th><td><select name="arewa_mbn_buttons[{{idx}}][has_submenu]" class="has-submenu-select arewa-mbn-enable-disable-select" data-button-index="{{idx}}"><option value="disable">Disable</option><option value="enable">Enable</option></select><p class="description">Enable or disable submenu for this button. When enabled, this button will open a submenu instead of navigating to a URL.</p></td></tr>'
			. '</table></div></div>';
	}

	/**
	 * Check if navigation should be displayed
	 */
	private function should_display_navigation() {
		$settings = get_option( 'arewa_mbn_settings', array() );

		// Check if plugin is enabled.
		if ( empty( $settings['enabled'] ) || '1' !== $settings['enabled'] ) {
			return false;
		}

		// Check visibility conditions.
		$visibility = new Arewa_MBN_Visibility_Conditions();
		$result     = $visibility->should_display( $settings );

		// Add cache-busting headers to prevent caching issues.
		if ( ! is_admin() && ! wp_doing_ajax() ) {
			$this->add_cache_busting_headers();
		}

		return $result;
	}

	/**
	 * Send Vary: User-Agent header early when output varies by User-Agent.
	 * Ensures full-page caches (CDN, plugins) create separate entries for mobile vs desktop,
	 * preventing the nav from disappearing when cache serves wrong version.
	 */
	public function maybe_send_user_agent_vary_header() {
		if ( headers_sent() || is_admin() || wp_doing_ajax() ) {
			return;
		}
		$settings = get_option( 'arewa_mbn_settings', array() );
		// Skip if plugin is disabled.
		if ( empty( $settings['enabled'] ) || '1' !== $settings['enabled'] ) {
			return;
		}
		$needs_vary = false;
		// User-Agent detection: output varies by User-Agent.
		if ( ! empty( $settings['user_agent_enabled'] ) && '1' === $settings['user_agent_enabled'] ) {
			$needs_vary = true;
		}
		// Device visibility (mobile/tablet): wp_is_mobile() uses User-Agent, so output varies.
		$device_visibility = isset( $settings['device_visibility'] ) ? $settings['device_visibility'] : 'all';
		if ( in_array( $device_visibility, array( 'mobile_only', 'mobile_tablet', 'tablet_only' ), true ) ) {
			$needs_vary = true;
		}
		if ( $needs_vary ) {
			header( 'Vary: User-Agent', true );
		}
	}

	/**
	 * Add cache-busting headers to prevent visibility condition caching.
	 */
	private function add_cache_busting_headers() {
		// Only add headers if not already sent and not in admin.
		if ( ! headers_sent() && ! is_admin() && ! wp_doing_ajax() ) {
			// Add Vary header to ensure different User-Agents get different cached versions.
			if ( ! headers_sent() ) {
				header( 'Vary: User-Agent', false );

				// Add a custom header to indicate dynamic content.
				header( 'X-Arewa-MBN-Dynamic: 1', false );
			}
		}
	}

	/**
	 * Clear visibility cache and purge full-page cache when settings are updated.
	 * Prevents nav from disappearing when cached HTML serves wrong device version.
	 */
	public function clear_visibility_cache() {
		// Bump visibility cache version so old visibility entries are orphaned.
		$version = (int) get_option( 'arewa_mbn_visibility_cache_version', 1 );
		update_option( 'arewa_mbn_visibility_cache_version', $version + 1 );

		// Purge popular full-page cache plugins so fresh pages are generated.
		$this->purge_page_cache();
	}

	/**
	 * Purge full-page cache from popular caching plugins.
	 * Ensures fresh pages are generated after settings change, preventing stale cached pages without nav.
	 */
	private function purge_page_cache() {
		// LiteSpeed Cache.
		do_action( 'litespeed_purge_all' );
		// WP Rocket.
		if ( function_exists( 'rocket_clean_domain' ) ) {
			rocket_clean_domain();
		}
		// W3 Total Cache.
		if ( function_exists( 'w3tc_flush_all' ) ) {
			w3tc_flush_all();
		}
		// WP Super Cache.
		if ( function_exists( 'wp_cache_clear_cache' ) ) {
			wp_cache_clear_cache();
		}
		// WP Fastest Cache.
		if ( class_exists( 'WpFastestCache' ) && isset( $GLOBALS['wp_fastest_cache'] ) ) {
			$GLOBALS['wp_fastest_cache']->deleteCache();
		}
		// SG Optimizer (SiteGround).
		if ( function_exists( 'sg_cachepress_purge_everything' ) ) {
			sg_cachepress_purge_everything();
		}
		// Cache Enabler.
		if ( function_exists( 'ce_purge_complete_cache' ) ) {
			ce_purge_complete_cache();
		}
		// Allow themes and other plugins to purge their cache.
		do_action( 'arewa_mbn_purge_page_cache' );
	}

	/**
	 * Render navigation bar
	 */
	public function render_navigation() {
		if ( ! $this->should_display_navigation() ) {
			return;
		}

		$settings = get_option( 'arewa_mbn_settings', array() );
		$buttons  = get_option( 'arewa_mbn_buttons', array() );
		$submenus = get_option( 'arewa_mbn_submenus', array() );

		// Get current URL for active state.
		$http_host   = isset( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : '';
		$request_uri = isset( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
		$current_url = ( is_ssl() ? 'https://' : 'http://' ) . $http_host . $request_uri;
		$current_url = trailingslashit( $current_url );

		include AREWA_MBN_PLUGIN_DIR . 'templates/navigation.php';
	}

	/**
	 * Plugin activation
	 * Restores previous settings if they exist, otherwise sets defaults
	 */
	public function activate() {
		// First, try to restore from backup options (if plugin was uninstalled).
		$backup_settings = get_option( 'arewa_mbn_settings_backup', false );
		$backup_buttons  = get_option( 'arewa_mbn_buttons_backup', false );
		$backup_submenus = get_option( 'arewa_mbn_submenus_backup', false );

		// Restore settings from backup if they exist.
		if ( false !== $backup_settings ) {
			update_option( 'arewa_mbn_settings', $backup_settings );
		}

		if ( false !== $backup_buttons ) {
			update_option( 'arewa_mbn_buttons', $backup_buttons );
		}

		if ( false !== $backup_submenus ) {
			update_option( 'arewa_mbn_submenus', $backup_submenus );
		}

		// Check if original options exist (plugin was deactivated, not uninstalled).
		$existing_settings = get_option( 'arewa_mbn_settings', false );
		$existing_buttons  = get_option( 'arewa_mbn_buttons', false );
		$existing_submenus = get_option( 'arewa_mbn_submenus', false );

		// If settings exist (from deactivation), keep them - no need to restore.
		// Only set defaults if no settings exist at all.
		if ( false === $existing_settings && false === $backup_settings ) {
			// Set default settings.
			$default_settings = array(
				'enabled'                          => '1',
				'button_count'                     => 4,
				'layout_style'                     => 'icon_text',
				'background_color'                 => '#ffffff',
				'border_radius'                    => '0',
				'box_shadow'                       => '1',
				'active_color'                     => '#0073aa',
				'hover_effect'                     => '1',
				'pulse_animation'                  => '1',
				'submenu_bg_color'                 => '#1e1e28',
				'submenu_bg_opacity'               => 95,
				'submenu_shadow'                   => '1',
				'submenu_width'                    => 320,
				'submenu_text_color'               => '#ffffff',
				'submenu_icon_spacing'             => 8,
				'submenu_shadow_color'             => '#000000',
				'button_text_size'                 => 12,
				'button_icon_size'                 => 20,
				'submenu_text_size'                => 14,
				'submenu_icon_size'                => 22,
				'submenu_header_enable'            => '1',
				'submenu_header_logo'              => '',
				'submenu_header_logo_id'           => 0,
				'submenu_header_logo_size'         => 120,
				'submenu_header_title'             => get_bloginfo( 'name' ),
				'submenu_header_description'       => '',
				'submenu_header_bg_color'          => 'transparent',
				'submenu_header_title_color'       => '#ffffff',
				'submenu_header_title_size'        => 18,
				'submenu_header_description_color' => 'rgba(255, 255, 255, 0.7)',
				'submenu_header_description_size'  => 13,
				'visibility_mode'                  => array( 'all' ),
				'device_visibility'                => 'all',
				'visibility_pages'                 => array(),
				'visibility_posts'                 => array(),
				'visibility_categories'            => array(),
				'visibility_post_types'            => array(),
				'hide_on_login'                    => '1',
				'hide_in_admin'                    => '1',
				'user_agent_enabled'               => '0',
				'user_agent_mode'                  => 'show_all',
				'user_agent_list'                  => array(),
			);
			add_option( 'arewa_mbn_settings', $default_settings );
		}

		// Set default buttons only if they don't exist.
		if ( false === $existing_buttons && false === $backup_buttons ) {
			$default_buttons = array(
				array(
					'id'           => 'btn_1',
					'label'        => __( 'Home', 'arewa-mobile-bottom-navigation-bar' ),
					'icon'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor"><path d="M10 20v-6h4v6h5v-8h3L12 3 2 12h3v8z"/></svg>',
					'icon_color'   => '#333333',
					'text_color'   => '#333333',
					'bg_color'     => 'transparent',
					'active_color' => '#0073aa',
					'url'          => home_url( '/' ),
					'target'       => '_self',
					'has_submenu'  => '0',
					'order'        => 1,
				),
				array(
					'id'           => 'btn_2',
					'label'        => __( 'About', 'arewa-mobile-bottom-navigation-bar' ),
					'icon'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-6h2v6zm0-8h-2V7h2v2z"/></svg>',
					'icon_color'   => '#333333',
					'text_color'   => '#333333',
					'bg_color'     => 'transparent',
					'active_color' => '#0073aa',
					'url'          => home_url( '/about/' ),
					'target'       => '_self',
					'has_submenu'  => '0',
					'order'        => 2,
				),
				array(
					'id'           => 'btn_3',
					'label'        => __( 'Blog', 'arewa-mobile-bottom-navigation-bar' ),
					'icon'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor"><path d="M20 4H4c-1.1 0-2 .9-2 2v12c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V6c0-1.1-.9-2-2-2zm0 4l-8 5-8-5V6l8 5 8-5v2z"/></svg>',
					'icon_color'   => '#333333',
					'text_color'   => '#333333',
					'bg_color'     => 'transparent',
					'active_color' => '#0073aa',
					'url'          => home_url( '/blog/' ),
					'target'       => '_self',
					'has_submenu'  => '0',
					'order'        => 3,
				),
				array(
					'id'           => 'btn_4',
					'label'        => __( 'Contact', 'arewa-mobile-bottom-navigation-bar' ),
					'icon'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor"><path d="M20 4H4c-1.1 0-1.99.9-1.99 2L2 18c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V6c0-1.1-.9-2-2-2zm0 4l-8 5-8-5V6l8 5 8-5v2z"/></svg>',
					'icon_color'   => '#333333',
					'text_color'   => '#333333',
					'bg_color'     => 'transparent',
					'active_color' => '#0073aa',
					'url'          => home_url( '/contact/' ),
					'target'       => '_self',
					'has_submenu'  => '0',
					'order'        => 4,
				),
			);
			add_option( 'arewa_mbn_buttons', $default_buttons );
		}

		// Set default submenus only if they don't exist.
		if ( false === $existing_submenus && false === $backup_submenus ) {
			add_option( 'arewa_mbn_submenus', array() );
		}

		// Flush rewrite rules if needed.
		flush_rewrite_rules();
	}

	/**
	 * Plugin deactivation
	 * Settings are automatically preserved by WordPress - no action needed
	 */
	public function deactivate() {
		// Settings are automatically preserved when plugin is deactivated.
		// WordPress keeps all options in the database.
		// No cleanup needed - settings will be available when plugin is reactivated.
	}

	/**
	 * Plugin uninstall
	 * Note: This method is only called if uninstall.php doesn't exist.
	 * Since we have uninstall.php, this serves as a fallback.
	 */
	public static function uninstall() {
		// Preserve settings instead of deleting them.
		// This allows users to reactivate without losing configuration.
		// The uninstall.php file handles the actual uninstall process.
		// This method is kept as a fallback for compatibility.
	}
}
