<?php
/**
 * Fired when the plugin is uninstalled.
 *
 * This file preserves all plugin settings and data when the plugin is uninstalled,
 * allowing users to reactivate or reinstall the plugin without losing their configuration.
 *
 * @package Arewa_Mobile_Bottom_Navigation
 */

// If uninstall not called from WordPress, then exit.
if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
	exit;
}

/**
 * Preserve plugin settings and data
 *
 * Instead of deleting the options, we preserve them so they can be restored
 * when the plugin is reactivated or reinstalled.
 */
function arewa_mbn_preserve_settings() {
	// Get all current plugin options.
	$settings = get_option( 'arewa_mbn_settings', false );
	$buttons  = get_option( 'arewa_mbn_buttons', false );
	$submenus = get_option( 'arewa_mbn_submenus', false );

	// Create backup options with a different prefix that won't be deleted.
	// These will persist even after plugin uninstall.
	if ( false !== $settings ) {
		// Store settings in a backup option.
		update_option( 'arewa_mbn_settings_backup', $settings );
		update_option( 'arewa_mbn_settings_backup_time', current_time( 'mysql' ) );
	}

	if ( false !== $buttons ) {
		// Store buttons in a backup option.
		update_option( 'arewa_mbn_buttons_backup', $buttons );
		update_option( 'arewa_mbn_buttons_backup_time', current_time( 'mysql' ) );
	}

	if ( false !== $submenus ) {
		// Store submenus in a backup option.
		update_option( 'arewa_mbn_submenus_backup', $submenus );
		update_option( 'arewa_mbn_submenus_backup_time', current_time( 'mysql' ) );
	}

	// Also keep the original options (don't delete them).
	// This ensures settings are preserved even if backup fails.
	// WordPress will keep these options in the database.
}

/**
 * Option 1: Keep all settings (Recommended)
 * This preserves all settings without deleting anything.
 */
function arewa_mbn_uninstall_keep_settings() {
	// Simply preserve settings - don't delete anything.
	arewa_mbn_preserve_settings();

	// The original options will remain in the database.
	// They will be automatically restored when the plugin is reactivated.
}

/**
 * Option 2: Backup and delete (Alternative approach)
 * This creates backups and then deletes original options.
 * Use this if you want to clean up but still preserve data.
 */
function arewa_mbn_uninstall_backup_and_delete() {
	// First, preserve settings in backup options.
	arewa_mbn_preserve_settings();

	// Then delete original options (they can be restored from backup).
	delete_option( 'arewa_mbn_settings' );
	delete_option( 'arewa_mbn_buttons' );
	delete_option( 'arewa_mbn_submenus' );

	// Note: The backup options will remain and can be restored on activation.
}

// Execute the preservation function.
// We use Option 1 (keep settings) as it's the safest approach.
arewa_mbn_uninstall_keep_settings();

// If you prefer Option 2 (backup and delete), uncomment the line below and comment the line above.
// arewa_mbn_uninstall_backup_and_delete(); // Alternative uninstall method.

/**
 * Optional: Clean up backup options after a period.
 * To implement, add a function arewa_mbn_cleanup_old_backups() that deletes
 * arewa_mbn_settings_backup and arewa_mbn_buttons_backup when backups are older than 90 days,
 * and call it from a cron or admin action.
 */
