<?php
/**
 * Plugin Name: Arewa Recently Viewed Content
 *
 * phpcs:ignoreFile WordPress.Files.FileName.InvalidClassFileName -- Main plugin file must keep slug name.
 * Plugin URI: https://arewa.dev/arewa-recently-viewed-content/
 * Description: Track and display recently viewed content for both logged-in and guest users with automatic history sync and multiple layout options.
 * Version: 2.0.6
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * Author: arewadev
 * Author URI: https://arewa.dev/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: arewa-recently-viewed-content
 * Domain Path: /languages
 *
 * @package Arewa_Recently_Viewed_Content
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Ensure WordPress functions are available.
if ( ! function_exists( 'add_action' ) ) {
	return;
}

// Define plugin constants.
if ( ! defined( 'ARWREV_VERSION' ) ) {
	define( 'ARWREV_VERSION', '2.0.6' );
}
if ( ! defined( 'ARWREV_PLUGIN_DIR' ) ) {
	define( 'ARWREV_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
}
if ( ! defined( 'ARWREV_PLUGIN_URL' ) ) {
	define( 'ARWREV_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
}
if ( ! defined( 'ARWREV_PLUGIN_FILE' ) ) {
	define( 'ARWREV_PLUGIN_FILE', __FILE__ );
}

/**
 * One-time migration: copy arwrev_* options to arwrev_* for backward compatibility.
 */
function arwrev_migrate_options_from_arwrev() {
	if ( get_option( 'arwrev_options_migrated', false ) ) {
		return;
	}
	$options_to_migrate = array(
		'arwrev_settings'                  => 'arwrev_settings',
		'arwrev_version'                   => 'arwrev_version',
		'arwrev_data_preserved'           => 'arwrev_data_preserved',
		'arwrev_show_restoration_notice'  => 'arwrev_show_restoration_notice',
		'arwrev_preserve_data_on_uninstall' => 'arwrev_preserve_data_on_uninstall',
		'arwrev_last_activation'          => 'arwrev_last_activation',
		'arwrev_migration_status'         => 'arwrev_migration_status',
	);
	foreach ( $options_to_migrate as $old_key => $new_key ) {
		$value = get_option( $old_key, null );
		if ( null !== $value ) {
			update_option( $new_key, $value );
		}
	}
	update_option( 'arwrev_options_migrated', true );
}

/**
 * Main plugin class
 */
class ARWREV_Plugin {

	/**
	 * Instance of this class
	 *
	 * @var object
	 */
	private static $instance = null;

	/**
	 * Get instance of this class
	 *
	 * @return ARWREV_Plugin
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->init();
	}

	/**
	 * Initialize plugin
	 */
	private function init() {
		// Load text domain for translations.
		add_action( 'init', array( $this, 'load_textdomain' ) );

		// Load plugin files.
		$this->load_dependencies();

		// Initialize classes.
		$this->init_classes();

		// Register custom image size for better quality.
		add_action( 'after_setup_theme', array( $this, 'register_image_size' ) );

		// Load assets with high priority to ensure they load after theme styles.
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_frontend_assets' ), 999 );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
		// Alternative hook for shortcode-triggered loading.
		add_action( 'arwrev_enqueue_assets', array( $this, 'enqueue_frontend_assets' ) );
	}

	/**
	 * Load plugin text domain for translations
	 *
	 * Note: Since WordPress 4.6+, translations are automatically loaded
	 * for plugins hosted on WordPress.org. No manual loading required.
	 */
	public function load_textdomain() {
		// WordPress 4.6+ automatically loads translations for plugins hosted on WordPress.org.
		// No manual loading required for modern WordPress versions.
	}

	/**
	 * Register custom image size for watch history
	 */
	public function register_image_size() {
		$settings     = get_option( 'arwrev_settings', array() );
		$image_width  = isset( $settings['image_width'] ) ? (int) $settings['image_width'] : 155;
		$image_height = isset( $settings['image_height'] ) ? (int) $settings['image_height'] : 250;

		// Register at 2x size for retina/HD displays.
		add_image_size( 'arwrev_thumb', $image_width * 2, $image_height * 2, true );
	}

	/**
	 * Load required files
	 */
	private function load_dependencies() {
		require_once ARWREV_PLUGIN_DIR . 'includes/class-arwrev-tracker.php';
		require_once ARWREV_PLUGIN_DIR . 'includes/class-arwrev-admin-settings-tabbed.php';
		require_once ARWREV_PLUGIN_DIR . 'includes/class-arwrev-shortcode-new.php';
		require_once ARWREV_PLUGIN_DIR . 'includes/class-arwrev-ajax.php';
		require_once ARWREV_PLUGIN_DIR . 'includes/class-arwrev-layout-manager.php';
		require_once ARWREV_PLUGIN_DIR . 'includes/class-arwrev-data-manager.php';

		// Load layout classes.
		require_once ARWREV_PLUGIN_DIR . 'includes/layouts/class-arwrev-base-layout.php';
		require_once ARWREV_PLUGIN_DIR . 'includes/layouts/class-arwrev-video-layout.php';
		require_once ARWREV_PLUGIN_DIR . 'includes/layouts/class-arwrev-article-layout.php';
		require_once ARWREV_PLUGIN_DIR . 'includes/layouts/class-arwrev-shop-layout.php';
		require_once ARWREV_PLUGIN_DIR . 'includes/layouts/class-arwrev-carousel-layout.php';
	}

	/**
	 * Initialize classes
	 */
	private function init_classes() {
		// Initialize admin settings only in admin.
		if ( is_admin() ) {
			ARWREV_Admin_Settings_Tabbed::get_instance();
		}

		// Initialize tracker only in frontend (not during AJAX).
		if ( ! is_admin() && ! wp_doing_ajax() ) {
			ARWREV_Tracker::get_instance();
		}

		// Initialize shortcode (used in frontend).
		ARWREV_Shortcode_New::get_instance();

		// Initialize AJAX handlers (always needed).
		ARWREV_Ajax::get_instance();

		// Initialize data manager (always needed for restoration checks).
		ARWREV_Data_Manager::get_instance();

		// Initialize layout manager after init to avoid translation issues.
		add_action( 'init', array( $this, 'init_layout_manager' ) );

		// Add paged query var for pagination.
		add_filter( 'query_vars', array( $this, 'add_query_vars' ) );
	}

	/**
	 * Initialize layout manager (delayed)
	 */
	public function init_layout_manager() {
		ARWREV_Layout_Manager::get_instance();
	}

	/**
	 * Add custom query vars.
	 *
	 * @param array $vars Query vars.
	 * @return array
	 */
	public function add_query_vars( $vars ) {
		$vars[] = 'paged';
		$vars[] = 'wh_page';
		return $vars;
	}

	/**
	 * Enqueue frontend assets
	 */
	public function enqueue_frontend_assets() {
		// Prevent multiple enqueues.
		static $enqueued = false;
		if ( $enqueued ) {
			return;
		}
		$enqueued = true;

		// Check if CSS file exists.
		$css_file = ARWREV_PLUGIN_DIR . 'assets/css/frontend.css';

		if ( file_exists( $css_file ) ) {
			wp_enqueue_style(
				'arwrev-frontend',
				ARWREV_PLUGIN_URL . 'assets/css/frontend.css',
				array(),
				ARWREV_VERSION . '.' . filemtime( $css_file ),
				'all'
			);

			// Add inline CSS to ensure styles are applied with proper specificity.
			$this->add_inline_styles();
		}

		// Check if JS file exists.
		$js_file = ARWREV_PLUGIN_DIR . 'assets/js/frontend.js';

		if ( file_exists( $js_file ) ) {
			wp_enqueue_script(
				'arwrev-frontend',
				ARWREV_PLUGIN_URL . 'assets/js/frontend.js',
				array( 'jquery' ),
				ARWREV_VERSION . '.' . filemtime( $js_file ),
				true
			);

			// Get settings for tracked post types.
			$settings = get_option( 'arwrev_settings', array() );
			$tracked_post_types = isset( $settings['post_types'] ) ? $settings['post_types'] : array();
			
			wp_localize_script(
				'arwrev-frontend',
				'arwrev',
				array(
					'ajaxurl'            => admin_url( 'admin-ajax.php' ),
					'nonce'              => wp_create_nonce( 'arwrev_nonce' ),
					'is_user_logged_in'  => is_user_logged_in(),
					'current_post_id'    => is_singular() ? get_the_ID() : null,
					'current_post_type'  => is_singular() ? get_post_type() : null,
					'cookie_path'        => COOKIEPATH,
					'cookie_domain'      => COOKIE_DOMAIN,
					'tracked_post_types' => $tracked_post_types,
				)
			);
		}
	}

	/**
	 * Add inline styles to ensure proper specificity.
	 * Grid gap is configurable via settings; output once to avoid duplicate inline CSS.
	 */
	private function add_inline_styles() {
		$settings = get_option( 'arwrev_settings', array() );

		$columns_desktop = isset( $settings['columns_desktop'] ) ? absint( $settings['columns_desktop'] ) : 6;
		$columns_tablet  = isset( $settings['columns_tablet'] ) ? absint( $settings['columns_tablet'] ) : 4;
		$columns_mobile  = isset( $settings['columns_mobile'] ) ? absint( $settings['columns_mobile'] ) : 2;
		$gap_desktop     = isset( $settings['gap_desktop'] ) ? absint( $settings['gap_desktop'] ) : 11;
		$gap_tablet      = isset( $settings['gap_tablet'] ) ? absint( $settings['gap_tablet'] ) : 9;
		$gap_mobile      = isset( $settings['gap_mobile'] ) ? absint( $settings['gap_mobile'] ) : 8;
		$image_width     = isset( $settings['image_width'] ) ? absint( $settings['image_width'] ) : 155;
		$image_height    = isset( $settings['image_height'] ) ? absint( $settings['image_height'] ) : 250;

		// Clamp columns to valid ranges.
		$columns_desktop = max( 1, min( 6, $columns_desktop ) );
		$columns_tablet  = max( 1, min( 4, $columns_tablet ) );
		$columns_mobile  = max( 1, min( 2, $columns_mobile ) );
		// Gap in px (reasonable range 0–100).
		$gap_desktop = max( 0, min( 100, $gap_desktop ) );
		$gap_tablet  = max( 0, min( 100, $gap_tablet ) );
		$gap_mobile  = max( 0, min( 100, $gap_mobile ) );

		$inline_css = "
			/* Grid columns and gap (configurable); used by Video, Article, Shop, Carousel layouts */
			body .arwrev-container {
				--arwrev-gap-desktop: {$gap_desktop}px;
				--arwrev-gap-tablet: {$gap_tablet}px;
				--arwrev-gap-mobile: {$gap_mobile}px;
			}
			body .arwrev-container .arwrev-grid {
				--columns-desktop: {$columns_desktop};
				--columns-tablet: {$columns_tablet};
				--columns-mobile: {$columns_mobile};
				display: grid;
				grid-template-columns: repeat({$columns_desktop}, 1fr);
				gap: var(--arwrev-gap-desktop);
			}
			body .arwrev-container .arwrev-item {
				--image-width: {$image_width}px;
				--image-height: {$image_height}px;
			}
			body .arwrev-container .arwrev-item-image {
				height: {$image_height}px;
			}
			@media (max-width: 1023px) {
				body .arwrev-container .arwrev-grid {
					grid-template-columns: repeat({$columns_tablet}, 1fr);
					gap: var(--arwrev-gap-tablet);
				}
			}
			@media (max-width: 767px) {
				body .arwrev-container .arwrev-grid {
					grid-template-columns: repeat({$columns_mobile}, 1fr);
					gap: var(--arwrev-gap-mobile);
				}
			}
		";

		wp_add_inline_style( 'arwrev-frontend', $inline_css );
	}

	/**
	 * Enqueue admin assets.
	 *
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_admin_assets( $hook ) {
		if ( 'toplevel_page_ar-viewed-content' !== $hook ) {
			return;
		}

		$admin_css = ARWREV_PLUGIN_DIR . 'assets/css/admin.css';
		$admin_js  = ARWREV_PLUGIN_DIR . 'assets/js/admin-settings.js';

		wp_enqueue_style(
			'arwrev-admin',
			ARWREV_PLUGIN_URL . 'assets/css/admin.css',
			array(),
			ARWREV_VERSION . ( file_exists( $admin_css ) ? '.' . filemtime( $admin_css ) : '' )
		);

		if ( file_exists( $admin_js ) ) {
			wp_enqueue_script(
				'arwrev-admin-settings',
				ARWREV_PLUGIN_URL . 'assets/js/admin-settings.js',
				array(),
				ARWREV_VERSION . '.' . filemtime( $admin_js ),
				true
			);
			wp_localize_script(
				'arwrev-admin-settings',
				'arwrevAdminSettings',
				array(
					'ajaxurl'      => admin_url( 'admin-ajax.php' ),
					'exportNonce'  => wp_create_nonce( 'arwrev_export_nonce' ),
					'importNonce'  => wp_create_nonce( 'arwrev_import_nonce' ),
					'i18n'         => array(
						'backupExportedSuccess' => __( 'Backup exported successfully!', 'arewa-recently-viewed-content' ),
						'exportFailed'          => __( 'Export failed: ', 'arewa-recently-viewed-content' ),
						'exportFailedRetry'     => __( 'Export failed. Please try again.', 'arewa-recently-viewed-content' ),
						'pleaseSelectFile'      => __( 'Please select a backup file to import.', 'arewa-recently-viewed-content' ),
						'importSuccess'         => __( 'Data imported successfully! The page will reload to reflect changes.', 'arewa-recently-viewed-content' ),
						'importFailed'          => __( 'Import failed: ', 'arewa-recently-viewed-content' ),
						'importFailedFormat'    => __( 'Import failed. Please check the file format and try again.', 'arewa-recently-viewed-content' ),
					),
				)
			);
		}
	}
}

// Run options migration from arwrev_* to arwrev_* once.
add_action( 'plugins_loaded', 'arwrev_migrate_options_from_arwrev', 0 );

/**
 * Initialize plugin
 */
function arwrev_init() {
	return ARWREV_Plugin::get_instance();
}

// Start the plugin.
add_action( 'plugins_loaded', 'arwrev_init' );

// Check for plugin updates and run migration if needed.
add_action( 'plugins_loaded', 'arwrev_check_version_and_migrate' );

/**
 * Check plugin version and run migration if needed.
 */
function arwrev_check_version_and_migrate() {
	$installed_version = get_option( 'arwrev_version', '0.0.0' );

	if ( version_compare( $installed_version, ARWREV_VERSION, '<' ) ) {
		// Plugin was updated, run migration.
		arwrev_migrate_user_data();

		// Update version.
		update_option( 'arwrev_version', ARWREV_VERSION );

		// Log update.
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when explicitly enabled
			error_log( "ARWREV: Plugin updated from {$installed_version} to " . ARWREV_VERSION );
		}
	}
}

/**
 * Activation hook.
 *
 * Sets default plugin options on activation.
 * Migrates from Arewa Watch History if upgrading.
 * Restores preserved data if plugin was previously uninstalled.
 */
function arwrev_activate() {
	// Check if WordPress functions are available.
	if ( ! function_exists( 'get_option' ) || ! function_exists( 'add_option' ) ) {
		return;
	}

	// Run options migration first so we read/write arwrev_*.
	arwrev_migrate_options_from_arwrev();

	// Store activation timestamp for debugging.
	update_option( 'arwrev_last_activation', time() );

	// Check if data was preserved from previous installation.
	$preserved_data = get_option( 'arwrev_data_preserved', false );
	if ( $preserved_data && is_array( $preserved_data ) && isset( $preserved_data['preserved'] ) ) {
		// Data was preserved, plugin is being reactivated.
		arwrev_restore_preserved_data();

		// Update the preserved data record.
		update_option(
			'arwrev_data_preserved',
			array(
				'restored_at'      => time(),
				'previous_version' => isset( $preserved_data['version'] ) ? $preserved_data['version'] : '1.0.0',
				'current_version'  => ARWREV_VERSION,
				'restored'         => true,
			)
		);

		// Add admin notice about successful restoration.
		add_option( 'arwrev_show_restoration_notice', true );
	}

	// Always run migration on activation to ensure data integrity.
	arwrev_migrate_user_data();

	// Migrate from old Arewa Watch History plugin if it exists.
	$old_settings = get_option( 'arewa_wh_settings', false );
	if ( false !== $old_settings && is_array( $old_settings ) ) {
		update_option( 'arwrev_settings', $old_settings );
	}

	// Set default options if not exists.
	$default_options = array(
		'enabled'                   => '1',
		'post_types'                => array( 'post', 'page' ),
		'max_items'                 => 100,
		'max_guest_items'           => 50,
		'auto_cleanup_enabled'      => '0',
		'columns_desktop'           => 6,
		'columns_tablet'            => 4,
		'columns_mobile'            => 2,
		'gap_desktop'               => 11,
		'gap_tablet'                => 9,
		'gap_mobile'                => 8,
		'image_width'               => 155,
		'image_height'              => 250,
		'items_per_page'            => 12,
		'post_type_labels'          => array(),
		'title_max_chars'           => 0,
		'login_url'                 => '',
		'login_button_text'         => 'Login My Account', // No translation during activation.
		'require_login_for_guests'  => '0', // Allow guests to view history by default.
		// Layout system options.
		'layout_type'               => 'video',
		'display_mode'              => 'grid',
		// Article layout options.
		'article_excerpt_length'    => 150,
		// Shop layout options.
		'shop_show_price'           => '1',
		'shop_show_add_to_cart'     => '1',
	);

	if ( false === get_option( 'arwrev_settings' ) ) {
		add_option( 'arwrev_settings', $default_options );
	}

	// Set default data preservation setting (preserve by default).
	if ( false === get_option( 'arwrev_preserve_data_on_uninstall' ) ) {
		add_option( 'arwrev_preserve_data_on_uninstall', true );
	}

	// Store plugin version.
	update_option( 'arwrev_version', ARWREV_VERSION );
}
register_activation_hook( __FILE__, 'arwrev_activate' );

/**
 * Restore preserved data after reactivation.
 */
function arwrev_restore_preserved_data() {
	// Check if settings exist and are valid.
	$settings = get_option( 'arwrev_settings', false );
	if ( false === $settings || ! is_array( $settings ) ) {
		// Settings were somehow lost, this shouldn't happen in preserve mode.
		// But we'll handle it gracefully.
		return false;
	}

	// Verify user history data integrity.
	$users         = get_users(
		array(
			'fields' => 'ID',
			'number' => 10,
		)
	); // Check first 10 users.
	$history_found = false;

	foreach ( $users as $user_id ) {
		$history = get_user_meta( $user_id, 'arewa_recently_viewed', true );
		if ( ! empty( $history ) && is_array( $history ) ) {
			$history_found = true;
			break;
		}
	}

	// Log restoration for debugging (in debug mode).
	if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when explicitly enabled
		error_log( 'ARWREV: Data restoration completed. Settings preserved: ' . ( $settings ? 'Yes' : 'No' ) . ', User history found: ' . ( $history_found ? 'Yes' : 'No' ) );
	}

	return true;
}

/**
 * Show admin notice after data restoration.
 */
function arwrev_show_restoration_notice() {
	if ( get_option( 'arwrev_show_restoration_notice' ) ) {
		add_action(
			'admin_notices',
			function() {
				$preserved_data   = get_option( 'arwrev_data_preserved', array() );
				$previous_version = isset( $preserved_data['previous_version'] ) ? $preserved_data['previous_version'] : 'unknown';

				echo '<div class="notice notice-success is-dismissible">';
				echo '<p><strong>' . esc_html__( 'AR Viewed Content Restored!', 'arewa-recently-viewed-content' ) . '</strong></p>';
				echo '<p>' . sprintf(
				/* translators: %s: previous plugin version number */
					esc_html__( 'Your plugin settings and watch history data have been successfully restored from version %s. All user data is intact and the plugin is ready to use.', 'arewa-recently-viewed-content' ),
					esc_html( $previous_version )
				) . '</p>';
				echo '</div>';
			}
		);

		// Remove the notice flag after showing it once.
		delete_option( 'arwrev_show_restoration_notice' );
	}

	// Show migration notice if migration just happened.
	$migration_status = get_option( 'arwrev_migration_status', array() );
	if ( isset( $migration_status['show_notice'] ) && $migration_status['show_notice'] ) {
		add_action(
			'admin_notices',
			function() use ( $migration_status ) {
				$migrated_users = isset( $migration_status['migrated_users'] ) ? $migration_status['migrated_users'] : 0;

				if ( $migrated_users > 0 ) {
					echo '<div class="notice notice-info is-dismissible">';
					echo '<p><strong>' . esc_html__( 'AR Viewed Content Migration Complete!', 'arewa-recently-viewed-content' ) . '</strong></p>';
					echo '<p>' . esc_html(
						sprintf(
							/* translators: %d: number of users migrated */
							esc_html__( 'Successfully migrated watch history data for %d users to the new storage system.', 'arewa-recently-viewed-content' ),
							(int) $migrated_users
						)
					) . '</p>';
					echo '</div>';
				}
			}
		);

		// Remove the notice flag.
		$migration_status['show_notice'] = false;
		update_option( 'arwrev_migration_status', $migration_status );
	}
}

// Hook the restoration notice.
add_action( 'admin_init', 'arwrev_show_restoration_notice' );

/**
 * Deactivation hook.
 *
 * Cleans up scheduled events on deactivation.
 */
function arwrev_deactivate() {
	// Clear scheduled cleanup event.
	if ( function_exists( 'wp_clear_scheduled_hook' ) ) {
		wp_clear_scheduled_hook( 'arwrev_cleanup' );
	}
}
register_deactivation_hook( __FILE__, 'arwrev_deactivate' );

/**
 * Merge two user history arrays, removing duplicates and sorting by timestamp.
 *
 * @param array $history1 First history array.
 * @param array $history2 Second history array.
 * @return array Merged and sorted history.
 */
function arwrev_merge_user_histories( $history1, $history2 ) {
	$merged     = array();
	$seen_posts = array();

	// Combine both histories.
	$all_items = array_merge( $history1, $history2 );

	// Remove duplicates, keeping the most recent timestamp.
	foreach ( $all_items as $item ) {
		if ( ! isset( $item['post_id'] ) ) {
			continue;
		}

		$post_id = (int) $item['post_id'];

		if ( ! isset( $seen_posts[ $post_id ] ) ||
			( isset( $item['timestamp'] ) && $item['timestamp'] > $seen_posts[ $post_id ]['timestamp'] ) ) {
			$seen_posts[ $post_id ] = $item;
		}
	}

	// Convert back to indexed array.
	$merged = array_values( $seen_posts );

	// Sort by timestamp (newest first).
	usort(
		$merged,
		function( $a, $b ) {
			$timestamp_a = isset( $a['timestamp'] ) ? $a['timestamp'] : 0;
			$timestamp_b = isset( $b['timestamp'] ) ? $b['timestamp'] : 0;
			return $timestamp_b - $timestamp_a;
		}
	);

	// Apply max items limit.
	$settings  = get_option( 'arwrev_settings', array() );
	$max_items = isset( $settings['max_items'] ) ? (int) $settings['max_items'] : 100;
	if ( $max_items > 0 && count( $merged ) > $max_items ) {
		$merged = array_slice( $merged, 0, $max_items );
	}

	return $merged;
}

/**
 * Migrate user data from old meta keys to new meta key.
 * This function is called on every activation to ensure data integrity.
 */
function arwrev_migrate_user_data() {
	if ( ! function_exists( 'get_users' ) ) {
		return;
	}

	// Get migration status.
	$migration_status = get_option( 'arwrev_migration_status', array() );
	$current_version  = ARWREV_VERSION;

	// Check if we've already migrated for this version.
	if ( isset( $migration_status['last_migrated_version'] ) &&
		$migration_status['last_migrated_version'] === $current_version ) {
		return; // Already migrated for this version.
	}

	$migrated_users = 0;
	$users          = get_users( array( 'fields' => 'ID' ) );

	foreach ( $users as $user_id ) {
		$migration_needed = false;

		// Get all possible history sources.
		$old_arewa_history = get_user_meta( $user_id, 'arewa_wh_history', true );
		$arwrev_history      = get_user_meta( $user_id, 'arwrev_history', true );
		$current_history   = get_user_meta( $user_id, 'arewa_recently_viewed', true );

		// Initialize final history.
		$final_history = array();

		// Start with current history if it exists and is valid.
		if ( ! empty( $current_history ) && is_array( $current_history ) ) {
			$final_history = $current_history;
		}

		// Merge old Arewa Watch History plugin data.
		if ( ! empty( $old_arewa_history ) && is_array( $old_arewa_history ) ) {
			if ( empty( $final_history ) ) {
				$final_history = $old_arewa_history;
			} else {
				$final_history = arwrev_merge_user_histories( $final_history, $old_arewa_history );
			}
			$migration_needed = true;
		}

		// Merge old arwrev_history data.
		if ( ! empty( $arwrev_history ) && is_array( $arwrev_history ) ) {
			if ( empty( $final_history ) ) {
				$final_history = $arwrev_history;
			} else {
				$final_history = arwrev_merge_user_histories( $final_history, $arwrev_history );
			}
			$migration_needed = true;
		}

		// Update the user meta if migration was needed.
		if ( $migration_needed && ! empty( $final_history ) ) {
			update_user_meta( $user_id, 'arewa_recently_viewed', $final_history );
			$migrated_users++;
		}
	}

	// Update migration status.
	update_option(
		'arwrev_migration_status',
		array(
			'last_migrated_version' => $current_version,
			'last_migration_time'   => time(),
			'migrated_users'        => $migrated_users,
			'total_users_checked'   => count( $users ),
			'show_notice'           => $migrated_users > 0, // Only show notice if users were actually migrated.
		)
	);

	// Log migration for debugging.
	if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when explicitly enabled.
		error_log( 'ARWREV Migration: Migrated ' . $migrated_users . ' users out of ' . count( $users ) . ' total users.' );
	}
}
