<?php
/**
 * Admin Settings with Tabbed Interface
 *
 * Handles admin settings page with tabs for better UX
 *
 * @package Arewa_Recently_Viewed_Content
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin Settings Class with Tabbed Interface.
 *
 * Creates a clean admin interface with separate tabs for settings and documentation.
 *
 * @package Arewa_Recently_Viewed_Content
 */
class ARWREV_Admin_Settings_Tabbed {

	/**
	 * Instance of this class
	 *
	 * @var object
	 */
	private static $instance = null;

	/**
	 * Get instance of this class
	 *
	 * @return ARWREV_Admin_Settings_Tabbed
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		add_action( 'init', array( $this, 'init_admin_hooks' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
	}

	/**
	 * Initialize admin hooks after textdomain is loaded
	 */
	public function init_admin_hooks() {
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
	}

	/**
	 * Add main admin menu
	 */
	public function add_admin_menu() {
		add_menu_page(
			__( 'AR Viewed Content', 'arewa-recently-viewed-content' ),
			__( 'AR Viewed Content', 'arewa-recently-viewed-content' ),
			'manage_options',
			'ar-viewed-content',
			array( $this, 'render_admin_page' ),
			'dashicons-visibility',
			30
		);
	}

	/**
	 * Register settings
	 */
	public function register_settings() {
		register_setting(
			'arwrev_settings_group',
			'arwrev_settings',
			array( $this, 'sanitize_settings' )
		);
	}

	/**
	 * Sanitize settings.
	 *
	 * @param array $input Raw input.
	 * @return array Sanitized input.
	 */
	public function sanitize_settings( $input ) {
		$sanitized = array();

		// Enable/Disable.
		$sanitized['enabled'] = isset( $input['enabled'] ) ? '1' : '0';

		// Post types.
		$sanitized['post_types'] = array();
		if ( isset( $input['post_types'] ) && is_array( $input['post_types'] ) ) {
			$all_post_types = get_post_types( array( 'public' => true ), 'names' );
			foreach ( $input['post_types'] as $post_type ) {
				if ( in_array( $post_type, $all_post_types, true ) ) {
					$sanitized['post_types'][] = sanitize_text_field( $post_type );
				}
			}
		}

		// Max items.
		$sanitized['max_items'] = isset( $input['max_items'] ) ? absint( $input['max_items'] ) : 100;

		// Max guest items.
		$sanitized['max_guest_items'] = isset( $input['max_guest_items'] ) ? absint( $input['max_guest_items'] ) : 50;

		// Auto cleanup.
		$sanitized['auto_cleanup_enabled'] = isset( $input['auto_cleanup_enabled'] ) ? '1' : '0';

		// Layout settings.
		if ( class_exists( 'ARWREV_Layout_Manager' ) ) {
			$layout_manager = ARWREV_Layout_Manager::get_instance();

			// Layout type.
			$layout_types = $layout_manager->get_layout_types();
			$layout_type  = isset( $input['layout_type'] ) ? sanitize_text_field( $input['layout_type'] ) : 'video';
			if ( ! isset( $layout_types[ $layout_type ] ) || ! $layout_manager->is_layout_available( $layout_type ) ) {
				$layout_type = 'video'; // Fallback.
			}
			$sanitized['layout_type'] = $layout_type;

			// Display mode is now determined by layout type, not user setting.
			$supported_modes           = $layout_manager->get_supported_display_modes( $layout_type );
			$sanitized['display_mode'] = ! empty( $supported_modes ) ? $supported_modes[0] : 'grid';
		} else {
			// Fallback if Layout Manager is not available.
			$sanitized['layout_type']  = isset( $input['layout_type'] ) ? sanitize_text_field( $input['layout_type'] ) : 'video';
			$sanitized['display_mode'] = 'grid'; // Default fallback.
		}

		// Columns settings.
		$sanitized['columns_desktop'] = isset( $input['columns_desktop'] ) ? absint( $input['columns_desktop'] ) : 6;
		$sanitized['columns_tablet']  = isset( $input['columns_tablet'] ) ? absint( $input['columns_tablet'] ) : 4;
		$sanitized['columns_mobile']  = isset( $input['columns_mobile'] ) ? absint( $input['columns_mobile'] ) : 2;

		// Validate column values (1-6 for desktop, 1-4 for tablet/mobile).
		if ( $sanitized['columns_desktop'] < 1 || $sanitized['columns_desktop'] > 6 ) {
			$sanitized['columns_desktop'] = 6;
		}
		if ( $sanitized['columns_tablet'] < 1 || $sanitized['columns_tablet'] > 4 ) {
			$sanitized['columns_tablet'] = 4;
		}
		if ( $sanitized['columns_mobile'] < 1 || $sanitized['columns_mobile'] > 2 ) {
			$sanitized['columns_mobile'] = 2;
		}

		// Grid gap settings (px).
		$sanitized['gap_desktop'] = isset( $input['gap_desktop'] ) ? absint( $input['gap_desktop'] ) : 11;
		$sanitized['gap_tablet']  = isset( $input['gap_tablet'] ) ? absint( $input['gap_tablet'] ) : 9;
		$sanitized['gap_mobile']  = isset( $input['gap_mobile'] ) ? absint( $input['gap_mobile'] ) : 8;
		$sanitized['gap_desktop'] = max( 0, min( 100, $sanitized['gap_desktop'] ) );
		$sanitized['gap_tablet']  = max( 0, min( 100, $sanitized['gap_tablet'] ) );
		$sanitized['gap_mobile']  = max( 0, min( 100, $sanitized['gap_mobile'] ) );

		// Image size settings.
		$sanitized['image_width']  = isset( $input['image_width'] ) ? absint( $input['image_width'] ) : 155;
		$sanitized['image_height'] = isset( $input['image_height'] ) ? absint( $input['image_height'] ) : 250;

		// Validate image size (min 50px, max 500px).
		if ( $sanitized['image_width'] < 50 || $sanitized['image_width'] > 500 ) {
			$sanitized['image_width'] = 155;
		}
		if ( $sanitized['image_height'] < 50 || $sanitized['image_height'] > 500 ) {
			$sanitized['image_height'] = 250;
		}

		// Pagination settings.
		$sanitized['items_per_page'] = isset( $input['items_per_page'] ) ? absint( $input['items_per_page'] ) : 12;
		if ( $sanitized['items_per_page'] < 1 || $sanitized['items_per_page'] > 100 ) {
			$sanitized['items_per_page'] = 12;
		}

		// Article layout settings.
		$sanitized['article_excerpt_length'] = isset( $input['article_excerpt_length'] ) ? absint( $input['article_excerpt_length'] ) : 150;
		if ( $sanitized['article_excerpt_length'] < 50 || $sanitized['article_excerpt_length'] > 500 ) {
			$sanitized['article_excerpt_length'] = 150;
		}

		// Shop layout settings.
		$sanitized['shop_show_price']       = isset( $input['shop_show_price'] ) ? '1' : '0';
		$sanitized['shop_show_add_to_cart'] = isset( $input['shop_show_add_to_cart'] ) ? '1' : '0';

		// Post type labels customization.
		$sanitized['post_type_labels'] = array();
		if ( isset( $input['post_type_labels'] ) && is_array( $input['post_type_labels'] ) ) {
			foreach ( $input['post_type_labels'] as $post_type => $label ) {
				$sanitized_post_type                                   = sanitize_text_field( $post_type );
				$sanitized['post_type_labels'][ $sanitized_post_type ] = sanitize_text_field( $label );
			}
		}

		// Title character limit.
		$sanitized['title_max_chars'] = isset( $input['title_max_chars'] ) ? absint( $input['title_max_chars'] ) : 0;
		if ( $sanitized['title_max_chars'] < 0 || $sanitized['title_max_chars'] > 200 ) {
			$sanitized['title_max_chars'] = 0; // 0 means no limit.
		}

		// Login URL.
		if ( isset( $input['login_url'] ) && is_string( $input['login_url'] ) ) {
			$sanitized['login_url'] = esc_url_raw( trim( $input['login_url'] ) );
		} else {
			$sanitized['login_url'] = '';
		}

		// Login button text.
		if ( isset( $input['login_button_text'] ) && is_string( $input['login_button_text'] ) ) {
			$sanitized['login_button_text'] = sanitize_text_field( trim( $input['login_button_text'] ) );
		} else {
			$sanitized['login_button_text'] = 'Login My Account';
		}

		// Require login for guests.
		$sanitized['require_login_for_guests'] = isset( $input['require_login_for_guests'] ) ? '1' : '0';

		// Data preservation setting.
		$preserve_data = isset( $input['preserve_data_on_uninstall'] ) ? '1' : '0';
		update_option( 'arwrev_preserve_data_on_uninstall', '1' === $preserve_data );

		return $sanitized;
	}

	/**
	 * Render admin page with tabs
	 */
	public function render_admin_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Get current tab.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Tab switching doesn't require nonce verification.
		$current_tab = isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : 'general';

		?>
		<?php // phpcs:disable WordPress.WhiteSpace.PrecisionAlignment.Found -- HTML output uses alignment for readability. ?>
		<div class="wrap arwrev-admin-page">
			<h1>
				<?php esc_html_e( 'AR Viewed Content', 'arewa-recently-viewed-content' ); ?>
			</h1>

			<!-- Tab Navigation -->
			<nav class="nav-tab-wrapper">
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=ar-viewed-content&tab=general' ) ); ?>"
					class="nav-tab <?php echo esc_attr( 'general' === $current_tab ? 'nav-tab-active' : '' ); ?>">
					<?php esc_html_e( 'General Settings', 'arewa-recently-viewed-content' ); ?>
				</a>
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=ar-viewed-content&tab=data-management' ) ); ?>"
				   class="nav-tab <?php echo esc_attr( 'data-management' === $current_tab ? 'nav-tab-active' : '' ); ?>">
					<?php esc_html_e( 'Data Management', 'arewa-recently-viewed-content' ); ?>
				</a>
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=ar-viewed-content&tab=shortcodes' ) ); ?>"
					class="nav-tab <?php echo esc_attr( 'shortcodes' === $current_tab ? 'nav-tab-active' : '' ); ?>">
					<?php esc_html_e( 'Shortcodes', 'arewa-recently-viewed-content' ); ?>
				</a>
				<a href="https://arewa.dev/arewa-recently-viewed-content/" class="nav-tab" target="_blank" rel="noopener noreferrer">
					<?php esc_html_e( 'Documentation', 'arewa-recently-viewed-content' ); ?>
				</a>
			</nav>

			<!-- Tab Content -->
			<div class="tab-content">
				<?php
				switch ( $current_tab ) {
					case 'data-management':
						$this->render_data_management_tab();
						break;
					case 'shortcodes':
						$this->render_documentation_tab();
						break;
					case 'general':
					default:
						$this->render_general_settings_tab();
						break;
				}
				?>
			</div>
		</div>
		<?php
	}

	/**
	 * Render General Settings tab
	 */
	private function render_general_settings_tab() {
		$settings = get_option( 'arwrev_settings', array() );

		// Get layout manager safely.
		$layout_manager = null;
		if ( class_exists( 'ARWREV_Layout_Manager' ) ) {
			$layout_manager = ARWREV_Layout_Manager::get_instance();
		}

		// Get current values.
		$enabled             = isset( $settings['enabled'] ) ? $settings['enabled'] : '1';
		$post_types_selected = isset( $settings['post_types'] ) ? $settings['post_types'] : array( 'post', 'page' );
		$max_items           = isset( $settings['max_items'] ) ? (int) $settings['max_items'] : 100;
		$max_guest_items     = isset( $settings['max_guest_items'] ) ? (int) $settings['max_guest_items'] : 50;
		$auto_cleanup        = isset( $settings['auto_cleanup_enabled'] ) ? $settings['auto_cleanup_enabled'] : '0';

		// Layout settings.
		$layout_type  = isset( $settings['layout_type'] ) ? $settings['layout_type'] : 'video';
		$display_mode = isset( $settings['display_mode'] ) ? $settings['display_mode'] : 'grid';

		$columns_desktop = isset( $settings['columns_desktop'] ) ? (int) $settings['columns_desktop'] : 6;
		$columns_tablet  = isset( $settings['columns_tablet'] ) ? (int) $settings['columns_tablet'] : 4;
		$columns_mobile  = isset( $settings['columns_mobile'] ) ? (int) $settings['columns_mobile'] : 2;
		$gap_desktop     = isset( $settings['gap_desktop'] ) ? (int) $settings['gap_desktop'] : 11;
		$gap_tablet      = isset( $settings['gap_tablet'] ) ? (int) $settings['gap_tablet'] : 9;
		$gap_mobile      = isset( $settings['gap_mobile'] ) ? (int) $settings['gap_mobile'] : 8;
		$image_width     = isset( $settings['image_width'] ) ? (int) $settings['image_width'] : 155;
		$image_height    = isset( $settings['image_height'] ) ? (int) $settings['image_height'] : 250;
		$items_per_page  = isset( $settings['items_per_page'] ) ? (int) $settings['items_per_page'] : 12;

		// Article settings.
		$article_excerpt_length = isset( $settings['article_excerpt_length'] ) ? (int) $settings['article_excerpt_length'] : 150;

		// Shop settings.
		$shop_show_price       = isset( $settings['shop_show_price'] ) ? $settings['shop_show_price'] : '1';
		$shop_show_add_to_cart = isset( $settings['shop_show_add_to_cart'] ) ? $settings['shop_show_add_to_cart'] : '1';

		$post_type_labels  = isset( $settings['post_type_labels'] ) ? $settings['post_type_labels'] : array();
		$title_max_chars   = isset( $settings['title_max_chars'] ) ? (int) $settings['title_max_chars'] : 0;
		$login_url         = isset( $settings['login_url'] ) ? esc_url( $settings['login_url'] ) : '';
		$login_button_text = isset( $settings['login_button_text'] ) ? esc_html( $settings['login_button_text'] ) : __( 'Login My Account', 'arewa-recently-viewed-content' );
		$preserve_data     = get_option( 'arwrev_preserve_data_on_uninstall', true );

		// Get all registered post types.
		$all_post_types = get_post_types( array( 'public' => true ), 'objects' );
		?>
		
		<form action="options.php" method="post">
			<?php settings_fields( 'arwrev_settings_group' ); ?>

			<table class="form-table" role="presentation">
				<tbody>
					<tr>
						<th scope="row">
							<label for="arwrev_enabled"><?php esc_html_e( 'Enable Watch History', 'arewa-recently-viewed-content' ); ?></label>
						</th>
						<td>
							<input type="checkbox" id="arwrev_enabled" name="arwrev_settings[enabled]" value="1" <?php checked( $enabled, '1' ); ?> />
							<p class="description"><?php esc_html_e( 'Enable or disable watch history tracking for logged-in users.', 'arewa-recently-viewed-content' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label><?php esc_html_e( 'Post Types to Track', 'arewa-recently-viewed-content' ); ?></label>
						</th>
						<td>
							<fieldset>
								<?php foreach ( $all_post_types as $post_type ) : ?>
									<label>
										<input
											type="checkbox"
											name="arwrev_settings[post_types][]"
											value="<?php echo esc_attr( $post_type->name ); ?>"
											<?php checked( in_array( $post_type->name, $post_types_selected, true ) ); ?>
										/>
										<?php echo esc_html( $post_type->label ); ?>
										<code><?php echo esc_html( $post_type->name ); ?></code>
									</label><br />
								<?php endforeach; ?>
							</fieldset>
							<p class="description"><?php esc_html_e( 'Select which post types should be tracked in watch history.', 'arewa-recently-viewed-content' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label><?php esc_html_e( 'Layout Settings', 'arewa-recently-viewed-content' ); ?></label>
						</th>
						<td>
							<fieldset>
								<legend class="screen-reader-text"><?php esc_html_e( 'Layout Type', 'arewa-recently-viewed-content' ); ?></legend>
								<p><strong><?php esc_html_e( 'Layout Type:', 'arewa-recently-viewed-content' ); ?></strong></p>
								<?php
								if ( $layout_manager ) {
									$layout_types = $layout_manager->get_layout_types();
									foreach ( $layout_types as $type => $config ) :
										$available = $layout_manager->is_layout_available( $type );
										?>
										<label style="display: block; margin-bottom: 10px;">
											<input
												type="radio"
												name="arwrev_settings[layout_type]"
												value="<?php echo esc_attr( $type ); ?>"
												<?php checked( $layout_type, $type ); ?>
												<?php disabled( ! $available ); ?>
											/>
											<strong><?php echo esc_html( $config['label'] ); ?></strong>
											<?php if ( ! $available && isset( $config['requires'] ) ) : ?>
												<?php /* translators: %s: name of the required plugin or feature */ ?>
												<em>(<?php printf( esc_html__( 'Requires %s', 'arewa-recently-viewed-content' ), esc_html( $config['requires'] ) ); ?>)</em>
											<?php endif; ?>
											<br />
											<span class="description" style="margin-left: 25px;"><?php echo esc_html( $config['description'] ); ?></span>
										</label>
										<?php
									endforeach;
								} else {
									// Fallback options when Layout Manager is not available.
									?>
									<label style="display: block; margin-bottom: 10px;">
										<input type="radio" name="arwrev_settings[layout_type]" value="video" <?php checked( $layout_type, 'video' ); ?> />
										<strong>Video Layout</strong><br />
										<span class="description" style="margin-left: 25px;">Card-based layout with overlay text, perfect for video content</span>
									</label>
									<label style="display: block; margin-bottom: 10px;">
										<input type="radio" name="arwrev_settings[layout_type]" value="article" <?php checked( $layout_type, 'article' ); ?> />
										<strong>Article Layout</strong><br />
										<span class="description" style="margin-left: 25px;">Clean blog-style layout with excerpts, ideal for articles and posts</span>
									</label>
									<label style="display: block; margin-bottom: 10px;">
										<input type="radio" name="arwrev_settings[layout_type]" value="shop" <?php checked( $layout_type, 'shop' ); ?> />
										<strong>Shop Layout</strong><br />
										<span class="description" style="margin-left: 25px;">Product-focused layout with pricing and cart buttons</span>
									</label>
									<label style="display: block; margin-bottom: 10px;">
										<input type="radio" name="arwrev_settings[layout_type]" value="carousel" <?php checked( $layout_type, 'carousel' ); ?> />
										<strong>Carousel Layout</strong><br />
										<span class="description" style="margin-left: 25px;">Horizontal scrolling carousel layout, perfect for showcasing content</span>
									</label>
								<?php } ?>
							</fieldset>
							<p class="description"><?php esc_html_e( 'Choose how the watch history should be displayed. Each layout type has its own optimized display mode.', 'arewa-recently-viewed-content' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="arwrev_max_items"><?php esc_html_e( 'Max History Items (Logged-in Users)', 'arewa-recently-viewed-content' ); ?></label>
						</th>
						<td>
							<input
								type="number"
								id="arwrev_max_items"
								name="arwrev_settings[max_items]"
								value="<?php echo esc_attr( $max_items ); ?>"
								min="0"
								step="1"
								class="small-text"
							/>
							<p class="description"><?php esc_html_e( 'Maximum number of items to store per logged-in user. Set to 0 for unlimited.', 'arewa-recently-viewed-content' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="arwrev_max_guest_items"><?php esc_html_e( 'Max History Items (Guest Users)', 'arewa-recently-viewed-content' ); ?></label>
						</th>
						<td>
							<input
								type="number"
								id="arwrev_max_guest_items"
								name="arwrev_settings[max_guest_items]"
								value="<?php echo esc_attr( $max_guest_items ); ?>"
								min="0"
								step="1"
								class="small-text"
							/>
							<p class="description"><?php esc_html_e( 'Maximum number of items to store for guest users in cookies/localStorage. Recommended: 50 or less for performance.', 'arewa-recently-viewed-content' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="arwrev_auto_cleanup"><?php esc_html_e( 'Auto Cleanup', 'arewa-recently-viewed-content' ); ?></label>
						</th>
						<td>
							<input
								type="checkbox"
								id="arwrev_auto_cleanup"
								name="arwrev_settings[auto_cleanup_enabled]"
								value="1"
								<?php checked( $auto_cleanup, '1' ); ?>
							/>
							<p class="description"><?php esc_html_e( 'Automatically remove old entries beyond the max history limit (runs daily).', 'arewa-recently-viewed-content' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label><?php esc_html_e( 'Grid Columns', 'arewa-recently-viewed-content' ); ?></label>
						</th>
						<td>
							<fieldset>
								<label for="arwrev_columns_desktop">
									<?php esc_html_e( 'Desktop (≥1024px):', 'arewa-recently-viewed-content' ); ?>
									<input
										type="number"
										id="arwrev_columns_desktop"
										name="arwrev_settings[columns_desktop]"
										value="<?php echo esc_attr( $columns_desktop ); ?>"
										min="1"
										max="6"
										step="1"
										class="small-text"
										style="margin-left: 10px;"
									/>
									<span class="description" style="margin-left: 5px;"><?php esc_html_e( '(1-6 columns)', 'arewa-recently-viewed-content' ); ?></span>
								</label><br /><br />

								<label for="arwrev_columns_tablet">
									<?php esc_html_e( 'Tablet (768px - 1023px):', 'arewa-recently-viewed-content' ); ?>
									<input
										type="number"
										id="arwrev_columns_tablet"
										name="arwrev_settings[columns_tablet]"
										value="<?php echo esc_attr( $columns_tablet ); ?>"
										min="1"
										max="4"
										step="1"
										class="small-text"
										style="margin-left: 10px;"
									/>
									<span class="description" style="margin-left: 5px;"><?php esc_html_e( '(1-4 columns)', 'arewa-recently-viewed-content' ); ?></span>
								</label><br /><br />

								<label for="arwrev_columns_mobile">
									<?php esc_html_e( 'Mobile (<768px):', 'arewa-recently-viewed-content' ); ?>
									<input
										type="number"
										id="arwrev_columns_mobile"
										name="arwrev_settings[columns_mobile]"
										value="<?php echo esc_attr( $columns_mobile ); ?>"
										min="1"
										max="2"
										step="1"
										class="small-text"
										style="margin-left: 10px;"
									/>
									<span class="description" style="margin-left: 5px;"><?php esc_html_e( '(1-2 columns)', 'arewa-recently-viewed-content' ); ?></span>
								</label>
							</fieldset>
							<p class="description"><?php esc_html_e( 'Configure the number of columns displayed in the watch history grid for different screen sizes.', 'arewa-recently-viewed-content' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label><?php esc_html_e( 'Grid Gap (px)', 'arewa-recently-viewed-content' ); ?></label>
						</th>
						<td>
							<fieldset>
								<label for="arwrev_gap_desktop">
									<?php esc_html_e( 'Desktop (≥1024px):', 'arewa-recently-viewed-content' ); ?>
									<input
										type="number"
										id="arwrev_gap_desktop"
										name="arwrev_settings[gap_desktop]"
										value="<?php echo esc_attr( $gap_desktop ); ?>"
										min="0"
										max="100"
										step="1"
										class="small-text"
										style="margin-left: 10px;"
									/>
									<span class="description" style="margin-left: 5px;"><?php esc_html_e( 'px', 'arewa-recently-viewed-content' ); ?></span>
								</label><br /><br />
								<label for="arwrev_gap_tablet">
									<?php esc_html_e( 'Tablet (768px - 1023px):', 'arewa-recently-viewed-content' ); ?>
									<input
										type="number"
										id="arwrev_gap_tablet"
										name="arwrev_settings[gap_tablet]"
										value="<?php echo esc_attr( $gap_tablet ); ?>"
										min="0"
										max="100"
										step="1"
										class="small-text"
										style="margin-left: 10px;"
									/>
									<span class="description" style="margin-left: 5px;"><?php esc_html_e( 'px', 'arewa-recently-viewed-content' ); ?></span>
								</label><br /><br />
								<label for="arwrev_gap_mobile">
									<?php esc_html_e( 'Mobile (<768px):', 'arewa-recently-viewed-content' ); ?>
									<input
										type="number"
										id="arwrev_gap_mobile"
										name="arwrev_settings[gap_mobile]"
										value="<?php echo esc_attr( $gap_mobile ); ?>"
										min="0"
										max="100"
										step="1"
										class="small-text"
										style="margin-left: 10px;"
									/>
									<span class="description" style="margin-left: 5px;"><?php esc_html_e( 'px', 'arewa-recently-viewed-content' ); ?></span>
								</label>
							</fieldset>
							<p class="description"><?php esc_html_e( 'Spacing between grid items for Video, Article, Shop and Carousel layouts. Defaults: 11 (desktop), 9 (tablet), 8 (mobile).', 'arewa-recently-viewed-content' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label><?php esc_html_e( 'Image Size', 'arewa-recently-viewed-content' ); ?></label>
						</th>
						<td>
							<fieldset>
								<label for="arwrev_image_width">
									<?php esc_html_e( 'Width (px):', 'arewa-recently-viewed-content' ); ?>
									<input
										type="number"
										id="arwrev_image_width"
										name="arwrev_settings[image_width]"
										value="<?php echo esc_attr( $image_width ); ?>"
										min="50"
										max="500"
										step="1"
										class="small-text"
										style="margin-left: 10px;"
									/>
									<span class="description" style="margin-left: 5px;"><?php esc_html_e( '(50-500px)', 'arewa-recently-viewed-content' ); ?></span>
								</label><br /><br />

								<label for="arwrev_image_height">
									<?php esc_html_e( 'Height (px):', 'arewa-recently-viewed-content' ); ?>
									<input
										type="number"
										id="arwrev_image_height"
										name="arwrev_settings[image_height]"
										value="<?php echo esc_attr( $image_height ); ?>"
										min="50"
										max="500"
										step="1"
										class="small-text"
										style="margin-left: 10px;"
									/>
									<span class="description" style="margin-left: 5px;"><?php esc_html_e( '(50-500px)', 'arewa-recently-viewed-content' ); ?></span>
								</label>
							</fieldset>
							<p class="description"><?php esc_html_e( 'Configure the default image size for watch history cards.', 'arewa-recently-viewed-content' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="arwrev_items_per_page"><?php esc_html_e( 'Items Per Page', 'arewa-recently-viewed-content' ); ?></label>
						</th>
						<td>
							<input
								type="number"
								id="arwrev_items_per_page"
								name="arwrev_settings[items_per_page]"
								value="<?php echo esc_attr( $items_per_page ); ?>"
								min="1"
								max="100"
								step="1"
								class="small-text"
							/>
							<p class="description"><?php esc_html_e( 'Number of items to display per page in the watch history grid. (1-100)', 'arewa-recently-viewed-content' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="arwrev_article_excerpt_length"><?php esc_html_e( 'Article Excerpt Length', 'arewa-recently-viewed-content' ); ?></label>
						</th>
						<td>
							<input
								type="number"
								id="arwrev_article_excerpt_length"
								name="arwrev_settings[article_excerpt_length]"
								value="<?php echo esc_attr( $article_excerpt_length ); ?>"
								min="50"
								max="500"
								step="1"
								class="small-text"
							/>
							<p class="description"><?php esc_html_e( 'Number of characters to show in article excerpts (50-500). Only applies to Article layout.', 'arewa-recently-viewed-content' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label><?php esc_html_e( 'Shop Layout Options', 'arewa-recently-viewed-content' ); ?></label>
						</th>
						<td>
							<fieldset>
								<label>
									<input
										type="checkbox"
										name="arwrev_settings[shop_show_price]"
										value="1"
										<?php checked( $shop_show_price, '1' ); ?>
									/>
									<?php esc_html_e( 'Show product prices', 'arewa-recently-viewed-content' ); ?>
								</label><br />

								<label>
									<input
										type="checkbox"
										name="arwrev_settings[shop_show_add_to_cart]"
										value="1"
										<?php checked( $shop_show_add_to_cart, '1' ); ?>
									/>
									<?php esc_html_e( 'Show add to cart buttons', 'arewa-recently-viewed-content' ); ?>
								</label>
							</fieldset>
							<p class="description"><?php esc_html_e( 'Configure options for the Shop layout. Only applies when WooCommerce is active.', 'arewa-recently-viewed-content' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="arwrev_title_max_chars"><?php esc_html_e( 'Title Max Characters', 'arewa-recently-viewed-content' ); ?></label>
						</th>
						<td>
							<input
								type="number"
								id="arwrev_title_max_chars"
								name="arwrev_settings[title_max_chars]"
								value="<?php echo esc_attr( $title_max_chars ); ?>"
								min="0"
								max="200"
								step="1"
								class="small-text"
							/>
							<p class="description"><?php esc_html_e( 'Maximum number of characters to display in the title (0 = no limit). Titles longer than this will be truncated with "...".', 'arewa-recently-viewed-content' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="arwrev_login_url"><?php esc_html_e( 'Login Page URL', 'arewa-recently-viewed-content' ); ?></label>
						</th>
						<td>
							<input
								type="url"
								id="arwrev_login_url"
								name="arwrev_settings[login_url]"
								value="<?php echo esc_attr( $login_url ); ?>"
								class="regular-text"
								placeholder="<?php echo esc_attr( wp_login_url() ); ?>"
							/>
							<p class="description"><?php esc_html_e( 'URL of the login page. If left empty, WordPress default login URL will be used.', 'arewa-recently-viewed-content' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="arwrev_login_button_text"><?php esc_html_e( 'Login Button Text', 'arewa-recently-viewed-content' ); ?></label>
						</th>
						<td>
							<input
								type="text"
								id="arwrev_login_button_text"
								name="arwrev_settings[login_button_text]"
								value="<?php echo esc_attr( $login_button_text ); ?>"
								class="regular-text"
								placeholder="<?php echo esc_attr__( 'Login My Account', 'arewa-recently-viewed-content' ); ?>"
							/>
							<p class="description"><?php esc_html_e( 'Text to display on the login button shown to non-logged-in users.', 'arewa-recently-viewed-content' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="arwrev_require_login_for_guests"><?php esc_html_e( 'Guest User Access', 'arewa-recently-viewed-content' ); ?></label>
						</th>
						<td>
							<input
								type="checkbox"
								id="arwrev_require_login_for_guests"
								name="arwrev_settings[require_login_for_guests]"
								value="1"
								<?php checked( isset( $settings['require_login_for_guests'] ) ? $settings['require_login_for_guests'] : '0', '1' ); ?>
							/>
							<label for="arwrev_require_login_for_guests"><?php esc_html_e( 'Require login to view watch history', 'arewa-recently-viewed-content' ); ?></label>
							<p class="description">
								<?php esc_html_e( 'When enabled, guest users (not logged in) will see a login message instead of their watch history. They will need to log in to view their history.', 'arewa-recently-viewed-content' ); ?><br />
								<?php esc_html_e( 'When disabled (default), guest users can view their watch history stored in cookies.', 'arewa-recently-viewed-content' ); ?>
							</p>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="arwrev_preserve_data"><?php esc_html_e( 'Data Preservation', 'arewa-recently-viewed-content' ); ?></label>
						</th>
						<td>
							<input
								type="checkbox"
								id="arwrev_preserve_data"
								name="arwrev_settings[preserve_data_on_uninstall]"
								value="1"
								<?php checked( $preserve_data, true ); ?>
							/>
							<label for="arwrev_preserve_data"><?php esc_html_e( 'Preserve settings and user data when plugin is uninstalled', 'arewa-recently-viewed-content' ); ?></label>
							<p class="description">
								<strong><?php esc_html_e( 'Recommended: Keep this enabled.', 'arewa-recently-viewed-content' ); ?></strong><br />
								<?php esc_html_e( 'When enabled, all plugin settings and user watch history will be preserved if you uninstall and reinstall the plugin. Disable only if you want to completely remove all data when uninstalling.', 'arewa-recently-viewed-content' ); ?>
							</p>
						</td>
					</tr>
				</tbody>
			</table>

			<?php submit_button(); ?>
		</form>

		<div style="margin-top: 30px; padding: 20px; background: #f9f9f9; border-left: 4px solid #0073aa;">
			<h3><?php esc_html_e( 'Data Management Information', 'arewa-recently-viewed-content' ); ?></h3>
			<p><strong><?php esc_html_e( 'What data is preserved:', 'arewa-recently-viewed-content' ); ?></strong></p>
			<ul style="margin-left: 20px;">
				<li><?php esc_html_e( 'All plugin settings and configuration', 'arewa-recently-viewed-content' ); ?></li>
				<li><?php esc_html_e( 'User watch history data for all users', 'arewa-recently-viewed-content' ); ?></li>
				<li><?php esc_html_e( 'Custom post type labels and layout preferences', 'arewa-recently-viewed-content' ); ?></li>
			</ul>
			<p><strong><?php esc_html_e( 'How it works:', 'arewa-recently-viewed-content' ); ?></strong></p>
			<ul style="margin-left: 20px;">
				<li><?php esc_html_e( 'When you deactivate the plugin, all data remains intact', 'arewa-recently-viewed-content' ); ?></li>
				<li><?php esc_html_e( 'When you uninstall (delete) the plugin with preservation enabled, settings and user data are kept', 'arewa-recently-viewed-content' ); ?></li>
				<li><?php esc_html_e( 'When you reinstall and reactivate, all your previous data is automatically restored', 'arewa-recently-viewed-content' ); ?></li>
				<li><?php esc_html_e( 'Only temporary data like scheduled events and caches are cleaned up', 'arewa-recently-viewed-content' ); ?></li>
			</ul>
		</div>
		<?php
	}

	/**
	 * Render Data Management tab.
	 */
	private function render_data_management_tab() {
		// Get data statistics.
		$data_manager   = ARWREV_Data_Manager::get_instance();
		$stats          = $data_manager->get_data_statistics();
		$preserve_data  = get_option( 'arwrev_preserve_data_on_uninstall', true );
		$preserved_info = get_option( 'arwrev_data_preserved', false );

		?>
		<div class="arwrev-data-management">
			<h2><?php esc_html_e( 'Data Management & Backup', 'arewa-recently-viewed-content' ); ?></h2>
			<p><?php esc_html_e( 'Monitor your plugin data and manage backup/restoration settings.', 'arewa-recently-viewed-content' ); ?></p>
			
			<!-- Data Statistics -->
			<div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin: 20px 0;">
				<div style="background: #fff; padding: 20px; border: 1px solid #ccd0d4; border-radius: 4px;">
					<h3><?php esc_html_e( 'Current Data Status', 'arewa-recently-viewed-content' ); ?></h3>
					<table class="widefat">
						<tbody>
							<tr>
								<td><strong><?php esc_html_e( 'Plugin Version:', 'arewa-recently-viewed-content' ); ?></strong></td>
								<td><?php echo esc_html( $stats['plugin_version'] ); ?></td>
							</tr>
							<tr>
								<td><strong><?php esc_html_e( 'Settings Status:', 'arewa-recently-viewed-content' ); ?></strong></td>
								<td>
									<?php if ( $stats['settings_exist'] ) : ?>
										<span style="color: #46b450;">✓ <?php esc_html_e( 'Configured', 'arewa-recently-viewed-content' ); ?></span>
									<?php else : ?>
										<span style="color: #dc3232;">✗ <?php esc_html_e( 'Not configured', 'arewa-recently-viewed-content' ); ?></span>
									<?php endif; ?>
								</td>
							</tr>
							<tr>
								<td><strong><?php esc_html_e( 'Users with History:', 'arewa-recently-viewed-content' ); ?></strong></td>
								<td><?php echo esc_html( number_format( $stats['total_users_with_history'] ) ); ?></td>
							</tr>
							<tr>
								<td><strong><?php esc_html_e( 'Total History Items:', 'arewa-recently-viewed-content' ); ?></strong></td>
								<td><?php echo esc_html( number_format( $stats['total_history_items'] ) ); ?></td>
							</tr>
							<?php if ( $stats['oldest_entry'] ) : ?>
							<tr>
								<td><strong><?php esc_html_e( 'Oldest Entry:', 'arewa-recently-viewed-content' ); ?></strong></td>
								<td><?php echo esc_html( date_i18n( get_option( 'date_format' ), $stats['oldest_entry'] ) ); ?></td>
							</tr>
							<?php endif; ?>
							<?php if ( $stats['newest_entry'] ) : ?>
							<tr>
								<td><strong><?php esc_html_e( 'Newest Entry:', 'arewa-recently-viewed-content' ); ?></strong></td>
								<td><?php echo esc_html( date_i18n( get_option( 'date_format' ), $stats['newest_entry'] ) ); ?></td>
							</tr>
							<?php endif; ?>
						</tbody>
					</table>
				</div>

				<div style="background: #fff; padding: 20px; border: 1px solid #ccd0d4; border-radius: 4px;">
					<h3><?php esc_html_e( 'Data Preservation Status', 'arewa-recently-viewed-content' ); ?></h3>
					<div style="margin-bottom: 15px;">
						<?php if ( $preserve_data ) : ?>
							<div style="padding: 10px; background: #d1edff; border-left: 4px solid #0073aa; margin-bottom: 10px;">
								<strong style="color: #0073aa;">✓ <?php esc_html_e( 'Data Preservation Enabled', 'arewa-recently-viewed-content' ); ?></strong><br>
								<small><?php esc_html_e( 'Your settings and user data will be preserved during uninstall/reinstall.', 'arewa-recently-viewed-content' ); ?></small>
							</div>
						<?php else : ?>
							<div style="padding: 10px; background: #ffebe8; border-left: 4px solid #dc3232; margin-bottom: 10px;">
								<strong style="color: #dc3232;">⚠ <?php esc_html_e( 'Data Preservation Disabled', 'arewa-recently-viewed-content' ); ?></strong><br>
								<small><?php esc_html_e( 'All data will be permanently deleted when you uninstall the plugin.', 'arewa-recently-viewed-content' ); ?></small>
							</div>
						<?php endif; ?>
					</div>

					<?php if ( $preserved_info && is_array( $preserved_info ) ) : ?>
						<h4><?php esc_html_e( 'Restoration History', 'arewa-recently-viewed-content' ); ?></h4>
						<table class="widefat" style="font-size: 12px;">
							<tbody>
								<?php if ( isset( $preserved_info['uninstalled_at'] ) ) : ?>
								<tr>
									<td><strong><?php esc_html_e( 'Last Uninstalled:', 'arewa-recently-viewed-content' ); ?></strong></td>
									<td><?php echo esc_html( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $preserved_info['uninstalled_at'] ) ); ?></td>
								</tr>
								<?php endif; ?>
								<?php if ( isset( $preserved_info['restored_at'] ) ) : ?>
								<tr>
									<td><strong><?php esc_html_e( 'Last Restored:', 'arewa-recently-viewed-content' ); ?></strong></td>
									<td><?php echo esc_html( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $preserved_info['restored_at'] ) ); ?></td>
								</tr>
								<?php endif; ?>
								<?php if ( isset( $preserved_info['previous_version'] ) ) : ?>
								<tr>
									<td><strong><?php esc_html_e( 'Previous Version:', 'arewa-recently-viewed-content' ); ?></strong></td>
									<td><?php echo esc_html( $preserved_info['previous_version'] ); ?></td>
								</tr>
								<?php endif; ?>
							</tbody>
						</table>
					<?php endif; ?>

					<p style="margin-top: 15px;">
						<a href="<?php echo esc_url( admin_url( 'admin.php?page=ar-viewed-content&tab=general' ) ); ?>" class="button">
							<?php esc_html_e( 'Change Preservation Settings', 'arewa-recently-viewed-content' ); ?>
						</a>
					</p>
				</div>
			</div>

			<!-- Manual Backup Section -->
			<div style="background: #fff; padding: 20px; border: 1px solid #ccd0d4; border-radius: 4px; margin: 20px 0;">
				<h3><?php esc_html_e( 'Manual Backup & Export', 'arewa-recently-viewed-content' ); ?></h3>
				<p><?php esc_html_e( 'Create a manual backup of all plugin data that you can save and restore later.', 'arewa-recently-viewed-content' ); ?></p>
				
				<div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-top: 15px;">
					<div>
						<h4><?php esc_html_e( 'Export Data', 'arewa-recently-viewed-content' ); ?></h4>
						<p><?php esc_html_e( 'Download a complete backup of all settings and user watch history.', 'arewa-recently-viewed-content' ); ?></p>
						<button type="button" class="button button-secondary" onclick="arwrevExportData()">
							<?php esc_html_e( 'Export All Data', 'arewa-recently-viewed-content' ); ?>
						</button>
					</div>
					
					<div>
						<h4><?php esc_html_e( 'Import Data', 'arewa-recently-viewed-content' ); ?></h4>
						<p><?php esc_html_e( 'Restore data from a previously exported backup file.', 'arewa-recently-viewed-content' ); ?></p>
						<input type="file" id="arwrev-import-file" accept=".json" style="margin-bottom: 10px;">
						<br>
						<button type="button" class="button button-secondary" onclick="arwrevImportData()">
							<?php esc_html_e( 'Import Data', 'arewa-recently-viewed-content' ); ?>
						</button>
					</div>
				</div>
			</div>

			<!-- Information Section -->
			<div style="background: #f9f9f9; padding: 20px; border-left: 4px solid #0073aa; margin: 20px 0;">
				<h3><?php esc_html_e( 'How Data Preservation Works', 'arewa-recently-viewed-content' ); ?></h3>
				<div style="display: grid; grid-template-columns: 1fr 1fr; gap: 30px;">
					<div>
						<h4><?php esc_html_e( 'Automatic Preservation (Recommended)', 'arewa-recently-viewed-content' ); ?></h4>
						<ul style="margin-left: 20px;">
							<li><?php esc_html_e( 'Enabled by default for seamless experience', 'arewa-recently-viewed-content' ); ?></li>
							<li><?php esc_html_e( 'Preserves all settings and user watch history', 'arewa-recently-viewed-content' ); ?></li>
							<li><?php esc_html_e( 'Automatically restores data when plugin is reactivated', 'arewa-recently-viewed-content' ); ?></li>
							<li><?php esc_html_e( 'Only removes temporary data like caches and scheduled events', 'arewa-recently-viewed-content' ); ?></li>
						</ul>
					</div>
					
					<div>
						<h4><?php esc_html_e( 'Manual Backup', 'arewa-recently-viewed-content' ); ?></h4>
						<ul style="margin-left: 20px;">
							<li><?php esc_html_e( 'Create exportable backup files', 'arewa-recently-viewed-content' ); ?></li>
							<li><?php esc_html_e( 'Transfer data between different WordPress installations', 'arewa-recently-viewed-content' ); ?></li>
							<li><?php esc_html_e( 'Keep offline copies of your data', 'arewa-recently-viewed-content' ); ?></li>
							<li><?php esc_html_e( 'Restore data even if automatic preservation fails', 'arewa-recently-viewed-content' ); ?></li>
						</ul>
					</div>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Render Documentation tab
	 */
	private function render_documentation_tab() {
		?>
		<div class="arwrev-documentation">
			<h2><?php esc_html_e( 'Shortcodes Documentation', 'arewa-recently-viewed-content' ); ?></h2>
			<p><?php esc_html_e( 'Learn how to use the watch history shortcodes to display recently viewed content on your website.', 'arewa-recently-viewed-content' ); ?></p>
			
			<h3><?php esc_html_e( 'Basic Usage', 'arewa-recently-viewed-content' ); ?></h3>
			<p><?php esc_html_e( 'Use the shortcode to display watch history on any page or post:', 'arewa-recently-viewed-content' ); ?></p>
			<div class="shortcode-example">
				<code>[watch_history]</code>
			</div>

			<h3><?php esc_html_e( 'Available Shortcodes', 'arewa-recently-viewed-content' ); ?></h3>
			
			<h4><?php esc_html_e( 'Main Shortcode', 'arewa-recently-viewed-content' ); ?></h4>
			<p><strong>[watch_history]</strong> - <?php esc_html_e( 'Displays the user\'s watch history with customizable options.', 'arewa-recently-viewed-content' ); ?></p>

			<h3><?php esc_html_e( 'Shortcode Attributes', 'arewa-recently-viewed-content' ); ?></h3>
			<p><?php esc_html_e( 'You can customize the display using these attributes:', 'arewa-recently-viewed-content' ); ?></p>
			
			<h4><?php esc_html_e( 'Layout Options', 'arewa-recently-viewed-content' ); ?></h4>
			<div class="shortcode-example">
				<code>[watch_history layout_type="video"]</code><br />
				<code>[watch_history layout_type="article"]</code><br />
				<code>[watch_history layout_type="shop"]</code><br />
				<code>[watch_history layout_type="carousel" post_type="movies"]</code>
			</div>

			<h4><?php esc_html_e( 'Filtering Options', 'arewa-recently-viewed-content' ); ?></h4>
			<div class="shortcode-example">
				<code>[watch_history post_type="movies"]</code><br />
				<code>[watch_history post_type="post,page"]</code><br />
				<code>[watch_history category="action,comedy"]</code><br />
				<code>[watch_history taxonomy="genre:action,comedy"]</code>
			</div>

			<h4><?php esc_html_e( 'Display Options', 'arewa-recently-viewed-content' ); ?></h4>
			<div class="shortcode-example">
				<code>[watch_history show_image="yes" show_date="yes" pagination="yes"]</code><br />
				<code>[watch_history items_per_page="20"]</code><br />
				<code>[watch_history limit="15"]</code><br />
				<code>[watch_history show_clear_all_history="no"]</code>
			</div>

			<h4><?php esc_html_e( 'Combined Examples', 'arewa-recently-viewed-content' ); ?></h4>
			<div class="shortcode-example">
				<code>[watch_history layout_type="video" post_type="movies" category="action" limit="12"]</code><br />
				<code>[watch_history layout_type="article" post_type="post" show_image="no" items_per_page="10"]</code><br />
				<code>[watch_history layout_type="shop" post_type="product" taxonomy="product_cat:electronics"]</code><br />
				<code>[watch_history layout_type="carousel" post_type="movies" limit="20" pagination="no" show_clear_all_history="no"]</code>
			</div>

			<h3><?php esc_html_e( 'Available Attributes', 'arewa-recently-viewed-content' ); ?></h3>
			<ul>
				<li><strong>layout_type:</strong> video, article, shop, carousel</li>
				<li><strong>layout:</strong> <?php esc_html_e( 'Alternative to layout_type (backward compatibility)', 'arewa-recently-viewed-content' ); ?></li>
				<li><strong>post_type:</strong> <?php esc_html_e( 'Filter by post type(s) - comma-separated', 'arewa-recently-viewed-content' ); ?></li>
				<li><strong>category:</strong> <?php esc_html_e( 'Filter by category name(s) - comma-separated', 'arewa-recently-viewed-content' ); ?></li>
				<li><strong>taxonomy:</strong> <?php esc_html_e( 'Filter by custom taxonomy (format: taxonomy:term1,term2)', 'arewa-recently-viewed-content' ); ?></li>
				<li><strong>items_per_page:</strong> <?php esc_html_e( 'Number (1-100)', 'arewa-recently-viewed-content' ); ?></li>
				<li><strong>limit:</strong> <?php esc_html_e( 'Alternative to items_per_page', 'arewa-recently-viewed-content' ); ?></li>
				<li><strong>show_image:</strong> yes, no</li>
				<li><strong>show_date:</strong> yes, no</li>
				<li><strong>pagination:</strong> yes, no</li>
				<li><strong>show_clear_all_history:</strong> yes, no (<?php esc_html_e( 'Show/hide Clear All History button - default: yes', 'arewa-recently-viewed-content' ); ?>)</li>
			</ul>

			<h3><?php esc_html_e( 'Layout Types', 'arewa-recently-viewed-content' ); ?></h3>
			<ul>
				<li><strong><?php esc_html_e( 'Video Layout:', 'arewa-recently-viewed-content' ); ?></strong> <?php esc_html_e( 'Card-based layout with overlay text, perfect for video content', 'arewa-recently-viewed-content' ); ?></li>
				<li><strong><?php esc_html_e( 'Article Layout:', 'arewa-recently-viewed-content' ); ?></strong> <?php esc_html_e( 'Clean blog-style layout with excerpts, ideal for articles and posts', 'arewa-recently-viewed-content' ); ?></li>
				<li><strong><?php esc_html_e( 'Shop Layout:', 'arewa-recently-viewed-content' ); ?></strong> <?php esc_html_e( 'Product-focused layout with pricing and cart buttons (requires WooCommerce)', 'arewa-recently-viewed-content' ); ?></li>
				<li><strong><?php esc_html_e( 'Carousel Layout:', 'arewa-recently-viewed-content' ); ?></strong> <?php esc_html_e( 'Horizontal scrolling carousel layout, perfect for showcasing content', 'arewa-recently-viewed-content' ); ?></li>
			</ul>

			<h3><?php esc_html_e( 'Auto Post Type Inference', 'arewa-recently-viewed-content' ); ?></h3>
			<p><?php esc_html_e( 'If post_type is not specified, it will be automatically inferred from the layout:', 'arewa-recently-viewed-content' ); ?></p>
			<ul>
				<li><strong>layout="video"</strong> → post_type="movies"</li>
				<li><strong>layout="article"</strong> → post_type="post"</li>
				<li><strong>layout="shop"</strong> → post_type="product"</li>
				<li><strong>layout="carousel"</strong> → <?php esc_html_e( 'no specific post_type (shows all tracked content)', 'arewa-recently-viewed-content' ); ?></li>
			</ul>

			<h3><?php esc_html_e( 'Advanced Examples', 'arewa-recently-viewed-content' ); ?></h3>
			<p><?php esc_html_e( 'Here are some advanced usage examples:', 'arewa-recently-viewed-content' ); ?></p>
			
			<h4><?php esc_html_e( 'Video Gallery Style:', 'arewa-recently-viewed-content' ); ?></h4>
			<div class="shortcode-example">
				<code>[watch_history layout_type="video" items_per_page="12" show_date="yes"]</code>
			</div>
			
			<h4><?php esc_html_e( 'Magazine-Style Articles:', 'arewa-recently-viewed-content' ); ?></h4>
			<div class="shortcode-example">
				<code>[watch_history layout_type="article" items_per_page="8" show_image="yes"]</code>
			</div>
			
			<h4><?php esc_html_e( 'News Feed Style:', 'arewa-recently-viewed-content' ); ?></h4>
			<div class="shortcode-example">
				<code>[watch_history layout_type="article" post_type="post" category="news,technology" limit="6"]</code>
			</div>
			
			<h4><?php esc_html_e( 'Product Grid:', 'arewa-recently-viewed-content' ); ?></h4>
			<div class="shortcode-example">
				<code>[watch_history layout_type="shop" items_per_page="6" show_date="no"]</code>
			</div>
			
			<h4><?php esc_html_e( 'Content Carousel:', 'arewa-recently-viewed-content' ); ?></h4>
			<div class="shortcode-example">
				<code>[watch_history layout_type="carousel" post_type="movies" items_per_page="8"]</code>
			</div>

			<h3><?php esc_html_e( 'Tips & Best Practices', 'arewa-recently-viewed-content' ); ?></h3>
			<ul>
				<li><strong><?php esc_html_e( 'Performance:', 'arewa-recently-viewed-content' ); ?></strong> <?php esc_html_e( 'Use pagination for better performance with large histories', 'arewa-recently-viewed-content' ); ?></li>
				<li><strong><?php esc_html_e( 'Mobile:', 'arewa-recently-viewed-content' ); ?></strong> <?php esc_html_e( 'All layouts automatically adjust for mobile devices', 'arewa-recently-viewed-content' ); ?></li>
				<li><strong><?php esc_html_e( 'WooCommerce:', 'arewa-recently-viewed-content' ); ?></strong> <?php esc_html_e( 'Shop layout only works when WooCommerce is active', 'arewa-recently-viewed-content' ); ?></li>
				<li><strong><?php esc_html_e( 'Filtering:', 'arewa-recently-viewed-content' ); ?></strong> <?php esc_html_e( 'Use post_type to show only specific content types', 'arewa-recently-viewed-content' ); ?></li>
				<li><strong><?php esc_html_e( 'Categories:', 'arewa-recently-viewed-content' ); ?></strong> <?php esc_html_e( 'Filter by category names or slugs (case-sensitive)', 'arewa-recently-viewed-content' ); ?></li>
				<li><strong><?php esc_html_e( 'Taxonomies:', 'arewa-recently-viewed-content' ); ?></strong> <?php esc_html_e( 'Use format "taxonomy:term1,term2" for custom taxonomies', 'arewa-recently-viewed-content' ); ?></li>
				<li><strong><?php esc_html_e( 'Multiple Values:', 'arewa-recently-viewed-content' ); ?></strong> <?php esc_html_e( 'Separate multiple values with commas', 'arewa-recently-viewed-content' ); ?></li>
				<li><strong><?php esc_html_e( 'Auto-Inference:', 'arewa-recently-viewed-content' ); ?></strong> <?php esc_html_e( 'Layout automatically determines post_type if not specified', 'arewa-recently-viewed-content' ); ?></li>
				<li><strong><?php esc_html_e( 'Responsive:', 'arewa-recently-viewed-content' ); ?></strong> <?php esc_html_e( 'All layouts are fully responsive and mobile-friendly', 'arewa-recently-viewed-content' ); ?></li>
			</ul>
		</div>
		<?php // phpcs:enable WordPress.WhiteSpace.PrecisionAlignment.Found ?>
		<?php
	}
}
