<?php
/**
 * Layout Manager
 *
 * Handles different layout types and display modes
 *
 * @package Arewa_Recently_Viewed_Content
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Ensure WordPress functions are available.
if ( ! function_exists( 'add_action' ) ) {
	return;
}

/**
 * Layout Manager Class.
 *
 * Manages different layout types and display modes.
 *
 * @package Arewa_Recently_Viewed_Content
 */
class ARWREV_Layout_Manager {

	/**
	 * Instance of this class
	 *
	 * @var object
	 */
	private static $instance = null;

	/**
	 * Available layout types
	 *
	 * @var array
	 */
	private $layout_types = array();

	/**
	 * Available display modes
	 *
	 * @var array
	 */
	private $display_modes = array();

	/**
	 * Get instance of this class
	 *
	 * @return ARWREV_Layout_Manager
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->init_layout_types();
		$this->init_display_modes();
	}

	/**
	 * Initialize layout types
	 */
	private function init_layout_types() {
		$this->layout_types = array(
			'video'    => array(
				'label'       => 'Video Layout', // No translation here.
				'description' => 'Card-based layout with overlay text, perfect for video content',
				'supports'    => array( 'grid' ),
			),
			'article'  => array(
				'label'       => 'Article Layout',
				'description' => 'Clean blog-style layout with excerpts, ideal for articles and posts',
				'supports'    => array( 'list' ),
			),
			'shop'     => array(
				'label'       => 'Shop Layout',
				'description' => 'Product-focused layout with pricing and cart buttons',
				'supports'    => array( 'grid' ),
				'requires'    => 'woocommerce',
			),
			'carousel' => array(
				'label'       => 'Carousel Layout',
				'description' => 'Horizontal scrolling carousel layout, perfect for showcasing content',
				'supports'    => array( 'carousel' ),
			),
		);

		// Allow plugins to add custom layout types.
		$this->layout_types = apply_filters( 'arwrev_layout_types', $this->layout_types );
	}

	/**
	 * Initialize display modes
	 */
	private function init_display_modes() {
		$this->display_modes = array(
			'grid'     => array(
				'label'       => 'Grid',
				'description' => 'Items displayed in a responsive grid layout',
			),
			'list'     => array(
				'label'       => 'List',
				'description' => 'Items displayed in a vertical list layout',
			),
			'carousel' => array(
				'label'       => 'Carousel',
				'description' => 'Items displayed in a horizontal scrolling carousel',
			),
		);

		// Allow plugins to add custom display modes.
		$this->display_modes = apply_filters( 'arwrev_display_modes', $this->display_modes );
	}

	/**
	 * Get available layout types
	 *
	 * @return array
	 */
	public function get_layout_types() {
		return $this->layout_types;
	}

	/**
	 * Get available display modes
	 *
	 * @return array
	 */
	public function get_display_modes() {
		return $this->display_modes;
	}

	/**
	 * Get supported display modes for a layout type
	 *
	 * @param string $layout_type Layout type.
	 * @return array
	 */
	public function get_supported_display_modes( $layout_type ) {
		if ( ! isset( $this->layout_types[ $layout_type ] ) ) {
			return array();
		}

		return isset( $this->layout_types[ $layout_type ]['supports'] ) ? $this->layout_types[ $layout_type ]['supports'] : array();
	}

	/**
	 * Check if layout type is available
	 *
	 * @param string $layout_type Layout type.
	 * @return bool
	 */
	public function is_layout_available( $layout_type ) {
		if ( ! isset( $this->layout_types[ $layout_type ] ) ) {
			return false;
		}

		// Check if layout has requirements.
		if ( isset( $this->layout_types[ $layout_type ]['requires'] ) ) {
			$requirement = $this->layout_types[ $layout_type ]['requires'];

			switch ( $requirement ) {
				case 'woocommerce':
					return class_exists( 'WooCommerce' );
				default:
					return apply_filters( 'arwrev_layout_requirement_check', true, $requirement, $layout_type );
			}
		}

		return true;
	}

	/**
	 * Get layout renderer class name
	 *
	 * @param string $layout_type Layout type.
	 * @return string|false
	 */
	public function get_layout_renderer( $layout_type ) {
		$class_map = array(
			'video'    => 'ARWREV_Video_Layout',
			'article'  => 'ARWREV_Article_Layout',
			'shop'     => 'ARWREV_Shop_Layout',
			'carousel' => 'ARWREV_Carousel_Layout',
		);

		$class_map = apply_filters( 'arwrev_layout_renderers', $class_map );

		return isset( $class_map[ $layout_type ] ) ? $class_map[ $layout_type ] : false;
	}

	/**
	 * Render layout
	 *
	 * @param array $items History items.
	 * @param array $settings Plugin settings.
	 * @param array $atts Shortcode attributes.
	 * @return string
	 */
	public function render_layout( $items, $settings, $atts ) {
		$layout_type  = isset( $settings['layout_type'] ) ? $settings['layout_type'] : 'video';
		$display_mode = isset( $settings['display_mode'] ) ? $settings['display_mode'] : 'grid';

		// Fallback to video layout if selected layout is not available.
		if ( ! $this->is_layout_available( $layout_type ) ) {
			$layout_type = 'video';
		}

		// Check if display mode is supported by layout type.
		$supported_modes = $this->get_supported_display_modes( $layout_type );
		if ( ! empty( $supported_modes ) && ! in_array( $display_mode, $supported_modes, true ) ) {
			$display_mode = $supported_modes[0]; // Use first supported mode.
		}

		$renderer_class = $this->get_layout_renderer( $layout_type );

		if ( ! $renderer_class || ! class_exists( $renderer_class ) ) {
			// Fallback to video layout.
			$renderer_class = 'ARWREV_Video_Layout';
		}

		if ( class_exists( $renderer_class ) ) {
			$renderer = new $renderer_class();
			return $renderer->render( $items, $settings, $atts, $display_mode );
		}

		return '<p class="arwrev-error">Layout renderer not found.</p>';
	}
}
