<?php
/**
 * Base Layout Class
 *
 * Abstract base class for all layout types
 *
 * @package Arewa_Recently_Viewed_Content
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Ensure WordPress functions are available.
if ( ! function_exists( 'get_post' ) ) {
	return;
}

/**
 * Base Layout Class.
 *
 * Abstract base class that provides common functionality for all layout types.
 *
 * @package Arewa_Recently_Viewed_Content
 */
abstract class ARWREV_Base_Layout {

	/**
	 * Layout type identifier
	 *
	 * @var string
	 */
	protected $layout_type = '';

	/**
	 * Supported display modes
	 *
	 * @var array
	 */
	protected $supported_modes = array();

	/**
	 * Render the layout
	 *
	 * @param array  $items History items.
	 * @param array  $settings Plugin settings.
	 * @param array  $atts Shortcode attributes.
	 * @param string $display_mode Display mode.
	 * @return string
	 */
	abstract public function render( $items, $settings, $atts, $display_mode );

	/**
	 * Get container CSS classes
	 *
	 * @param string $display_mode Display mode.
	 * @param array  $settings Plugin settings.
	 * @return string
	 */
	protected function get_container_classes( $display_mode, $settings ) {
		$classes = array(
			'arwrev-container',
			'arwrev-layout-' . $this->layout_type,
			'arwrev-mode-' . $display_mode,
		);

		// Add theme compatibility classes.
		$theme     = get_template();
		$classes[] = 'arwrev-theme-' . sanitize_html_class( $theme );

		return implode( ' ', array_filter( $classes ) );
	}

	/**
	 * Get item CSS classes
	 *
	 * @param array  $item History item.
	 * @param string $display_mode Display mode.
	 * @return string
	 */
	protected function get_item_classes( $item, $display_mode ) {
		$post_type = isset( $item['post_type'] ) ? $item['post_type'] : '';

		$classes = array(
			'arwrev-item',
			'arwrev-item-' . $this->layout_type,
			'arwrev-item-mode-' . $display_mode,
		);

		if ( $post_type ) {
			$classes[] = 'arwrev-post-type-' . sanitize_html_class( $post_type );
		}

		return implode( ' ', array_filter( $classes ) );
	}

	/**
	 * Get post type label
	 *
	 * @param string $post_type Post type.
	 * @param array  $settings Plugin settings.
	 * @return string
	 */
	protected function get_post_type_label( $post_type, $settings ) {
		// Check for custom labels first.
		$custom_labels = isset( $settings['post_type_labels'] ) ? $settings['post_type_labels'] : array();
		if ( isset( $custom_labels[ $post_type ] ) && ! empty( $custom_labels[ $post_type ] ) ) {
			return $custom_labels[ $post_type ];
		}

		// Get default post type label.
		$post_type_obj = get_post_type_object( $post_type );
		return $post_type_obj ? $post_type_obj->labels->singular_name : $post_type;
	}

	/**
	 * Get formatted date
	 *
	 * @param int $timestamp Timestamp.
	 * @return string
	 */
	protected function get_formatted_date( $timestamp ) {
		if ( ! $timestamp ) {
			return '';
		}

		$date_format = get_option( 'date_format' );
		return date_i18n( $date_format, $timestamp );
	}

	/**
	 * Get truncated title
	 *
	 * @param string $title Original title.
	 * @param int    $max_chars Maximum characters (0 for no limit).
	 * @return string
	 */
	protected function get_truncated_title( $title, $max_chars ) {
		if ( $max_chars > 0 && mb_strlen( $title ) > $max_chars ) {
			return mb_substr( $title, 0, $max_chars ) . '...';
		}
		return $title;
	}

	/**
	 * Get post excerpt
	 *
	 * @param int $post_id Post ID.
	 * @param int $length Excerpt length.
	 * @return string
	 */
	protected function get_post_excerpt( $post_id, $length = 150 ) {
		$post = get_post( $post_id );
		if ( ! $post ) {
			return '';
		}

		// Use manual excerpt if available.
		if ( ! empty( $post->post_excerpt ) ) {
			$excerpt = $post->post_excerpt;
		} else {
			// Generate excerpt from content.
			$excerpt = wp_strip_all_tags( $post->post_content );
		}

		// Truncate to specified length.
		if ( mb_strlen( $excerpt ) > $length ) {
			$excerpt = mb_substr( $excerpt, 0, $length );
			// Try to break at word boundary.
			$last_space = mb_strrpos( $excerpt, ' ' );
			if ( false !== $last_space && $last_space > $length * 0.8 ) {
				$excerpt = mb_substr( $excerpt, 0, $last_space );
			}
			$excerpt .= '...';
		}

		return $excerpt;
	}

	/**
	 * Get posts in history order using WP_Query with post__in and orderby post__in.
	 * Re-maps query results to the original history order. Never use orderby date.
	 *
	 * @param array  $items     History items (each with post_id, post_type, timestamp).
	 * @param string $post_type Optional. Limit query to this post type (e.g. 'product' for shop).
	 * @return array Array of array( 'post' => WP_Post, 'item' => history_item ) in history order.
	 */
	protected function get_posts_in_history_order( $items, $post_type = '' ) {
		if ( empty( $items ) ) {
			return array();
		}

		// Extract post IDs while preserving order.
		$post_ids    = array();
		$items_by_id = array();

		foreach ( $items as $item ) {
			$post_id = isset( $item['post_id'] ) ? (int) $item['post_id'] : 0;
			if ( $post_id > 0 ) {
				// Only add if not already present (avoid duplicates but preserve first occurrence).
				if ( ! isset( $items_by_id[ $post_id ] ) ) {
					$post_ids[]              = $post_id;
					$items_by_id[ $post_id ] = $item;
				}
			}
		}

		if ( empty( $post_ids ) ) {
			return array();
		}

		$query_args = array(
			'post_type'      => $post_type ? $post_type : 'any',
			'post__in'       => $post_ids,
			'orderby'        => 'post__in',
			// phpcs:ignore WordPress.WP.PostsPerPage.posts_per_page_posts_per_page -- History list is limited by shortcode; no arbitrary limit.
			'posts_per_page' => count( $post_ids ),
			'post_status'    => 'publish',
			'no_found_rows'  => true,
		);

		$query       = new \WP_Query( $query_args );
		$posts_by_id = array();
		if ( $query->have_posts() ) {
			foreach ( $query->posts as $post ) {
				$posts_by_id[ (int) $post->ID ] = $post;
			}
		}
		wp_reset_postdata();

		// Re-map to original history order (not WP_Query order).
		// This ensures that the most recently viewed items appear first.
		$ordered = array();
		foreach ( $post_ids as $post_id ) {
			if ( isset( $posts_by_id[ $post_id ] ) && isset( $items_by_id[ $post_id ] ) ) {
				$ordered[] = array(
					'post' => $posts_by_id[ $post_id ],
					'item' => $items_by_id[ $post_id ],
				);
			}
		}

		return $ordered;
	}

	/**
	 * Check if layout supports display mode
	 *
	 * @param string $display_mode Display mode.
	 * @return bool
	 */
	public function supports_display_mode( $display_mode ) {
		return in_array( $display_mode, $this->supported_modes, true );
	}

	/**
	 * Get supported display modes
	 *
	 * @return array
	 */
	public function get_supported_modes() {
		return $this->supported_modes;
	}

	/**
	 * Get layout type
	 *
	 * @return string
	 */
	public function get_layout_type() {
		return $this->layout_type;
	}
}
