<?php
/**
 * Carousel Layout Class
 *
 * Handles carousel-style layout rendering
 *
 * @package Arewa_Recently_Viewed_Content
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Ensure base class is loaded.
if ( ! class_exists( 'ARWREV_Base_Layout' ) ) {
	require_once ARWREV_PLUGIN_DIR . 'includes/layouts/class-arwrev-base-layout.php';
}

/**
 * Carousel Layout Class.
 *
 * Renders items in a horizontal scrolling carousel layout.
 *
 * @package Arewa_Recently_Viewed_Content
 */
class ARWREV_Carousel_Layout extends ARWREV_Base_Layout {

	/**
	 * Layout type identifier
	 *
	 * @var string
	 */
	protected $layout_type = 'carousel';

	/**
	 * Supported display modes
	 *
	 * @var array
	 */
	protected $supported_modes = array( 'carousel' );

	/**
	 * Render the carousel layout
	 *
	 * @param array  $items History items.
	 * @param array  $settings Plugin settings.
	 * @param array  $atts Shortcode attributes.
	 * @param string $display_mode Display mode.
	 * @return string
	 */
	public function render( $items, $settings, $atts, $display_mode ) {
		if ( empty( $items ) ) {
			return '<div class="arwrev-container"><p class="arwrev-message">' . esc_html__( 'Your watch history is empty.', 'arewa-recently-viewed-content' ) . '</p></div>';
		}

		// Preserve history order: WP_Query post__in + orderby post__in, re-map to history order. Do not re-sort in JS.
		$ordered = $this->get_posts_in_history_order( $items );
		if ( empty( $ordered ) ) {
			return '<div class="arwrev-container"><p class="arwrev-message">' . esc_html__( 'Your watch history is empty.', 'arewa-recently-viewed-content' ) . '</p></div>';
		}

		$image_width     = isset( $settings['image_width'] ) ? (int) $settings['image_width'] : 155;
		$image_height    = isset( $settings['image_height'] ) ? (int) $settings['image_height'] : 250;
		$title_max_chars = isset( $settings['title_max_chars'] ) ? (int) $settings['title_max_chars'] : 0;

		$show_image = isset( $atts['show_image'] ) ? 'yes' === strtolower( $atts['show_image'] ) : true;
		$show_date  = isset( $atts['show_date'] ) ? 'yes' === strtolower( $atts['show_date'] ) : true;

		ob_start();
		?>
		<?php // phpcs:disable WordPress.WhiteSpace.PrecisionAlignment.Found -- HTML template alignment. ?>
		<div class="<?php echo esc_attr( $this->get_container_classes( 'carousel', $settings ) ); ?>">
			<div class="arwrev-carousel-container">
				<div class="arwrev-carousel-wrapper">
					<div class="arwrev-carousel-track">
						<?php foreach ( $ordered as $entry ) : ?>
							<?php
							// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Method handles escaping internally
							echo $this->render_carousel_item( $entry['item'], $settings, $show_image, $show_date, $image_width, $image_height, $title_max_chars, $entry['post'] );
							?>
						<?php endforeach; ?>
					</div>
				</div>

				<!-- Navigation Buttons -->
				<button class="arwrev-carousel-nav arwrev-carousel-prev" aria-label="<?php esc_attr_e( 'Previous items', 'arewa-recently-viewed-content' ); ?>">
					<svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
						<path d="M15 18L9 12L15 6"></path>
					</svg>
				</button>
				<button class="arwrev-carousel-nav arwrev-carousel-next" aria-label="<?php esc_attr_e( 'Next items', 'arewa-recently-viewed-content' ); ?>">
					<svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
						<path d="M9 18L15 12L9 6"></path>
					</svg>
				</button>

				<!-- Dots Indicator -->
				<div class="arwrev-carousel-dots"></div>
			</div>
		</div>
		<?php // phpcs:enable WordPress.WhiteSpace.PrecisionAlignment.Found ?>
		<?php
		return ob_get_clean();
	}

	/**
	 * Render individual carousel item
	 *
	 * @param array  $item History item.
	 * @param array  $settings Plugin settings.
	 * @param bool   $show_image Show image flag.
	 * @param bool   $show_date Show date flag.
	 * @param int    $image_width Image width.
	 * @param int    $image_height Image height.
	 * @param int    $title_max_chars Title max characters.
	 * @param object $post            Optional. WP_Post from query (avoids extra get_post).
	 * @return string
	 */
	private function render_carousel_item( $item, $settings, $show_image, $show_date, $image_width, $image_height, $title_max_chars, $post = null ) {
		$post_id   = isset( $item['post_id'] ) ? (int) $item['post_id'] : 0;
		$post_type = isset( $item['post_type'] ) ? $item['post_type'] : '';
		$timestamp = isset( $item['timestamp'] ) ? (int) $item['timestamp'] : 0;

		if ( ! $post ) {
			$post = get_post( $post_id );
		}
		if ( ! $post ) {
			return '';
		}

		$post_type_label = $this->get_post_type_label( $post_type, $settings );
		$date_str        = $show_date ? $this->get_formatted_date( $timestamp ) : '';
		$title           = $this->get_truncated_title( get_the_title( $post_id ), $title_max_chars );

		// Get categories for the post.
		$categories = array();
		if ( 'product' === $post_type && function_exists( 'wc_get_product' ) ) {
			// WooCommerce product categories.
			$product_cats = wp_get_post_terms( $post_id, 'product_cat', array( 'fields' => 'names' ) );
			if ( ! is_wp_error( $product_cats ) && ! empty( $product_cats ) ) {
				$categories = array_slice( $product_cats, 0, 2 ); // Max 2 categories.
			}
		} else {
			// Regular post categories.
			$post_cats = wp_get_post_terms( $post_id, 'category', array( 'fields' => 'names' ) );
			if ( ! is_wp_error( $post_cats ) && ! empty( $post_cats ) ) {
				$categories = array_slice( $post_cats, 0, 2 ); // Max 2 categories.
			}
		}

		// Get price for WooCommerce products.
		$price_html      = '';
		$add_to_cart_url = '';
		if ( 'product' === $post_type && function_exists( 'wc_get_product' ) ) {
			$product = wc_get_product( $post_id );
			if ( $product ) {
				$price_html      = $product->get_price_html();
				$add_to_cart_url = $product->add_to_cart_url();
			}
		} else {
			// For non-products, show a generic "Free" or custom price.
			$custom_price = get_post_meta( $post_id, '_custom_price', true );
			if ( ! empty( $custom_price ) ) {
				$price_html = '<span class="amount">' . esc_html( $custom_price ) . '</span>';
			} else {
				$price_html = '<span class="amount">' . esc_html__( 'Free', 'arewa-recently-viewed-content' ) . '</span>';
			}
		}

		ob_start();
		?>
		<?php // phpcs:disable WordPress.WhiteSpace.PrecisionAlignment.Found -- HTML template alignment. ?>
		<div class="<?php echo esc_attr( $this->get_item_classes( $item, 'carousel' ) ); ?>"
			data-post-id="<?php echo esc_attr( $post_id ); ?>">

			<div class="arwrev-carousel-card">
				<div class="arwrev-carousel-image-container">
					<a href="<?php echo esc_url( get_permalink( $post_id ) ); ?>" class="arwrev-carousel-image-link">
						<?php
						if ( has_post_thumbnail( $post_id ) ) {
							$thumbnail_id = get_post_thumbnail_id( $post_id );
							// Get the highest quality image available - try full size first.
							$image_src = wp_get_attachment_image_src( $thumbnail_id, 'full' );

							if ( ! $image_src || empty( $image_src[0] ) ) {
								$image_src = wp_get_attachment_image_src( $thumbnail_id, 'large' );
							}

							if ( $image_src && ! empty( $image_src[0] ) ) {
								printf(
									'<img src="%s" alt="%s" class="arwrev-carousel-image" loading="lazy" />',
									esc_url( $image_src[0] ),
									esc_attr( get_the_title( $post_id ) )
								);
							} else {
								// Fallback to WordPress generated thumbnail with full size.
								echo get_the_post_thumbnail(
									$post_id,
									'full',
									array(
										'alt'     => esc_attr( get_the_title( $post_id ) ),
										'class'   => 'arwrev-carousel-image',
										'loading' => 'lazy',
									)
								);
							}
						} else {
							echo '<div class="arwrev-carousel-placeholder"></div>';
						}
						?>
					</a>

					<!-- Gradient Overlay - Covers full image -->
					<div class="arwrev-carousel-overlay"></div>

					<!-- Top Meta (Categories & Date) - On image overlay -->
					<div class="arwrev-carousel-top-meta">
						<?php if ( ! empty( $categories ) ) : ?>
							<div class="arwrev-carousel-categories">
								<?php foreach ( $categories as $category ) : ?>
									<span class="arwrev-carousel-category"><?php echo esc_html( $category ); ?></span>
								<?php endforeach; ?>
							</div>
						<?php endif; ?>
						<?php if ( $date_str ) : ?>
							<div class="arwrev-carousel-date"><?php echo esc_html( $date_str ); ?></div>
						<?php endif; ?>
					</div>

					<!-- Content Overlay - Over image with gradient background -->
					<div class="arwrev-carousel-content">
						<h3 class="arwrev-carousel-title">
							<a href="<?php echo esc_url( get_permalink( $post_id ) ); ?>">
								<?php echo esc_html( $title ); ?>
							</a>
						</h3>

						<?php if ( ! empty( $price_html ) ) : ?>
							<div class="arwrev-carousel-price">
								<?php echo wp_kses_post( $price_html ); ?>
							</div>
						<?php endif; ?>

						<div class="arwrev-carousel-actions">
							<?php if ( ! empty( $add_to_cart_url ) ) : ?>
								<a href="<?php echo esc_url( $add_to_cart_url ); ?>" class="arwrev-carousel-btn arwrev-carousel-btn-cart">
									<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
										<circle cx="9" cy="21" r="1"></circle>
										<circle cx="20" cy="21" r="1"></circle>
										<path d="m1 1 4 4 2.68 13.39a2 2 0 0 0 2 1.61h9.72a2 2 0 0 0 2-1.61L23 6H6"></path>
									</svg>
									<?php esc_html_e( 'Add to Cart', 'arewa-recently-viewed-content' ); ?>
								</a>
							<?php endif; ?>

							<!-- Always show View button -->
							<a href="<?php echo esc_url( get_permalink( $post_id ) ); ?>" class="arwrev-carousel-btn arwrev-carousel-btn-view">
								<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
									<path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path>
									<circle cx="12" cy="12" r="3"></circle>
								</svg>
								<?php
								if ( 'product' === $post_type ) {
									esc_html_e( 'View Product', 'arewa-recently-viewed-content' );
								} else {
									esc_html_e( 'View Details', 'arewa-recently-viewed-content' );
								}
								?>
							</a>
						</div>
					</div>

					<!-- Remove Button -->
					<button type="button" class="arwrev-carousel-remove"
						data-post-id="<?php echo esc_attr( $post_id ); ?>"
						data-nonce="<?php echo esc_attr( wp_create_nonce( 'arwrev_remove_item_' . $post_id ) ); ?>"
						aria-label="<?php esc_attr_e( 'Remove this item', 'arewa-recently-viewed-content' ); ?>">
						<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
							<path d="M18 6L6 18M6 6l12 12"></path>
						</svg>
					</button>
				</div>
			</div>
		</div>
		<?php // phpcs:enable WordPress.WhiteSpace.PrecisionAlignment.Found ?>
		<?php
		return ob_get_clean();
	}
}
