<?php
/**
 * Shop Layout Class
 *
 * Handles WooCommerce product layout rendering
 *
 * @package Arewa_Recently_Viewed_Content
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Ensure base class is loaded.
if ( ! class_exists( 'ARWREV_Base_Layout' ) ) {
	require_once ARWREV_PLUGIN_DIR . 'includes/layouts/class-arwrev-base-layout.php';
}

/**
 * Shop Layout Class.
 *
 * Renders WooCommerce products in grid or carousel layout.
 *
 * @package Arewa_Recently_Viewed_Content
 */
class ARWREV_Shop_Layout extends ARWREV_Base_Layout {

	/**
	 * Layout type identifier
	 *
	 * @var string
	 */
	protected $layout_type = 'shop';

	/**
	 * Supported display modes
	 *
	 * @var array
	 */
	protected $supported_modes = array( 'grid' );

	/**
	 * Render the shop layout
	 *
	 * @param array  $items History items.
	 * @param array  $settings Plugin settings.
	 * @param array  $atts Shortcode attributes.
	 * @param string $display_mode Display mode.
	 * @return string
	 */
	public function render( $items, $settings, $atts, $display_mode ) {
		// Check if WooCommerce is active.
		if ( ! class_exists( 'WooCommerce' ) ) {
			return '<div class="arwrev-container"><p class="arwrev-message">' . esc_html__( 'WooCommerce is required for shop layout.', 'arewa-recently-viewed-content' ) . '</p></div>';
		}

		if ( empty( $items ) ) {
			return '<div class="arwrev-container"><p class="arwrev-message">' . esc_html__( 'Your shopping history is empty.', 'arewa-recently-viewed-content' ) . '</p></div>';
		}

		// Filter items to only include products; preserve history order.
		$product_items = array_filter( $items, array( $this, 'is_product_item' ) );
		$product_items = array_values( $product_items );

		if ( empty( $product_items ) ) {
			return '<div class="arwrev-container"><p class="arwrev-message">' . esc_html__( 'No products in your history.', 'arewa-recently-viewed-content' ) . '</p></div>';
		}

		// Preserve history order: WP_Query post_type=product, post__in, orderby post__in, re-map to history order.
		$ordered = $this->get_posts_in_history_order( $product_items, 'product' );
		if ( empty( $ordered ) ) {
			return '<div class="arwrev-container"><p class="arwrev-message">' . esc_html__( 'No products in your history.', 'arewa-recently-viewed-content' ) . '</p></div>';
		}

		ob_start();
		?>
		<div class="arwrev-container arwrev-shop-container">
			<div class="arwrev-shop-grid">
				<?php foreach ( $ordered as $entry ) : ?>
					<?php
					// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Method handles escaping internally
					echo $this->render_product_item( $entry['item'], $settings, $entry['post'] );
					?>
				<?php endforeach; ?>
			</div>
		</div>
		<?php
		return ob_get_clean();
	}

	/**
	 * Check if item is a product
	 *
	 * @param array $item History item.
	 * @return bool
	 */
	private function is_product_item( $item ) {
		return isset( $item['post_type'] ) && 'product' === $item['post_type'];
	}

	/**
	 * Render individual product item
	 *
	 * @param array  $item History item.
	 * @param array  $settings Plugin settings.
	 * @param object $post    Optional. WP_Post from query (avoids extra lookup).
	 * @return string
	 */
	private function render_product_item( $item, $settings, $post = null ) {
		$post_id   = isset( $item['post_id'] ) ? (int) $item['post_id'] : 0;
		$timestamp = isset( $item['timestamp'] ) ? (int) $item['timestamp'] : 0;

		$product = $post ? wc_get_product( $post ) : wc_get_product( $post_id );
		if ( ! $product ) {
			return '';
		}

		$show_price       = isset( $settings['shop_show_price'] ) ? '1' === $settings['shop_show_price'] : true;
		$show_add_to_cart = isset( $settings['shop_show_add_to_cart'] ) ? '1' === $settings['shop_show_add_to_cart'] : true;

		$title      = $product->get_name();
		$permalink  = $product->get_permalink();
		$price_html = $show_price ? $product->get_price_html() : '';

		ob_start();
		?>
		<div class="arwrev-shop-card" data-post-id="<?php echo esc_attr( $post_id ); ?>">
			<!-- Product Image -->
			<div class="arwrev-shop-image">
				<a href="<?php echo esc_url( $permalink ); ?>" class="arwrev-shop-link">
					<?php
					$image_id = $product->get_image_id();
					if ( $image_id ) {
						echo wp_get_attachment_image(
							$image_id,
							'woocommerce_thumbnail',
							false,
							array(
								'alt'     => esc_attr( $title ),
								'loading' => 'lazy',
								'class'   => 'arwrev-shop-thumbnail',
							)
						);
					} else {
						echo '<div class="arwrev-shop-placeholder">';
						echo '<svg width="48" height="48" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">';
						echo '<path d="M21 19V5C21 3.9 20.1 3 19 3H5C3.9 3 3 3.9 3 5V19C3 20.1 3.9 21 5 21H19C20.1 21 21 20.1 21 19ZM8.5 13.5L11 16.51L14.5 12L19 18H5L8.5 13.5Z" fill="currentColor"/>';
						echo '</svg>';
						echo '<span>' . esc_html__( 'No Image', 'arewa-recently-viewed-content' ) . '</span>';
						echo '</div>';
					}
					?>
				</a>

				<!-- Sale Badge -->
				<?php if ( $product->is_on_sale() ) : ?>
					<div class="arwrev-shop-sale-badge">
						<?php esc_html_e( 'Sale!', 'arewa-recently-viewed-content' ); ?>
					</div>
				<?php endif; ?>

				<!-- Recently Viewed Badge -->
				<div class="arwrev-shop-viewed-badge">
					<svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
						<path d="M12 4.5C7 4.5 2.73 7.61 1 12C2.73 16.39 7 19.5 12 19.5S21.27 16.39 23 12C21.27 7.61 17 4.5 12 4.5ZM12 17C9.24 17 7 14.76 7 12S9.24 7 12 7S17 9.24 17 12S14.76 17 12 17ZM12 9C10.34 9 9 10.34 9 12S10.34 15 12 15S15 13.66 15 12S13.66 9 12 9Z" fill="currentColor"/>
					</svg>
					<span><?php esc_html_e( 'Recently Viewed', 'arewa-recently-viewed-content' ); ?></span>
				</div>

				<!-- Remove Button -->
				<button type="button" class="arwrev-shop-remove-btn"
					data-post-id="<?php echo esc_attr( $post_id ); ?>"
					data-nonce="<?php echo esc_attr( wp_create_nonce( 'arwrev_remove_item_' . $post_id ) ); ?>"
					aria-label="<?php esc_attr_e( 'Remove this product', 'arewa-recently-viewed-content' ); ?>"
					title="<?php esc_attr_e( 'Remove from history', 'arewa-recently-viewed-content' ); ?>">
					<svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
						<path d="M19 6.41L17.59 5L12 10.59L6.41 5L5 6.41L10.59 12L5 17.59L6.41 19L12 13.41L17.59 19L19 17.59L13.41 12L19 6.41Z" fill="currentColor"/>
					</svg>
				</button>
			</div>

			<!-- Product Info -->
			<div class="arwrev-shop-content">
				<h3 class="arwrev-shop-title">
					<a href="<?php echo esc_url( $permalink ); ?>" class="arwrev-shop-title-link">
						<?php echo esc_html( $title ); ?>
					</a>
				</h3>

				<?php if ( $price_html ) : ?>
					<div class="arwrev-shop-price">
						<?php echo wp_kses_post( $price_html ); ?>
					</div>
				<?php endif; ?>

				<!-- Product Actions -->
				<div class="arwrev-shop-actions">
					<a href="<?php echo esc_url( $permalink ); ?>" class="arwrev-shop-btn arwrev-shop-btn-view">
						<svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
							<path d="M12 4.5C7 4.5 2.73 7.61 1 12C2.73 16.39 7 19.5 12 19.5S21.27 16.39 23 12C21.27 7.61 17 4.5 12 4.5ZM12 17C9.24 17 7 14.76 7 12S9.24 7 12 7S17 9.24 17 12S14.76 17 12 17ZM12 9C10.34 9 9 10.34 9 12S10.34 15 12 15S15 13.66 15 12S13.66 9 12 9Z" fill="currentColor"/>
						</svg>
						<span><?php esc_html_e( 'View Product', 'arewa-recently-viewed-content' ); ?></span>
					</a>

					<?php if ( $show_add_to_cart && $product->is_purchasable() && $product->is_in_stock() ) : ?>
						<?php
						$add_to_cart_url  = $product->add_to_cart_url();
						$add_to_cart_text = $product->add_to_cart_text();
						?>
						<a href="<?php echo esc_url( $add_to_cart_url ); ?>" 
							class="arwrev-shop-btn arwrev-shop-btn-cart"
							data-product_id="<?php echo esc_attr( $post_id ); ?>"
							data-product_sku="<?php echo esc_attr( $product->get_sku() ); ?>">
							<svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
								<path d="M7 18C5.9 18 5.01 18.9 5.01 20S5.9 22 7 22S9 21.1 9 20S8.1 18 7 18ZM1 2V4H3L6.6 11.59L5.25 14.04C5.09 14.32 5 14.65 5 15C5 16.1 5.9 17 7 17H19V15H7.42C7.28 15 7.17 14.89 7.17 14.75L7.2 14.63L8.1 13H15.55C16.3 13 16.96 12.59 17.3 11.97L20.88 5H5.21L4.27 3H1ZM17 18C15.9 18 15.01 18.9 15.01 20S15.9 22 17 22S19 21.1 19 20S18.1 18 17 18Z" fill="currentColor"/>
							</svg>
							<span><?php echo esc_html( $add_to_cart_text ); ?></span>
						</a>
					<?php endif; ?>
				</div>

				<!-- Viewed Date -->
				<div class="arwrev-shop-meta">
					<small class="arwrev-shop-date">
						<?php
						/* translators: %s: formatted date */
						printf( esc_html__( 'Viewed: %s', 'arewa-recently-viewed-content' ), esc_html( $this->get_formatted_date( $timestamp ) ) );
						?>
					</small>
				</div>
			</div>
		</div>
		<?php
		return ob_get_clean();
	}
}
